package com.elitesland.tw.tw5.server.prd.common;

import com.elitesland.tw.tw5.api.common.util.TreeListUtil;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgDataRefVO;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgEmployeeRefVO;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgEmployeeVO;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgOrganizationVO;
import com.elitesland.tw.tw5.api.prd.system.query.*;
import com.elitesland.tw.tw5.api.prd.system.vo.*;
import com.elitesland.tw.tw5.server.common.util.RedisUtils;
import com.elitesland.tw.tw5.server.prd.my.dao.PrdUserDAO;
import com.elitesland.tw.tw5.server.prd.org.convert.PrdOrgEmployeeConvert;
import com.elitesland.tw.tw5.server.prd.org.convert.PrdOrgOrganizationConvert;
import com.elitesland.tw.tw5.server.prd.org.dao.PrdOrgEmployeeDAO;
import com.elitesland.tw.tw5.server.prd.org.dao.PrdOrgOrganizationDAO;
import com.elitesland.tw.tw5.server.prd.org.entity.PrdOrgEmployeeDO;
import com.elitesland.tw.tw5.server.prd.org.entity.PrdOrgOrganizationDO;
import com.elitesland.tw.tw5.server.prd.system.dao.*;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

@Component
@RequiredArgsConstructor
public class CacheUtil {
    private final PrdSystemRemindDAO systemRemindDAO;
    private final PrdSystemSelectionDAO prdSystemSelectionDAO;
    private final PrdSystemTagDAO prdSystemTagDAO;
    private final PrdSystemSettingDAO prdSystemSettingDAO;
    private final RedisUtils redisUtils;

    private final PrdSystemFunctionDAO prdSystemFunctionDAO;
    private final PrdSystemFunctionSpecialDAO prdSystemFunctionSpecialDAO;

    private final PrdSystemFunctionDataDAO prdSystemFunctionDataDAO;

    private final PrdSystemConfigDAO prdSystemConfigDAO;

    private final PrdOrgEmployeeDAO employeeDAO;
    private final PrdOrgOrganizationDAO orgOrganizationDAO;
    private final PrdSystemRoleDAO systemRoleDAO;
    private final PrdUserDAO prdUserDAO;
    /**
     * 用户登录
     */
    private final Map<Long, String> userRoleMaps = new HashMap<>();
    /**
     * 角色对应的用户idMap
     */
    private final Map<String, List<Long>> roleUserMaps = new HashMap<>();
    /**
     * 系统提醒缓存
     */
    private final Map<String, PrdSystemRemindVO> remindCacheMap = new HashMap<>();
    /**
     * 系统选择项缓存
     */
    private final Map<String, PrdSystemSelectionVO> systemSelectionCacheMap = new HashMap<>();
    /**
     * 系统标签缓存
     */
    private final List<PrdSystemTagVO> systemTagCacheMap = new ArrayList<>();
    /**
     * 系统设置项缓存
     */
    private final Map<String, PrdSystemSettingVO> systemSettingCacheMap = new HashMap<>();

    /**
     * 系统功能项缓存
     */
    private final Map<String, PrdSystemFunctionVO> systemFunctionCacheMap = new HashMap<>();

    /**
     * 系统功能项缓存
     */
    private final Map<Long, PrdSystemFunctionVO> systemFunctionCacheMap0 = new HashMap<>();
    /**
     * 系统关键行为缓存
     */
    private final Map<String, PrdSystemFunctionSpecialVO> systemFunctionSpecialCacheMap = new HashMap<>();

    /**
     * 系统功能权限缓存
     */
    private final Map<Long, List<PrdSystemFunctionDataVO>> systemFunctionDataCacheMap = new HashMap<Long, List<PrdSystemFunctionDataVO>>();

    /**
     * 系统功能配置缓存
     */
    private final Map<String, PrdSystemConfigVO> systemConfigCacheMap = new HashMap<String, PrdSystemConfigVO>();

    /**
     * 用户信息缓存
     */
    private final Map<Long, PrdOrgEmployeeVO> employeeCacheMap = new HashMap<>();

    /**
     * 组织信息缓存 key
     */
    private final static String ORGANIZATION_CACHE_KEY = "ORGANIZATION_CACHE_KEY";

    /**
     * 资源组织信息缓存 key
     */
    private final static String EMPLOYEES_ORGANIZATION_CACHE_KEY = "EMPLOYEES_ORGANIZATION_CACHE_KEY";
    /**
     * 默认资源组织信息缓存 key
     */
    private final static String EMPLOYEES_DEFAULT_ORGANIZATION_INFO_CACHE_KEY = "EMPLOYEES_DEFAULT_ORGANIZATION_INFO_CACHE_KEY";

    /**
     * 员工信息缓存 key
     */
    private final static String EMPLOYEES_CACHE_KEY = "EMPLOYEES_CACHE_KEY";

    /**
     * 员工手机号缓存 key
     */
    private final static String EMPLOYEES_CACHE_KEY_MOBILE = "EMPLOYEES_CACHE_MOBILE_KEY";

    /**
     * 查询负责组织的 key
     */
    private final static String EMPLOYEES_MANAGER_ORG_KEY = "EMPLOYEES_MANAGER_ORG_KEY";
    /**
     * 查询我在的所有组织的 key
     */
    private final static String EMPLOYEES_MY_ORG_KEY = "EMPLOYEES_MY_ORG_KEY";
    /**
     * 查询我下属集合
     */
    private final static String EMPLOYEES_MANAGER_SUBORDINATES_KEY = "EMPLOYEES_MANAGER_SUBORDINATES_KEY";

    /**
     * 系统快捷入口
     */
    private final List<PrdSystemShortcutVO> systemShortcutCache = new ArrayList<PrdSystemShortcutVO>();


    private static long lastLoadTimeRemind = 0L;
    private static long lastLoadTimeSystemSelection = 0L;
    private static long lastLoadTimeSystemSetting = 0L;
    private static long lastLoadTimeSystemFunction = 0L;
    private static long lastLoadTimeSystemFunctionSpecial = 0L;
    private static long lastLoadTimeSystemFunctionData = 0L;
    private static long lastLoadTimeSystemConfig = 0L;
    private static long lastLoadTimeSystemTag = 0L;
    private static long lastLoadTimeSystemRole = 0L;

    /**
     * 加载系统配置缓存
     */
    public void loadSystemTagCache() {
        systemTagCacheMap.clear();
        List<PrdSystemTagVO> functionVOs = prdSystemTagDAO.queryListDynamic(new PrdSystemTagQuery());
        functionVOs.forEach(functionVO -> {
            systemTagCacheMap.add(functionVO);
        });
        lastLoadTimeSystemTag = new Date().getTime();
    }

    /**
     * 获取系统标签值
     */
    public List<PrdSystemTagVO> getSystemTag(Long id) {
        if (id == null) {
            if ((this.systemTagCacheMap == null || this.systemTagCacheMap.size() == 0) && shouldLoadCache(lastLoadTimeSystemTag)) {
                loadSystemTagCache();
            }
            return this.systemTagCacheMap;
        } else {
            List<PrdSystemTagVO> functionVOs = this.systemTagCacheMap.stream().filter(systemTag -> systemTag.getId().longValue() == id.longValue()).collect(Collectors.toList());
            if ((functionVOs == null || functionVOs.size() == 0) && shouldLoadCache(lastLoadTimeSystemTag)) {
                loadSystemTagCache();
                List<PrdSystemTagVO> functionVO0 = this.systemTagCacheMap.stream().filter(systemTag -> systemTag.getId().longValue() == id.longValue()).collect(Collectors.toList());
                return functionVO0;
            } else {
                return functionVOs;
            }
        }

    }

    /**
     * 加载系统配置缓存
     */
    public void loadSystemConfigCache() {
        systemConfigCacheMap.clear();
        List<PrdSystemConfigVO> functionVOs = prdSystemConfigDAO.queryListDynamic();
        functionVOs.forEach(functionVO -> {
            systemConfigCacheMap.put(functionVO.getConfigCode(), functionVO);
        });
        lastLoadTimeSystemConfig = new Date().getTime();
    }

    /**
     * 加载系统配置缓存
     */
    public PrdSystemConfigVO getSystemConfig(String code) {
        PrdSystemConfigVO functionVO = this.systemConfigCacheMap.get(code);
        if (functionVO == null && shouldLoadCache(lastLoadTimeSystemConfig)) {
            loadSystemConfigCache();
            return this.systemConfigCacheMap.get(code);
        } else {
            return functionVO;
        }
    }

    /**
     * 加载系统功能权限缓存
     */
    public void loadSystemFunctionDataCache() {
        systemFunctionDataCacheMap.clear();
        PrdSystemFunctionDataQuery query = new PrdSystemFunctionDataQuery();

        List<PrdSystemFunctionDataVO> functionVOs = prdSystemFunctionDataDAO.queryListDynamic(query);
        functionVOs.forEach(functionVO -> {
            Long functionId = functionVO.getFunctionId();
            List<PrdSystemFunctionDataVO> list = systemFunctionDataCacheMap.get(functionId);
            if (list != null && list.size() > 0) {
                list.add(functionVO);
            } else {
                list = new ArrayList<>();
                list.add(functionVO);
                systemFunctionDataCacheMap.put(functionId, list);
            }
        });
        lastLoadTimeSystemFunctionData = new Date().getTime();
    }

    /**
     * 获取系统功权限
     *
     * @param functionId
     * @return 系统功能
     */
    public List<PrdSystemFunctionDataVO> getFunctionData(Long functionId) {
        List<PrdSystemFunctionDataVO> list = systemFunctionDataCacheMap.get(functionId);
        if (list == null && shouldLoadCache(lastLoadTimeSystemFunctionData)) {
            loadSystemFunctionDataCache();
            return this.systemFunctionDataCacheMap.get(functionId);
        } else {
            return list;
        }
    }

    /**
     * 加载系统功能缓存
     */
    public void loadSystemFunctionSpecialCache() {
        systemFunctionSpecialCacheMap.clear();
        PrdSystemFunctionSpecialQuery query = new PrdSystemFunctionSpecialQuery();

        List<PrdSystemFunctionSpecialVO> functionVOs = prdSystemFunctionSpecialDAO.queryListDynamic(query);
        functionVOs.forEach(functionVO -> {
            systemFunctionSpecialCacheMap.put(functionVO.getSpecialUrl(), functionVO);
        });

        lastLoadTimeSystemFunctionSpecial = new Date().getTime();
    }

    /**
     * 获取系统功能
     *
     * @param url
     * @return 系统功能
     */
    public PrdSystemFunctionSpecialVO getFunctionSpecial(String url) {
        PrdSystemFunctionSpecialVO functionVO = this.systemFunctionSpecialCacheMap.get(url);
        if (functionVO == null && shouldLoadCache(lastLoadTimeSystemFunctionSpecial)) {
            loadSystemFunctionSpecialCache();
            return this.systemFunctionSpecialCacheMap.get(url);
        } else {
            return functionVO;
        }
    }

    /**
     * 加载系统功能缓存
     */
    public void loadSystemFunctionCache() {
        systemFunctionCacheMap.clear();
        systemFunctionCacheMap0.clear();
        PrdSystemFunctionQuery query = new PrdSystemFunctionQuery();

        List<PrdSystemFunctionVO> functionVOs = prdSystemFunctionDAO.queryListDynamic(query);
        functionVOs.forEach(functionVO -> {
            systemFunctionCacheMap.put(functionVO.getFunctionCode(), functionVO);
            systemFunctionCacheMap0.put(functionVO.getId(), functionVO);
        });
        TreeListUtil.toTree(functionVOs);
        lastLoadTimeSystemFunction = new Date().getTime();
    }

    /**
     * 获取系统功能
     *
     * @param id
     * @return 系统功能
     */
    public PrdSystemFunctionVO getFunctionById(Long id) {
        PrdSystemFunctionVO functionVO = this.systemFunctionCacheMap0.get(id);
        if (functionVO == null && shouldLoadCache(lastLoadTimeSystemFunction)) {
            loadSystemFunctionCache();
            return this.systemFunctionCacheMap0.get(id);
        } else {
            return functionVO;
        }
    }

    /**
     * 获取系统功能
     *
     * @param functionCode
     * @return 系统功能
     */
    public PrdSystemFunctionVO getFunction(String functionCode) {
        PrdSystemFunctionVO functionVO = this.systemFunctionCacheMap.get(functionCode);
        if (functionVO == null && shouldLoadCache(lastLoadTimeSystemFunction)) {
            loadSystemFunctionCache();
            return this.systemFunctionCacheMap.get(functionCode);
        } else {
            return functionVO;
        }
    }

    /**
     * 加载系统提醒缓存
     */
    public void loadRemindCache() {
        remindCacheMap.clear();
        PrdSystemRemindQuery query = new PrdSystemRemindQuery();
//        query.setLimit(0);
        List<PrdSystemRemindVO> remindViews = systemRemindDAO.queryListDynamic(query);
        remindViews.forEach(remindView -> {
            remindCacheMap.put(remindView.getRemindCode(), remindView);
        });
        lastLoadTimeRemind = new Date().getTime();
    }

    /**
     * 获取消息提醒
     *
     * @param remindCode 消息提醒码
     * @return 消息提醒
     */
    public PrdSystemRemindVO getRemind(String remindCode) {
        PrdSystemRemindVO remindView = this.remindCacheMap.get(remindCode);
        if (remindView == null && shouldLoadCache(lastLoadTimeRemind)) {
            loadRemindCache();
            return this.remindCacheMap.get(remindCode);
        } else {
            return remindView;
        }
    }

    /**
     * 获取不到缓存时是否应该重新
     *
     * @param lastLoadTime 上次加载缓存时间
     * @return 是否
     */
    private boolean shouldLoadCache(long lastLoadTime) {
        long interval = 1000L * 60;
        return new Date().getTime() - lastLoadTime > interval;

    }

    /**
     * 加载系统选择项缓存
     */
    public void loadSystemSelectionCache() {
        systemSelectionCacheMap.clear();
        PrdSystemSelectionQuery query = new PrdSystemSelectionQuery();

        List<PrdSystemSelectionVO> selectionViews = prdSystemSelectionDAO.queryListDynamic(query);
        selectionViews.forEach(selectionView -> {
            systemSelectionCacheMap.put(selectionView.getSelectionKey(), selectionView);
        });
        TreeListUtil.toTree(selectionViews);
        lastLoadTimeSystemSelection = new Date().getTime();
    }

    /**
     * 翻译系统选择项
     *
     * @param selectionKey 选择项key
     * @param value        选择项值
     * @return 选择项描述
     */
    public String transferSystemSelection(String selectionKey, String value) {
        if (value == null) {
            return null;
        }
        PrdSystemSelectionVO selection = getSystemSelection(selectionKey);
        String result = value;
        if (selection == null) {
            return result;
        }
        List<PrdSystemSelectionVO> children = new ArrayList<>();
        getAllChildren(selection, children);
        if (children != null && children.size() > 0) {
            for (PrdSystemSelectionVO selectionView : children) {
                if (value.equals(selectionView.getSelectionValue())) {
                    result = selectionView.getSelectionName();
                }
            }
        }
        return result;
    }

    /**
     * 翻译系统选择项
     *
     * @param selectionKey 选择项key
     * @return PrdSystemSelectionVO
     */
    public PrdSystemSelectionVO transferSystemSelectionObj(String selectionKey) {
        PrdSystemSelectionVO selection = getSystemSelection(selectionKey);
        if (selection == null) {
            return null;
        }
        List<PrdSystemSelectionVO> children = new ArrayList<>();
        getAllChildren(selection, children);
        return selection;
    }

    /**
     * 翻译系统选择项
     *
     * @param selectionKey 选择项key
     * @param value        选择项值
     * @return PrdSystemSelectionVO
     */
    public PrdSystemSelectionVO transferSystemSelectionObj(String selectionKey, String value) {
        if (value == null) {
            return null;
        }
        PrdSystemSelectionVO selection = getSystemSelection(selectionKey);
        if (selection == null) {
            return null;
        }
        List<PrdSystemSelectionVO> children = new ArrayList<>();
        getAllChildren(selection, children);
        if (children != null && children.size() > 0) {
            for (PrdSystemSelectionVO selectionView : children) {
                if (value.equals(selectionView.getSelectionValue())) {
                    return selectionView;
                }
            }
        }
        return null;
    }

    public void getAllChildren(PrdSystemSelectionVO parent, List<PrdSystemSelectionVO> allChildren) {
        if (org.apache.commons.collections4.CollectionUtils.isEmpty(parent.getChildren())) {
            return;
        }
        allChildren.addAll(parent.getChildren());
        for (var child : parent.getChildren()) {
            getAllChildren(child, allChildren);
        }
    }

    /**
     * 系统选择项 根据选项名称 获取选项值
     *
     * @param selectionKey 选择关键
     * @param name         名字
     * @return {@link String}
     */
    public String getSystemSelectionValueByName(String selectionKey, String name) {
        if (!StringUtils.hasText(name)) {
            return null;
        } else {
            name = name.trim();
        }
        PrdSystemSelectionVO selection = getSystemSelection(selectionKey);
        String result = name;
        if (selection == null) {
            return result;
        }
        List<PrdSystemSelectionVO> children = selection.getChildren();
        if (children != null && children.size() > 0) {
            for (PrdSystemSelectionVO selectionView : children) {
                if (name.equals(selectionView.getSelectionName())) {
                    result = selectionView.getSelectionValue();
                }
            }
        }
        return result;
    }

    /**
     * 翻译系统选择项的index
     *
     * @param selectionKey 选择项key
     * @param value        选择项值
     * @return 选择项描述
     */
    public Integer transferSystemSelectionSortIndex(String selectionKey, String value) {
        if (value == null) {
            return null;
        }
        PrdSystemSelectionVO selection = getSystemSelection(selectionKey);
        Integer result = 0;
        if (selection == null) {
            return result;
        }
        List<PrdSystemSelectionVO> children = selection.getChildren();
        if (children != null && children.size() > 0) {
            for (PrdSystemSelectionVO selectionView : children) {
                if (value.equals(selectionView.getSelectionValue())) {
                    result = selectionView.getSortNo() == null ? 0 : selectionView.getSortNo();
                }
            }
        }
        return result;
    }

    /**
     * 获取系统选择项
     *
     * @param selectionKey 选择项key
     * @return 消息提醒
     */
    public PrdSystemSelectionVO getSystemSelection(String selectionKey) {
        PrdSystemSelectionVO prdSystemSelectionVO = this.systemSelectionCacheMap.get(selectionKey);
        if (prdSystemSelectionVO == null && shouldLoadCache(lastLoadTimeSystemSelection)) {
            loadSystemSelectionCache();
            return this.systemSelectionCacheMap.get(selectionKey);
        } else {
            return prdSystemSelectionVO;
        }
    }

    /**
     * 加载系统设置缓存
     */
    public void loadSystemSettingCache() {
        systemSettingCacheMap.clear();
        PrdSystemSettingQuery query = new PrdSystemSettingQuery();
        List<PrdSystemSettingVO> systemSettingViews = prdSystemSettingDAO.queryListDynamic(query);
        systemSettingViews.forEach(systemSettingView -> {
            systemSettingCacheMap.put(systemSettingView.getSettingKey(), systemSettingView);
        });
        lastLoadTimeSystemSetting = new Date().getTime();
    }

    /**
     * 获取系统设置
     *
     * @param settingKey 设置项key
     * @return 系统设置
     */
    public PrdSystemSettingVO getSystemSetting(String settingKey) {
        PrdSystemSettingVO systemSettingView = this.systemSettingCacheMap.get(settingKey);
        if (systemSettingView == null && shouldLoadCache(lastLoadTimeSystemSetting)) {
            loadSystemSettingCache();
            return this.systemSettingCacheMap.get(settingKey);
        } else {
            return systemSettingView;
        }
    }


    /**
     * 获取用户姓名
     *
     * @param userId 用户id
     * @return 系统设置
     */
    public String getUserName(Long userId) {
        if (userId == null) {
            return "";
        }
        PrdOrgEmployeeVO employeeVO = this.employeeCacheMap.get(userId);
        if (employeeVO == null) {
            //加载缓存数据
            PrdOrgEmployeeDO prdOrgEmployeeDO = employeeDAO.queryByUserId(userId);
            if (prdOrgEmployeeDO == null) {
                return null;
            }
            employeeCacheMap.put(userId, PrdOrgEmployeeConvert.INSTANCE.toVo(prdOrgEmployeeDO));
            employeeVO = this.employeeCacheMap.get(userId);
        }
        return employeeVO.getEmployeeName();
    }

    /**
     * 获取用户手机号
     *
     * @param userId 用户id
     * @return 系统设置
     */
    public String getUserMobile(Long userId) {
        if (userId == null) {
            return "";
        }
        String key = EMPLOYEES_CACHE_KEY_MOBILE + ":" + userId;
        String cacheValue = (String) redisUtils.get(key);
        if (cacheValue == null) {
            //加载缓存数据
            String phoneByUserId = employeeDAO.getPhoneByUserId(userId);
            if (null == phoneByUserId) {
                return null;
            }
            redisUtils.set(key, phoneByUserId, 10, TimeUnit.MINUTES);
            return phoneByUserId;
        }
        return cacheValue;
    }

    /**
     * 获取用户userId
     *
     * @param employeeName 员工姓名
     * @return 系统设置
     */
    public Long getUserIdByName(String employeeName) {
        String key = EMPLOYEES_CACHE_KEY + ":" + employeeName;
        final PrdOrgEmployeeVO vo = (PrdOrgEmployeeVO) redisUtils.get(key);
        if (vo == null) {
            //加载缓存数据
            final PrdOrgEmployeeDO employeeDO = employeeDAO.queryByUserName(employeeName);
            if (employeeDO == null) {
                return null;
            }
            final PrdOrgEmployeeVO orgEmployeeVO = PrdOrgEmployeeConvert.INSTANCE.toVo(employeeDO);
            redisUtils.set(key, orgEmployeeVO, 10, TimeUnit.MINUTES);
            return orgEmployeeVO.getUserId();
        }
        return vo.getUserId();
    }

    /**
     * 查询用户默认组织id
     *
     * @param userId 用户id
     * @return {@link Long}
     */
    public Long getDefaultOrgIdByUserId(Long userId) {
        if (null == userId) {
            return null;
        }
        String key = EMPLOYEES_ORGANIZATION_CACHE_KEY + ":" + userId;
        final Long orgId = (Long) redisUtils.get(key);
        if (orgId == null) {
            final List<PrdOrgDataRefVO> prdOrgDataRefVOS = prdUserDAO.queryOrgListByKey(userId);
            if (!CollectionUtils.isEmpty(prdOrgDataRefVOS)) {
                final List<PrdOrgDataRefVO> list = prdOrgDataRefVOS.stream()
                        .filter(prdOrgDataRefVO -> prdOrgDataRefVO.getIsDefault().equals(0))
                        .collect(Collectors.toList());
                if (!CollectionUtils.isEmpty(list)) {
                    final Long orgIdDb = list.get(0).getOrgId();
                    if (null == orgIdDb) {
                        return null;
                    }
                    redisUtils.set(key, orgIdDb, 10, TimeUnit.MINUTES);
                    return orgIdDb;
                }
            }
        }
        return orgId;
    }

    /**
     * 查询我负责的所有组织
     *
     * @param userId 用户id
     * @return {@link Long}
     */
    public List<Long> getManagerOrgIdsByUserId(Long userId) {
        if (null == userId) {
            return null;
        }
        String key = EMPLOYEES_MANAGER_ORG_KEY + ":" + userId;
        final List<Long> orgIds = (List<Long>) redisUtils.get(key);
        //归属组织
        if (ObjectUtils.isEmpty(orgIds)) {
            final List<PrdOrgOrganizationDO> organizationDOList = orgOrganizationDAO.queryByManagerId(userId);
            if (!CollectionUtils.isEmpty(organizationDOList)) {
                List<Long> orgIdList = organizationDOList.stream().map(PrdOrgOrganizationDO::getId).collect(Collectors.toList());
                redisUtils.set(key, orgIdList, 10, TimeUnit.MINUTES);
                return orgIdList;
            }
        }
        return orgIds;
    }

    /**
     * 查询我归属的所有
     *
     * @param userId 用户id
     * @return {@link Long}
     */
    public List<Long> getMyOrgIdsByUserId(Long userId) {
        if (null == userId) {
            return null;
        }
        String key = EMPLOYEES_MY_ORG_KEY + ":" + userId;
        final List<Long> orgIds = (List<Long>) redisUtils.get(key);
        //归属组织
        if (ObjectUtils.isEmpty(orgIds)) {
            List<Long> longs = orgOrganizationDAO.queryOrgIdSByKey(userId);
            if (!CollectionUtils.isEmpty(longs)) {
                redisUtils.set(key, longs, 10, TimeUnit.MINUTES);
                return longs;
            }
        }
        return orgIds;
    }

    /**
     * 根据用户id默认组织信息 包含上级、组织等信息
     *
     * @param userId 用户id
     * @return {@link PrdOrgDataRefVO}
     */
    public PrdOrgDataRefVO getDefaultOrgInfoByUserId(Long userId) {
        if (null == userId) {
            return null;
        }
        String key = EMPLOYEES_DEFAULT_ORGANIZATION_INFO_CACHE_KEY + ":" + userId;
        final PrdOrgDataRefVO result = (PrdOrgDataRefVO) redisUtils.get(key);
        if (result == null) {
            final List<PrdOrgDataRefVO> prdOrgDataRefVOS = prdUserDAO.queryOrgListByKey(userId);
            if (!CollectionUtils.isEmpty(prdOrgDataRefVOS)) {
                final List<PrdOrgDataRefVO> list = prdOrgDataRefVOS.stream()
                        .filter(prdOrgDataRefVO -> prdOrgDataRefVO.getIsDefault().equals(0))
                        .collect(Collectors.toList());
                if (!CollectionUtils.isEmpty(list)) {
                    final PrdOrgDataRefVO prdOrgDataRefVO = list.get(0);
                    if (null == prdOrgDataRefVO) {
                        return null;
                    }
                    redisUtils.set(key, prdOrgDataRefVO, 10, TimeUnit.MINUTES);
                    return prdOrgDataRefVO;
                }
            }
        }
        return result;
    }

    /**
     * 得到组织名字
     *
     * @param ordId 奥德id
     * @return {@link String}
     */
    public String getOrgName(Long ordId) {
        if (ordId == null) {
            return "";
        }
        String key = ORGANIZATION_CACHE_KEY + ":" + ordId;
        final PrdOrgOrganizationVO vo = (PrdOrgOrganizationVO) redisUtils.get(key);
        if (vo == null) {
            //加载缓存数据
            final PrdOrgOrganizationDO prdOrgOrganizationDO = orgOrganizationDAO.queryById(ordId);
            if (prdOrgOrganizationDO == null) {
                return null;
            }
            final PrdOrgOrganizationVO prdOrgOrganizationVO = PrdOrgOrganizationConvert.INSTANCE.toVo(prdOrgOrganizationDO);
            redisUtils.set(key, prdOrgOrganizationVO, 10, TimeUnit.MINUTES);
            return prdOrgOrganizationVO.getOrgName();
        }
        return vo.getOrgName();
    }

    /**
     * 获取部门负责人
     *
     * @param ordId 部门主键
     * @return {@link String}
     */
    public Long getOrgManageUserId(Long ordId) {
        if (ordId == null) {
            return 0L;
        }
        String key = ORGANIZATION_CACHE_KEY + ":" + ordId;
        final PrdOrgOrganizationVO vo = (PrdOrgOrganizationVO) redisUtils.get(key);
        if (vo == null) {
            //加载缓存数据
            final PrdOrgOrganizationDO prdOrgOrganizationDO = orgOrganizationDAO.queryById(ordId);
            if (prdOrgOrganizationDO == null) {
                return null;
            }
            final PrdOrgOrganizationVO prdOrgOrganizationVO = PrdOrgOrganizationConvert.INSTANCE.toVo(prdOrgOrganizationDO);
            redisUtils.set(key, prdOrgOrganizationVO, 10, TimeUnit.MINUTES);
            return prdOrgOrganizationVO.getManageId();
        }
        return vo.getManageId();
    }

    /**
     * 获取下属集合
     *
     * @param userId 部门负责人
     * @return {@link String}
     */
    public List<Long> getSubordinatesByManageUserId(Long userId) {
        String key = EMPLOYEES_MANAGER_SUBORDINATES_KEY + ":" + userId;
        List<Long> resultCache = (List<Long>) redisUtils.get(key);
        if (CollectionUtils.isEmpty(resultCache)) {
            List<Long> subordinateCollection;
            PrdOrgEmployeeRefVO prdOrgEmployeeRefVO = employeeDAO.queryUserOrgData(userId);
            if (prdOrgEmployeeRefVO != null && prdOrgEmployeeRefVO.getManageId() != null && userId.equals(prdOrgEmployeeRefVO.getManageId())) {
                Long orgId = prdOrgEmployeeRefVO.getOrgId();
                //查询部门下的成员
                List<PrdOrgEmployeeRefVO> prdOrgEmployeeRefVOS = orgOrganizationDAO.queryEmployeeList(orgId);
                subordinateCollection = prdOrgEmployeeRefVOS.stream().map(e -> e.getUserId()).distinct().collect(Collectors.toList());
            } else {
                subordinateCollection = Collections.singletonList(userId);
            }
            redisUtils.set(key, subordinateCollection, 10, TimeUnit.MINUTES);
            return subordinateCollection;
        }
        return resultCache;
    }

    /**
     * 得到组织名字
     *
     * @param orgName 组织名称
     * @return {@link String}
     */
    public Long getOrgIdByOrgName(String orgName) {
        String key = ORGANIZATION_CACHE_KEY + ":" + orgName;
        final PrdOrgOrganizationVO vo = (PrdOrgOrganizationVO) redisUtils.get(key);
        if (vo == null) {
            //加载缓存数据
            final PrdOrgOrganizationDO prdOrgOrganizationDO = orgOrganizationDAO.queryByName(orgName);
            if (prdOrgOrganizationDO == null) {
                return null;
            }
            final PrdOrgOrganizationVO prdOrgOrganizationVO = PrdOrgOrganizationConvert.INSTANCE.toVo(prdOrgOrganizationDO);
            redisUtils.set(key, prdOrgOrganizationVO, 10, TimeUnit.MINUTES);
            return prdOrgOrganizationVO.getId();
        }
        return vo.getId();
    }

    /**
     * 获取用户编号
     *
     * @param userId 用户id
     * @return 系统设置
     */
    public PrdOrgEmployeeVO getEmployee(Long userId) {
        PrdOrgEmployeeVO employeeVO = this.employeeCacheMap.get(userId);
        if (employeeVO == null) {
            //加载缓存数据
            PrdOrgEmployeeDO prdOrgEmployeeDO = employeeDAO.queryByUserId(userId);
            if (prdOrgEmployeeDO == null) {
                return null;
            }
            employeeCacheMap.put(userId, PrdOrgEmployeeConvert.INSTANCE.toVo(prdOrgEmployeeDO));
            employeeVO = this.employeeCacheMap.get(userId);
        }
        return employeeVO;
    }

    //    /**
//     * 获取系统设置
//     *
//     * @param settingKey 设置项key
//     * @return 系统设置
//     */
//    public PrdSystemSettingVO getSystemSetting(String settingKey) {
//        PrdSystemSettingVO systemSettingView = this.systemSettingCacheMap.get(settingKey);
//        if (systemSettingView == null && shouldLoadCache(lastLoadTimeSystemSetting)) {
//            loadSystemSettingCache();
//            return this.systemSettingCacheMap.get(settingKey);
//        } else {
//            return systemSettingView;
//        }
//    }


    /**
     * 判断某个用户是否拥有这个角色
     *
     * @param roleList
     * @return
     */
    public Boolean hasSystemRolePermission(List<String> roleList) {
        Long loginUserId = GlobalUtil.getLoginUserId();
        String longString = userRoleMaps.get(loginUserId);
        if (!StringUtils.hasText(longString) || shouldLoadCache(Long.valueOf(longString.split(",")[0]))) {
            loadSystemRoleCache(loginUserId);
            return hasSystemRolePermission(roleList);
        }
        List<String> stringList = Arrays.asList(longString.split(","));
        for (String roleCode : roleList) {
            if (stringList.contains(roleCode)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 查询角色下的所有用户
     *
     * @param roleCode
     * @return
     */
    public List<Long> queryRoleUserIds(String roleCode) {
        List<Long> longs = roleUserMaps.get(roleCode);
        if (ObjectUtils.isEmpty(longs) || shouldLoadCache(lastLoadTimeSystemRole)) {
            loadSystemRoleCache(roleCode);
            return roleUserMaps.get(roleCode);
        }
        return longs;
    }

    /**
     * 加载系统设置缓存
     */
    public void loadSystemRoleCache(String roleCode) {
        List<Long> userIds = systemRoleDAO.queryUserIdByRoleCode(roleCode);
        roleUserMaps.put(roleCode, userIds);
        lastLoadTimeSystemSetting = new Date().getTime();
    }

    /**
     * 返回用户的系统角色编码
     *
     * @param userId 用户id
     * @return {@link List}<{@link String}>
     */
    public List<String> getSystemRoleCodes(Long userId) {
        if (null == userId) {
            userId = GlobalUtil.getLoginUserId();
        }
        List<String> roleCodes = systemRoleDAO.queryUserRoleCodes(userId);
        return roleCodes;
    }

    /**
     * 返回用户的系统角色编码
     *
     * @param userId 用户id
     * @return
     */
    public List<String> getSystemRoleCodesCache(Long userId) {
        if (null == userId) {
            userId = GlobalUtil.getLoginUserId();
        }
        String roleCodesStr = userRoleMaps.get(userId);
        if (!StringUtils.hasText(roleCodesStr) || shouldLoadCache(Long.valueOf(roleCodesStr.split(",")[0]))) {
            loadSystemRoleCache(userId);
            roleCodesStr = userRoleMaps.get(userId);
        }
        if (StringUtils.hasText(roleCodesStr)) {
            List<String> roleCodes = Arrays.asList(roleCodesStr.split(","));
            return roleCodes;
        }
        return new ArrayList<>();
    }


    /**
     * 加载系统设置缓存
     */
    public void loadSystemRoleCache(Long loginUserId) {
        List<String> roleCodes = systemRoleDAO.queryUserRoleCodes(loginUserId);

        String str = new Date().getTime() + "";
        if (!ObjectUtils.isEmpty(roleCodes)) {
            str = str.concat(",").concat(StringUtils.collectionToCommaDelimitedString(roleCodes));
        }
        userRoleMaps.put(loginUserId, str);
    }
}
