package com.elitesland.tw.tw5.server.prd.system.dao;

import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgEmployeeRefVO;
import com.elitesland.tw.tw5.api.prd.system.payload.PrdSystemRolePayload;
import com.elitesland.tw.tw5.api.prd.system.query.PrdSystemRoleQuery;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemRoleVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.org.entity.QPrdOrgEmployeeDO;
import com.elitesland.tw.tw5.server.prd.system.entity.*;
import com.elitesland.tw.tw5.server.prd.system.repo.PrdSystemRoleMenuRepo;
import com.elitesland.tw.tw5.server.prd.system.repo.PrdSystemUserRoleRepo;
import com.elitesland.tw.tw5.server.prd.system.repo.PrdSystemRoleRepo;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.QueryResults;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.dsl.Expressions;
import com.querydsl.core.types.dsl.StringTemplate;
import com.querydsl.jpa.impl.JPADeleteClause;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.util.ObjectUtils;

import java.util.List;

/**
 * 系统选择项dao
 *
 * @author wangding
 */
@Repository
@RequiredArgsConstructor
public class PrdSystemRoleDAO {
    private final JPAQueryFactory jpaQueryFactory;
    private final PrdSystemRoleRepo repo;
    private final PrdSystemUserRoleRepo repoUserRole;
    private final PrdSystemRoleMenuRepo repoMenu;
    private final QPrdSystemRoleDO qdo = QPrdSystemRoleDO.prdSystemRoleDO;
    private final QPrdSystemRoleMenuDO qdoRoleMenu = QPrdSystemRoleMenuDO.prdSystemRoleMenuDO;
    private final QPrdSystemMenuDO qdoMenu = QPrdSystemMenuDO.prdSystemMenuDO;
    private final QPrdSystemUserRoleDO qdoUserRole = QPrdSystemUserRoleDO.prdSystemUserRoleDO;
    private final QPrdOrgEmployeeDO qdoEmployee = new QPrdOrgEmployeeDO("prdOrgEmployeeDO");

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PrdOrgEmployeeRefVO> getJpaQuerySelectUsers() {
        return jpaQueryFactory.select(Projections.bean(PrdOrgEmployeeRefVO.class,
                qdoEmployee.userId,
                qdoEmployee.employeeName,
                qdoEmployee.sortIndex
        )).from(qdoEmployee).leftJoin(qdoUserRole).on(qdoEmployee.userId.longValue().eq(qdoUserRole.userId.longValue()));
    }

    /**
     * 查询所有
     *
     * @return 结果
     */
    public List<PrdOrgEmployeeRefVO> queryUsers(Long roleId) {
        JPAQuery<PrdOrgEmployeeRefVO> jpaQuery = getJpaQuerySelectUsers();
        jpaQuery.where(qdoUserRole.roleId.eq(roleId));
        jpaQuery.where(qdoEmployee.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 调用jpa的保存
     *
     * @param ados do对象
     * @return 保存后的对象
     */
    public List<PrdSystemUserRoleDO> saveUserRoleAll(List<PrdSystemUserRoleDO> ados) {
        return repoUserRole.saveAll(ados);
    }

    /**
     * 物理删除
     *
     * @param roleId
     * @return 删除的行数
     */
    public long deleteRolesByUserId(Long roleId) {
        JPADeleteClause delete = jpaQueryFactory.delete(qdoUserRole).where(qdoUserRole.roleId.eq(roleId));
        return delete.execute();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PrdSystemRoleDO save(PrdSystemRoleDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存
     *
     * @param ados do对象
     * @return 保存后的对象
     */
    public List<PrdSystemRoleMenuDO> saveRoleMenuAll(List<PrdSystemRoleMenuDO> ados) {
        return repoMenu.saveAll(ados);
    }

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PrdSystemRoleVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PrdSystemRoleVO.class,
                qdo.id,
                qdo.roleCode,
                qdo.roleName,
                qdo.sortIndex,
                qdo.enabled,
                qdo.sortIndex,
                qdo.remark,
                qdo.deleteFlag,
                qdo.createUserId,
                qdo.createTime
        )).from(qdo);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    public long updateByKeyDynamic(PrdSystemRolePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));

        if (payload.getRoleCode() != null) {
            update.set(qdo.roleCode, payload.getRoleCode());
        }
        if (payload.getRoleName() != null) {
            update.set(qdo.roleName, payload.getRoleName());
        }
        if (payload.getEnabled() != null) {
            update.set(qdo.enabled, payload.getEnabled());
        }

        if (payload.getSortIndex() != null) {
            update.set(qdo.sortIndex, payload.getSortIndex());
        }
        if (payload.getRemark() != null) {
            update.set(qdo.remark, payload.getRemark());
        }

        // 处理要设置成空的字段
//        List<String> nullFields = payload.getNullFields();
//        if (nullFields != null && nullFields.size() > 0) {
//
////            if (nullFields.contains("remindContent")) {
////                update.setNull(qdo.remindContent);
////            }
////            if (nullFields.contains("remindContentLocale")) {
////                update.setNull(qdo.remindContentLocale);
////            }
//            if (nullFields.contains("remark")) {
//                update.setNull(qdo.remark);
//            }
//        }
        // 执行修改
        return update.execute();
    }

    /**
     * 物理删除
     *
     * @param keys 主键
     * @return 删除的行数
     */
    public long delete(List<Long> keys) {
        JPADeleteClause delete = jpaQueryFactory.delete(qdo).where(qdo.id.in(keys));

        return delete.execute();
    }

    /**
     * 物理删除
     *
     * @param roleId
     * @return 删除的行数
     */
    public long deleteRoleMenusByRoleId(Long roleId) {
        JPADeleteClause delete = jpaQueryFactory.delete(qdoRoleMenu).where(qdoRoleMenu.roleId.eq(roleId));

        return delete.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主键
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        return update.execute();
    }

    /**
     * 更新状态
     *
     * @param id      主键
     * @param enabled 状态
     * @return 删除的行数
     */
    public long updateStatus(Long id, Boolean enabled) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.enabled, enabled)
                .where(qdo.id.eq(id));
        return update.execute();
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PrdSystemRoleVO queryByKey(Long id) {
        JPAQuery<PrdSystemRoleVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 通过编号查询
     *
     * @param code 编号
     * @return vo对象
     */
    public PrdSystemRoleVO queryByCode(String code) {
        JPAQuery<PrdSystemRoleVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.roleCode.eq(code));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    public List<PrdSystemRoleVO> queryByCodeIn(List<String> codes) {
        JPAQuery<PrdSystemRoleVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.roleCode.in(codes));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 列表查询
     *
     * @param roleId 角色id
     * @return vo对象
     */
    public List<Long> queryRoleMenuIds(Long roleId) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory.select(qdoRoleMenu.menuId).from(qdoRoleMenu).leftJoin(qdoMenu).on(qdoRoleMenu.menuId.longValue().eq(qdoMenu.id.longValue()));
        jpaQuery.where(qdoMenu.menuStatus.eq(0));
        jpaQuery.where(qdoMenu.deleteFlag.eq(0));
        jpaQuery.where(qdoRoleMenu.roleId.eq(roleId));
        return jpaQuery.fetch();
    }

    /**
     * 列表查询
     *
     * @return vo对象
     */
    public List<PrdSystemRoleVO> queryList() {
        JPAQuery<PrdSystemRoleVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.enabled.eq(true));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }


    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PrdSystemRoleVO> getJpaQueryWhere(PrdSystemRoleQuery query) {
        JPAQuery<PrdSystemRoleVO> jpaQuery = getJpaQuerySelect0();
        if (!ObjectUtils.isEmpty(query.getRoleCode())) {
            jpaQuery.where(qdo.roleCode.eq(query.getRoleCode()));
        }
        if (!ObjectUtils.isEmpty(query.getRoleName())) {
            jpaQuery.where(qdo.roleName.like(SqlUtil.toSqlLikeString(query.getRoleName())));
        }
        if (!ObjectUtils.isEmpty(query.getEnabled())) {
            jpaQuery.where(qdo.enabled.eq(query.getEnabled()));
        }
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, SqlUtil.getOrderse()));
        return jpaQuery;
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PrdSystemRoleVO> queryPaging(PrdSystemRoleQuery query) {
        JPAQuery<PrdSystemRoleVO> jpaQuery = getJpaQueryWhere(query);
        QueryResults<PrdSystemRoleVO> result = jpaQuery.groupBy(qdo.id).offset(query.getPageRequest().getOffset()).limit(query.getPageRequest().getPageSize()).fetchResults();
        return PagingVO.<PrdSystemRoleVO>builder().records(result.getResults()).total(result.getTotal()).build();
    }

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PrdSystemRoleVO> getJpaQuerySelect0() {
        StringTemplate stringTemplate = Expressions.stringTemplate("CONCAT(GROUP_CONCAT({0}),'')", qdoEmployee.employeeName);
        return jpaQueryFactory.select(Projections.bean(PrdSystemRoleVO.class,
                qdo.id,
                qdo.roleCode,
                qdo.roleName,
                qdo.sortIndex,
                qdo.enabled,
                stringTemplate.as("userNames"),
                qdo.sortIndex,
                qdo.remark,
                qdo.deleteFlag,
                qdo.createUserId,
                qdo.createTime
        )).from(qdo).leftJoin(qdoUserRole).on(qdoUserRole.roleId.eq(qdo.id)).leftJoin(qdoEmployee).on(qdoEmployee.userId.eq(qdoUserRole.userId));
    }

    /**
     * 根据角色获取用户id列表
     *
     * @return
     */
    public List<Long> queryUserIdByRoleCode(String roleNo) {
        return repoUserRole.queryUserIdByRoleCode(roleNo);
    }

    /**
     * 根据角色获取用户id列表
     *
     * @return
     */
    public List<Long> queryUserIdByRoleCodes(List<String> roleNos) {
        return repoUserRole.queryUserIdByRoleCodes(roleNos);
    }

    /**
     * 获取用户的所有角色
     *
     * @return
     */
    public List<String> queryUserRoleCodes(Long userId) {
        return repoUserRole.queryUserRoleCodes(userId);
    }

}
