package com.elitesland.tw.tw5.server.common.annotation;

import com.elitesland.tw.tw5.server.common.constants.TwConstants;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.util.concurrent.TimeUnit;

/**
 * 避免重复提交
 *
 * 示例
 * @ReSubmitCheck(argExpressions = {"[0].id", "[0].name"}, conditionExpressions = "[0].address != null")
 * public TwOutputUtil save(@RequestBody Person person) {
 *         return TwOutputUtil.ok();
 *     }
 *
 *
 *     通过argExpressions可以将指定参数作为防重复校验key的一部分，可以通过此参数控制锁的粒度；
 *
 *     conditionExpressions表示进行重复提交校验的条件，如果指定了多个表达式，多个表达式之间是&&的关系，只有当所有的条件都满足时才进行重复提交校验；
 *
 *     withUserInfoInKey参数表示是否将当前操作人的信息作为key的一部分，可以通过此参数控制锁的粒度，即使是相同数据的提交，只能对同一个人进行防重复提交限制；
 *
 *     onlyInCurrentSession参数表示是否仅在当前session内进行防重复提交校验，是对withUserInfoInKey参数的补充,
 *     如果withUserInfoInKey指定为false，可以在session粒度内对重复数据进行提交校验；
 *
 *     interval表示同一份数据防重复提交的时间间隔，也即是key在Redis中存放的时间。
 *
 *
 *
 * @author duwh
 * @date 2023/02/13
 */
@Target(ElementType.METHOD)
@Retention(RetentionPolicy.RUNTIME)
public @interface ReSubmitCheck {

    // 资源名称，用于描述接口功能
    String name() default "";

    /**
     * 参数Spring EL表达式例如 #{param.name},表达式的值作为防重复校验key的一部分
     */
    String[] argExpressions();

    /**
     * 重复提交报错信息
     */
    String message() default TwConstants.RESUBMIT_MSG;

    /**
     * Spring EL表达式,决定是否进行重复提交校验,多个条件之间为且的关系,默认是进行校验
     */
    String[] conditionExpressions() default {"true"};

    /**
     * 是否选用当前操作用户的信息作为防重复提交校验key的一部分
     */
    boolean withUserInfoInKey() default true;

    /**
     * 是否仅在当前session内进行防重复提交校验
     */
    boolean onlyInCurrentSession() default false;

    /**
     * 防重复提交校验的时间间隔
     * 默认 500毫秒=0.5秒
     *    1500毫秒=1.5秒
     */
    long interval() default 1500;

    /**
     * 防重复提交校验的时间间隔的单位
     * 默认毫秒单位
     */
    TimeUnit timeUnit() default TimeUnit.MILLISECONDS;
}
