package com.elitesland.tw.tw5.server.aspect;

import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.annotation.ReSubmitCheck;
import com.elitesland.tw.tw5.server.common.constants.TwConstants;
import com.elitesland.tw.tw5.server.common.util.ExpressionUtils;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.core.annotation.Order;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;

/**
 * 避免重复提交 切面
 *
 * @author duwh
 * @date 2023/02/14
 */
@Component
@Aspect
@Slf4j
@Order(-1)
@ConditionalOnProperty(name = "enabled", prefix = "resubmit-check", havingValue = "true", matchIfMissing = true)
public class ReSubmitCheckAspect {

    private static final String REDIS_SEPARATOR = "::";

    private static final String RESUBMIT_CHECK_KEY_PREFIX = "ReSubmitCheckKey" + REDIS_SEPARATOR;

    @Resource
    private RedisTemplate<String, String> redisTemplate;

    @Resource
    private HttpServletRequest request;

    /**
     * 配置环绕通知,使用在方法pointcut()上注册的切入点
     *
     * @param joinPoint  join point for advice
     * @param annotation 注解
     */
    @Before("@annotation(annotation)")
    public void beforeReSubmitCheck(JoinPoint joinPoint, ReSubmitCheck annotation) {
        final Object[] args = joinPoint.getArgs();
        final String[] conditionExpressions = annotation.conditionExpressions();

        //根据条件判断是否需要进行防重复提交检查
        if (!ExpressionUtils.getConditionValue(args, conditionExpressions) || ArrayUtils.isEmpty(args)) {
            return;
        }
        doCheck(annotation, args);
    }

    /**
     * key的组成为: prefix::userInfo::sessionId::uri::method::(根据spring EL表达式对参数进行拼接)
     *
     * @param annotation 注解
     * @param args       方法入参
     */
    private void doCheck(@NonNull ReSubmitCheck annotation, Object[] args) {
        final String[] argExpressions = annotation.argExpressions();
        final String message = annotation.message();
        final boolean withUserInfoInKey = annotation.withUserInfoInKey();
        final boolean onlyInCurrentSession = annotation.onlyInCurrentSession();

        String methodDesc = request.getMethod();
        String uri = request.getRequestURI();

        StringBuilder stringBuilder = new StringBuilder(64);
        Object[] argsForKey = ExpressionUtils.getExpressionValue(args, argExpressions);
        for (Object obj : argsForKey) {
            stringBuilder.append(obj != null ? obj.toString() : "");
        }

        StringBuilder keyBuilder = new StringBuilder();
        keyBuilder.append(RESUBMIT_CHECK_KEY_PREFIX)
            .append(withUserInfoInKey ? getLoginUserId() + REDIS_SEPARATOR : "")
            .append(onlyInCurrentSession ? request.getSession().getId() + REDIS_SEPARATOR : "")
            .append(uri)
            .append(REDIS_SEPARATOR)
            .append(methodDesc).append(REDIS_SEPARATOR)
            .append(stringBuilder);
        if (redisTemplate.opsForValue().get(keyBuilder.toString()) != null) {
            throw TwException.error("", StringUtils.isBlank(message) ? TwConstants.RESUBMIT_MSG : message);
        }
        //值为空
        redisTemplate.opsForValue().set(keyBuilder.toString(), "", annotation.interval(), annotation.timeUnit());
    }


    public Long getLoginUserId() {
        try {
            return GlobalUtil.getLoginUserId();
        } catch (Exception e) {
            return 0L;
        }
    }


}
