package com.elitesland.tw.tw5.server.common.change.dao;

import com.elitesland.tw.tw5.api.common.change.payload.ComChangePayload;
import com.elitesland.tw.tw5.api.common.change.query.ComChangeQuery;
import com.elitesland.tw.tw5.api.common.change.vo.ComChangeVO;
import com.elitesland.tw.tw5.server.common.change.entity.QComChangeDO;
import com.elitesland.tw.tw5.server.common.change.repo.ComChangeRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.common.util.SqlUtilPlus;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 通用变更信息
 *
 * @author carl.wang
 * @date 2023-03-06
 */
@Repository
@RequiredArgsConstructor
public class ComChangeDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ComChangeRepo repo;
    private final QComChangeDO qdo = QComChangeDO.comChangeDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ComChangeVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ComChangeVO.class,
                qdo.id,
                qdo.createTime,
                qdo.changeContent,
                qdo.versionNo,
                qdo.createUserId,
                qdo.creator,
                qdo.changeDocId,
                qdo.apprStatus,
                qdo.changeStatus,
                qdo.changeType,
                qdo.extString1,
                qdo.extString2,
                qdo.extString3,
                qdo.extString4,
                qdo.extString5
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ComChangeVO> getJpaQueryWhere(ComChangeQuery query) {
        JPAQuery<ComChangeVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
//        // 常用基础查询条件拼装
//        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtilPlus.getSortedColumn(query.getOrders(), qdo));
        return jpaQuery;
    }


    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ComChangeQuery query) {
        List<Predicate> list = new ArrayList<>();

        if (!ObjectUtils.isEmpty(query.getChangeDocId())) {
            list.add(qdo.changeDocId.eq(query.getChangeDocId()));
        }
        if (!ObjectUtils.isEmpty(query.getChangeType())) {
            list.add(qdo.changeType.eq(query.getChangeType()));
        }

        return ExpressionUtils.allOf(list);
    }

    /**
     * 变更工作流相关数据
     *
     * @param payload
     * @return
     */
    public long updateWorkFlow(ComChangePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));

        // 流程实例id
        if (!ObjectUtils.isEmpty(payload.getApprProcInstId())) {
            update.set(qdo.apprProcInstId, payload.getApprProcInstId());
        } else {
            update.setNull(qdo.apprProcInstId);
        }
        // 审批状态
        if (!ObjectUtils.isEmpty(payload.getApprStatus())) {
            update.set(qdo.apprStatus, payload.getApprStatus());
        } else {
            update.setNull(qdo.apprStatus);
        }
        // 状态
        if (!ObjectUtils.isEmpty(payload.getChangeStatus())) {
            update.set(qdo.changeStatus, payload.getChangeStatus());
        } else {
            update.setNull(qdo.changeStatus);
        }

        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ComChangeVO> queryListDynamic(ComChangeQuery query) {
        JPAQuery<ComChangeVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ComChangeVO queryByKey(Long id) {
        JPAQuery<ComChangeVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }
}

