package com.elitesland.tw.tw5.server.prd.qixin.controller;

import com.elitesland.tw.tw5.api.common.TwOutputUtil;
import com.elitesland.tw.tw5.server.common.util.RedisUtils;
import com.elitesland.tw.tw5.server.prd.qixin.service.QiXinService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

/**
 * 启信宝-第三方服务接口
 *
 * @author duwh
 * @date 2022/11/24
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/api/qixin")
@Slf4j
public class QiXinController {

    private final QiXinService qiXinService;
    private final RedisUtils redisUtils;
    /**
     * 新闻舆情
     */
    private final static String QXB_NEW_KEY = "QXB_NEW_KEY";

    /**
     * 1.31 模糊搜索
     *
     * @param matchType 匹配类型（可选），keyword 需要匹配的 字段，可以输入以下一个或者几个类型， 多个类型使用|分隔： partner：股东 oper：法人 4 member：高管 contact：联系方式 scope：经营范围 name：公司名称 patent：专利 copyright：著作权作品名称 software：软件著作权名称 trademark：商标 domain：网址 product：产品
     * @param region    地区编码（可选），输入参数为空返回全 部区域匹配记录。可传入 省/直辖市，城市，区县的国家区划码。 只支持传入一个区划码。 省/直辖市：传入国家区划码前两位数字， 如搜索“江苏省”则传入 32，各省/直辖 市编码可参考本文档附录 城市：传入国家区划码前四位数字，如搜 索“苏州市”，则传入 3205 区县：传入国家区划码六位数字，如搜索 “苏州市吴中区”，则传入 320506 可以通过下面网页查询国家区划编码 http://www.mca.gov.cn/article/sj/tj bz/a/2017/2018
     * @param keyword   关键字（必填） 企业全名/注册号/统一社会信用代码
     * @param skip      跳过(可选) 跳过条目数（默认为0，单页返回10条数据）
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/search")
    public TwOutputUtil search(String matchType, String region, String keyword, String skip) {
        if (!StringUtils.hasText(keyword)) {
            log.error("启信宝 1.31 模糊搜索 缺少参数 keyword");
            return TwOutputUtil.error("", "参数缺失", null);
        }
        String result = qiXinService.search(matchType, region, keyword, skip);
        return TwOutputUtil.ok(result);
    }

    /**
     * 1.2 高级搜索
     *
     * @param keyword       企业全名/注册号/统一社会信用代码
     * @param skip          跳过条目数（默认为0，单页返回20条数据）
     * @param method        搜索范围，多个使用逗号分隔，具体值参 考附录
     * @param org_type      组织类型，多个使用逗号分隔，具体值参 考附录
     * @param area_code     地区码
     * @param industry_code 行业码
     * @param econ_type     企业类型，多个使用逗号分隔，具体值参 考附录
     * @param capi_from     注册资本起
     * @param capi_to       注册资本止
     * @param date_from     成立日期起
     * @param date_to       成立日期止
     * @param status        企业状态，多个使用逗号分隔，具体值参 考附录
     * @param canbao_from   参保人数起
     * @param canbao_to     参保人数止
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/advanceSearchNew")
    public TwOutputUtil advanceSearchNew(String keyword, String skip, String method, String org_type, String area_code, String industry_code,
                                         String econ_type, String capi_from, String capi_to, String date_from, String date_to,
                                         String status, String canbao_from, String canbao_to) {
        if (!StringUtils.hasText(keyword)) {
            log.error("启信宝 1.2 高级搜索 缺少参数 keyword");
            return TwOutputUtil.error("", "参数缺失", null);
        }
        String result = qiXinService.advanceSearchNew(keyword, skip, method, org_type, area_code, industry_code,
            econ_type, capi_from, capi_to, date_from, date_to, status, canbao_from, canbao_to);
        return TwOutputUtil.ok(result);
    }

    /**
     * 获取企业联系方式 1.51 企业联系方式
     *
     * @param keyword 关键字 企业全名/注册号/统一社会信用代码
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/getContactInfo")
    public TwOutputUtil getContactInfo(String keyword) {
        if (!StringUtils.hasText(keyword)) {
            log.error("启信宝 获取企业联系方式 缺少参数 keyword");
            return TwOutputUtil.error("", "参数缺失", null);
        }
        String result = qiXinService.getContactInfo(keyword);
        return TwOutputUtil.ok(result);
    }

    /**
     * 10.1 新闻列表
     * 缓存1天（key = name+create_time_range+skip）
     * 企业新闻列表，包括新闻来源、情感属性、舆情标签、负面指数等
     * 申请接口
     * 单价：0.1元/次
     *
     * @param name              必填 企业全名/注册号/统一社会信用代码
     * @param skip              跳过条目数（默认为 0）
     * @param tag_list          舆情分类，参考数据字典tag_list字段，示例：["131","132"]
     * @param sentiment         情感属性，参考数据字典sentiment字段，示例：["neg","pos"]
     * @param company_sentiment 企业的情感属性，参考数据字典sentiment字段，示例：["neg","pos"]
     * @param class_types       新闻类别，全部可不传，参考数据字典class_types字段，示例：["news"]
     * @param new_tag_list      事件分类，参考数据字典new_tag_list字段，示例：["101","1024"]
     * @param company_new_tag   企业的事件分类，参考数据字典new_tag_list字段，示例：["0102004","0102005"]
     * @param create_time_range 舆情发布时间范围，示例：["2019-01-01","2019-08-14"]
     * @return {@link String}
     */
    @GetMapping("/getNewsListByName")
    public TwOutputUtil getNewsListByName(String name, String skip, String tag_list, String sentiment,
                                          String company_sentiment, String class_types, String new_tag_list,
                                          String company_new_tag, String create_time_range) {
        if (!StringUtils.hasText(skip)) {
            skip = "0";
        }
        if (!StringUtils.hasText(name)) {
            log.error("启信宝 10.1 新闻列表 缺少参数 name");
            return TwOutputUtil.error("", "参数缺失", null);
        }
        if (!StringUtils.hasText(create_time_range)) {
            create_time_range = "";
        }
//        String key = QXB_NEW_KEY + ":" + name + "_" + create_time_range + "_" + skip;
//        final String resultCache = (String) redisUtils.get(key);
//        if (!StringUtils.hasText(resultCache)) {
//            // 调用启信宝接口
//            String result = qiXinService.getNewsListByName(name, skip, tag_list, sentiment, company_sentiment,
//                class_types, new_tag_list, company_new_tag, create_time_range);
//            redisUtils.set(key, result, 1, TimeUnit.DAYS);
//            return TwOutputUtil.ok(result);
//        }
        // 调用启信宝接口
        String result = qiXinService.getNewsListByName(name, skip, tag_list, sentiment, company_sentiment,
            class_types, new_tag_list, company_new_tag, create_time_range);
        return TwOutputUtil.ok(result);
    }

    /**
     * 1.41 工商照面
     * 企业工商照面及相关信息，包括统一社会信用代码、注册资本、经营范围、企业法定代表人等
     *
     * @param keyword 企业全名/注册号/统一社会信用代码
     * @return
     */
    @GetMapping("/getBasicInfo")
    public TwOutputUtil getBasicInfo(String keyword) {
        if (!StringUtils.hasText(keyword)) {
            log.error("启信宝 工商照面 缺少参数 keyword");
            return TwOutputUtil.error("", "参数缺失", null);
        }
        String result = qiXinService.getBasicInfo(keyword);
        return TwOutputUtil.ok(result);
    }

    /**
     * 1.17 企业LOGO
     *
     * @param name 企业全名
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/getEntLogoByName")
    public TwOutputUtil getEntLogoByName(String name) {
        if (!StringUtils.hasText(name)) {
            log.error("启信宝 企业LOGO 缺少参数 name");
            return TwOutputUtil.error("", "参数缺失", null);
        }
        String result = qiXinService.getEntLogoByName(name);
        return TwOutputUtil.ok(result);
    }

    /**
     * 1.16 企业简介
     *
     * @param name 企业全名
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/getEntBriefByName")
    public TwOutputUtil getEntBriefByName(String name) {
        if (!StringUtils.hasText(name)) {
            log.error("启信宝 企业简介 缺少参数 name");
            return TwOutputUtil.error("", "参数缺失", null);
        }
        String result = qiXinService.getEntBriefByName(name);
        return TwOutputUtil.ok(result);
    }

    /**
     * 1.53 企业年报网址
     * 企业年报网址信息，包括网站名称、网址、网址类型、网址来源、审核时间等
     *
     * @param keyword 企业全名/注册号/统一社会信用代码
     * @param skip    跳过条目数（默认为0，单页返回20条数据）
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/getGetWebsitesUri")
    public TwOutputUtil getGetWebsitesUri(String keyword, String skip) {
        if (!StringUtils.hasText(keyword)) {
            log.error("启信宝 企业年报网址 缺少参数 keyword");
            return TwOutputUtil.error("", "参数缺失", null);
        }
        String result = qiXinService.getGetWebsites(keyword, skip);
        return TwOutputUtil.ok(result);
    }

    /**
     * 1.43 工商股东
     *
     * @param keyword 企业全名/注册号/统一社会信用代码
     * @param skip    跳过条目数（默认为0，单页返回20条数据）
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/getPartners")
    public TwOutputUtil getPartners(String keyword, String skip) {
        if (!StringUtils.hasText(keyword)) {
            log.error("启信宝 工商股东 缺少参数 keyword");
            return TwOutputUtil.error("", "参数缺失", null);
        }
        String result = qiXinService.getPartners(keyword, skip);
        return TwOutputUtil.ok(result);
    }

    /**
     * 1.45 主要人员
     * 企业工商主要人员信息，包括主要人员姓名及职位
     * <p>
     * 单价：0.1元/次
     *
     * @param keyword 企业全名/注册号/统一社会信用代码
     * @param skip    跳过条目数（默认为0，单页返回20条数据）
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/getGetEmployees")
    public TwOutputUtil getGetEmployees(String keyword, String skip) {
        if (!StringUtils.hasText(keyword)) {
            log.error("启信宝 主要人员 缺少参数 keyword");
            return TwOutputUtil.error("", "参数缺失", null);
        }
        String result = qiXinService.getGetEmployees(keyword, skip);
        return TwOutputUtil.ok(result);
    }


}
