package com.elitesland.tw.tw5.server.prd.system.controller;

import com.elitesland.tw.tw5.api.prd.system.payload.PrdSystemSelectionPayload;
import com.elitesland.tw.tw5.api.prd.system.query.PrdSystemSelectionQuery;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemSelectionService;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemSelectionVO;
import com.elitesland.tw.tw5.api.common.TwOutputUtil;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 系统选择项管理
 *
 * @author wangding
 */
@Api("系统选择项")
@RestController
@RequiredArgsConstructor
//@RequestMapping(API_SYSTEM)
@RequestMapping("/api/system")

@Slf4j
public class PrdSystemSelectionController {

    private final PrdSystemSelectionService service;
    private final CacheUtil cacheUtil;

    /**
     * 新建
     *
     * @param entity po类
     * @return result
     */
    @ApiOperation("新增")
    @PostMapping("/systemSelection")
    public TwOutputUtil insert(PrdSystemSelectionPayload entity) {
        PrdSystemSelectionVO vo = service.insert(entity);
        return TwOutputUtil.ok(vo);
    }

    /**
     * 清空消息提醒缓存
     *
     * @return result
     */
    @ApiOperation("清空消息提醒缓存")
    @GetMapping("/systemSelection/clearCache")
    public TwOutputUtil clearCache() {
        cacheUtil.loadSystemSelectionCache();
        return TwOutputUtil.ok();
    }

    /**
     * 全部修改
     *
     * @param entity po类
     * @return result
     */
    @ApiOperation("修改")
    @PutMapping("/systemSelection")
    public TwOutputUtil update(PrdSystemSelectionPayload entity) {
        service.update(entity);
        return TwOutputUtil.ok();
    }

    /**
     * 物理删除
     *
     * @param keys 主键
     * @return result
     */
    @ApiOperation("物理删除")
    @DeleteMapping("/systemSelection")
    public TwOutputUtil delete(Long[] keys) {
        service.delete(Arrays.asList(keys));
        return TwOutputUtil.ok();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主键
     * @return result
     */
    @ApiOperation("逻辑删除")
    @PatchMapping("/systemSelection")
    public TwOutputUtil logicalDelete(Long[] keys) {
        service.deleteSoft(Arrays.asList(keys));
        return TwOutputUtil.ok();
    }

    /**
     * 根据主键查询
     *
     * @param key 主键
     * @return result
     */
    @ApiOperation("主键查询")
    @GetMapping("/systemSelection/{key}")
    public TwOutputUtil findByKey(@PathVariable Long key) {
        PrdSystemSelectionVO vo = service.queryByKey(key);
        return TwOutputUtil.ok(vo);
    }

    /**
     * 分页查询
     *
     * @param query 查询条件
     * @return result
     */
    @ApiOperation("分页查询")
    @GetMapping("/systemSelection/paging")
    public TwOutputUtil findByConditionPaging(PrdSystemSelectionQuery query) {
        PagingVO<PrdSystemSelectionVO> paging = service.paging(query);
        return TwOutputUtil.ok(paging);
    }

    /**
     * 列表查询
     *
     * @param query 查询条件
     * @return result
     */
    @ApiOperation("列表查询")
    @GetMapping("/systemSelection/list")
    public TwOutputUtil queryList(PrdSystemSelectionQuery query) {
        List<PrdSystemSelectionVO> paging = service.queryList(query);
        return TwOutputUtil.ok(paging);
    }

    /**
     * 通过KEY查询直属子集
     *
     * @param key 下拉选择key
     * @return result
     */
    @ApiOperation("通过KEY查询直属子集")
    @GetMapping("/systemSelection/byKey/{key}")
    public TwOutputUtil getSystemSelectionByKey(@PathVariable String key) {
        List<PrdSystemSelectionVO> collect = service.selectByCondition(key);
        return TwOutputUtil.ok(collect);
    }

    /**
     * 通过KEY查询树状子集
     *
     * @param key 下拉选择key
     * @return result
     */
    @ApiOperation("通过KEY查询树状子集")
    @GetMapping("/systemSelection/byKeyAll/{key}")
    public TwOutputUtil getSystemSelectionAllByKey(@PathVariable String key) {
        PrdSystemSelectionVO view = cacheUtil.getSystemSelection(key);
        if (view == null || view.getChildren() == null || view.getChildren().isEmpty()) {
            return TwOutputUtil.ok(Collections.emptyList());
        }
        return TwOutputUtil.ok(view.getChildren());
    }

    /**
     * 根据父节点key查询子节点list，可以对子节点进行过滤
     * @param query 父节点key，及子节点过滤条件
     * @return
     */
    @GetMapping("/systemSelection/childrenList")
    public TwOutputUtil getSystemSelectionChildrenList(PrdSystemSelectionQuery query){
        long t = System.currentTimeMillis();
        var data = service.getSystemSelectionChildrenList(query);
        log.info("getSystemSelectionChildrenList耗时：{}ms", System.currentTimeMillis() - t);
        return TwOutputUtil.ok(data);
    }


    /**
     * 查询级联系统选择项
     *
     * @param key            下拉选择key
     * @param cascaderValues 级联各级值
     * @return result
     */
    @ApiOperation("查询级联系统选择项")
    @GetMapping("/systemSelection/cascader")
    public TwOutputUtil getSystemSelectionByKey(String key, String[] cascaderValues) {
        PrdSystemSelectionVO view = cacheUtil.getSystemSelection(key);
        if (view == null || view.getChildren() == null || view.getChildren().isEmpty()) {
            return TwOutputUtil.ok(Collections.emptyList());
        }
        List<PrdSystemSelectionVO> children = view.getChildren();

        if (cascaderValues != null) {
            for (int i = 0; i < cascaderValues.length; i++) {
                int valueIndex = i;
                children = children.stream().filter(selectionView -> selectionView.getSelectionValue().equals(cascaderValues[valueIndex]))
                        .findFirst()
                        .orElse(new PrdSystemSelectionVO()).getChildren();
                if (children == null) {
                    children = Collections.emptyList();
                }
            }
        }

        List<PrdSystemSelectionVO> list = new ArrayList<>(children.size());
        children.forEach(selectionView -> list.add(selectionView.clone()));
        list.forEach(selectionView -> selectionView.setChildren(null));
        List<PrdSystemSelectionVO> collect = list.stream().sorted(Comparator.comparing(selectionView -> selectionView.getSortNo() == null ? Integer.MAX_VALUE : selectionView.getSortNo())).collect(Collectors.toList());
        return TwOutputUtil.ok(collect);
    }

}
