package com.elitesland.tw.tw5.server.prd.system.service;

import com.elitesland.tw.tw5.api.common.TwCommonPayload;
import com.elitesland.tw.tw5.api.common.TwException;
import com.elitesland.tw.tw5.api.prd.system.payload.PrdSystemRemindPayload;
import com.elitesland.tw.tw5.api.prd.system.query.PrdSystemRemindQuery;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemRemindService;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemRemindVO;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.system.convert.PrdSystemRemindConvert;
import com.elitesland.tw.tw5.server.prd.system.dao.PrdSystemRemindDAO;
import com.elitesland.tw.tw5.server.prd.system.entity.PrdSystemRemindDO;
import com.elitescloud.cloudt.common.base.PagingVO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

/**
 * 消息提醒service
 *
 * @author wangding
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class PrdSystemRemindServiceImpl implements PrdSystemRemindService {
    private final PrdSystemRemindDAO dao;
    private final CacheUtil cacheUtil;

    /**
     * 新建
     *
     * @param payload po类
     * @return result
     */
    @Transactional
    @Override
    public PrdSystemRemindVO insert(PrdSystemRemindPayload payload) {
        String remindCode = "SYS:E:REMIND_CODE_NOT_REPEAT";
        // 判断是否允许return
        boolean allowReturn = allowReturn(remindCode, payload);
        if (allowReturn) {
            // 检查编码不能重复
            PrdSystemRemindQuery query = new PrdSystemRemindQuery();
            query.setRemindCode(payload.getRemindCode());
            int count = dao.queryListDynamic(query).size();
            if (count > 0) {
                throw TwException.error("", "编号不可重复");
            }
        }
        PrdSystemRemindDO ado = PrdSystemRemindConvert.INSTANCE.toDo(payload);
        dao.save(ado);
        cacheUtil.loadRemindCache();
        return PrdSystemRemindConvert.INSTANCE.toVo(ado);
    }

    /**
     * 修改
     *
     * @param payload po类
     * @return result
     */
    @Transactional
    @Override
    public Long update(PrdSystemRemindPayload payload) {
        String remindCode = "SYS:E:REMIND_CODE_NOT_REPEAT";
        // 判断是否允许return
        boolean allowReturn = allowReturn(remindCode, payload);
        if (allowReturn) {
            // 检查编码不能重复
            PrdSystemRemindQuery query = new PrdSystemRemindQuery();
            query.setRemindCode(payload.getRemindCode());
            int count = dao.queryListDynamic(query).size();
            if (count > 0) {
                throw TwException.error("", "编号不可重复");
            }
        }
        dao.updateByKeyDynamic(payload);
        cacheUtil.loadRemindCache();
        return 0L;
    }

    /**
     * 物理删除
     *
     * @param keys ids
     * @return result
     */
    @Transactional
    @Override
    public boolean delete(List<Long> keys) {
        dao.delete(keys);
        cacheUtil.loadRemindCache();
        return true;
    }

    /**
     * 逻辑删除
     *
     * @param keys ids
     * @return result
     */
    @Transactional
    @Override
    public boolean deleteSoft(List<Long> keys) {
        dao.deleteSoft(keys);
        cacheUtil.loadRemindCache();
        return true;
    }

    /**
     * 主键查询
     *
     * @param key id
     * @return result
     */
    @Override
    public PrdSystemRemindVO queryByKey(Long key) {
        return dao.queryByKey(key);
    }

    /**
     * 分页查询
     *
     * @param query po类
     * @return result
     */
    @Override
    public PagingVO<PrdSystemRemindVO> paging(PrdSystemRemindQuery query) {
        return dao.queryPaging(query);
    }

    /**
     * 修改
     *
     * @param remindCode
     * @return result
     */
    @Override
    public boolean allowReturn(String remindCode, TwCommonPayload payload) {
        PrdSystemRemindVO remind = cacheUtil.getRemind(remindCode);
        if (remind == null) {
            return true;
        }
        if ("ERROR".equals(remind.getRemindType())) {
            return true;
        }
        if ("INFO".equals(remind.getRemindType())) {
            return false;
        }
        if ("WARN".equals(remind.getRemindType())) {
            return payload == null || !payload.isIgnoreWarning();
        }
        return true;
    }

    @Override
    public List<PrdSystemRemindVO> selectByCondition(PrdSystemRemindQuery query) {
        return dao.queryListDynamic(query);
    }
}
