package com.elitesland.tw.tw5.server.log.service;

import com.elitesland.tw.tw5.server.log.constant.ApiRequestLogChannelEnum;
import com.elitesland.tw.tw5.server.log.constant.ApiRequestLogTypeEnum;
import com.elitesland.tw.tw5.server.log.dto.ApiRequestLogQuery;
import com.elitesland.tw.tw5.server.log.entity.ApiRequestLogDO;
import com.elitesland.tw.tw5.server.log.repo.ApiRequestLogRepo;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.util.PageUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestMethod;

import java.util.List;
import java.util.Optional;

/**
 * 接口请求日志记录服务 实现
 *
 * @author duwh
 * @date 2022/11/10
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class ApiRequestLogServiceImpl implements ApiRequestLogService {

    private final ApiRequestLogRepo repo;

    /**
     * 最大长度参数
     */
    private final static int MAX_LENGTH_PARAM = 3000;
    /**
     * 最大长度结果
     */
    private final static int MAX_LENGTH_RESULT = 5000;

    @Override
    public void saveLog(ApiRequestLogTypeEnum type, ApiRequestLogChannelEnum channel, String uri, RequestMethod requestMethod, String param, String header, String result, String status, String remark) {
        ApiRequestLogDO logDO = new ApiRequestLogDO(type, channel, uri, requestMethod, param, header, result, status, remark);
        saveLog(logDO);
    }

    @Override
    public void saveLog(ApiRequestLogTypeEnum type, ApiRequestLogChannelEnum channel, String uri, RequestMethod requestMethod, String param, String header, String result, String status, String remark, String ext1, String ext2, String ext3, String ext4) {
        ApiRequestLogDO logDO = new ApiRequestLogDO(type, channel, uri, requestMethod, param, header, result, status, remark, ext1, ext2, ext3, ext4);
        saveLog(logDO);
    }

    @Override
    public void saveLog(ApiRequestLogDO logDO) {
        String param = logDO.getParam();
        if (StringUtils.hasText(param)) {
            if (param.length() > MAX_LENGTH_PARAM) {
                logDO.setParam(param.substring(0, MAX_LENGTH_PARAM));
            }
        }
        String result = logDO.getResult();
        if (StringUtils.hasText(result)) {
            if (result.length() > MAX_LENGTH_RESULT) {
                logDO.setResult(result.substring(0, MAX_LENGTH_RESULT));
            }
        }
        repo.save(logDO);
    }

    @Override
    public void saveOutLog(ApiRequestLogTypeEnum type, String uri, RequestMethod requestMethod, String param, String header, String result, String status) {
        ApiRequestLogDO logDO = new ApiRequestLogDO(type, ApiRequestLogChannelEnum.OUT, uri, requestMethod, param, header, result, status, "");
        saveLog(logDO);
    }

    @Override
    public void saveOutLog(ApiRequestLogTypeEnum type, String uri, RequestMethod requestMethod, String param, String header, String result, String status, Long time) {
        ApiRequestLogDO logDO = new ApiRequestLogDO(type, ApiRequestLogChannelEnum.OUT, uri, requestMethod, param, header, result, status, "");
        logDO.setTime(time);
        saveLog(logDO);
    }

    @Override
    public void saveOutLog(ApiRequestLogTypeEnum type, String uri, RequestMethod requestMethod, String param, String header, String result, String status, String remark) {
        ApiRequestLogDO logDO = new ApiRequestLogDO(type, ApiRequestLogChannelEnum.OUT, uri, requestMethod, param, header, result, status, remark);
        saveLog(logDO);
    }

    @Override
    public void saveInLog(ApiRequestLogTypeEnum type, String uri, RequestMethod requestMethod, String param, String header, String result, String status) {
        ApiRequestLogDO logDO = new ApiRequestLogDO(type, ApiRequestLogChannelEnum.IN, uri, requestMethod, param, header, result, status, "");
        saveLog(logDO);
    }

    @Override
    public void saveInLog(ApiRequestLogTypeEnum type, String uri, RequestMethod requestMethod, String param, String header, String result, String status, String remark) {
        ApiRequestLogDO logDO = new ApiRequestLogDO(type, ApiRequestLogChannelEnum.IN, uri, requestMethod, param, header, result, status, remark);
        saveLog(logDO);
    }

    @Override
    public ApiRequestLogDO queryByKey(Long key) {
        ApiRequestLogDO entity = repo.findById(key).orElseGet(ApiRequestLogDO::new);
        Assert.notNull(entity.getId(), "日志不存在");
        return entity;
    }

    @Override
    public PagingVO<ApiRequestLogDO> paging(ApiRequestLogQuery query) {
        //默认按照时间倒叙排序
        OrderItem orderItem = OrderItem.desc("createTime");
        query.defaultOrder(orderItem);
        Page<ApiRequestLogDO> page = repo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder), query.getPageRequest());
        return PageUtil.toPageVo(page);
    }

    @Override
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            keys.stream().forEach(id -> {
                Optional<ApiRequestLogDO> optional = repo.findById(id);
                if (!optional.isEmpty()) {
                    ApiRequestLogDO entity = optional.get();
                    entity.setDeleteFlag(1);
                    repo.save(entity);
                }
            });
        }
    }

    @Override
    public void delete(List<Long> keys) {
        repo.deleteAllById(keys);
    }
}
