package com.elitesland.tw.tw5.server.prd.crm.controller;

import com.elitesland.tw.tw5.api.prd.crm.payload.CrmActPlanDetailPayload;
import com.elitesland.tw.tw5.api.prd.crm.payload.CrmActPlanPayload;
import com.elitesland.tw.tw5.api.prd.crm.query.CrmActPlanQuery;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmActPlanService;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmActPlanDetailVO;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmActPlanVO;
import com.elitesland.tw.tw5.server.common.TwOutputUtil;
import io.seata.spring.annotation.GlobalTransactional;
import io.swagger.annotations.Api;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.format.annotation.DateTimeFormat;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.time.LocalDateTime;

/**
 * 市场计划管理
 *
 * @author carl.wang
 * @date 2022/5/30
 */
@Api("市场计划管理")
@RestController
@RequiredArgsConstructor
@RequestMapping("/api/crm")
@Slf4j
public class CrmActPlanController {

    private final CrmActPlanService service;

    /**
     * 新建
     *
     * @param payload po类
     * @return result
     */
    @PostMapping("/plan/insert")
    public TwOutputUtil insert(CrmActPlanPayload payload) {
        CrmActPlanVO vo = service.insert(payload);
        return TwOutputUtil.ok(vo);
    }

    /**
     * 提交审核
     *
     * @param id 主键
     * @return result
     */
    @GetMapping("/plan/submitCheck/{id}")
    //@GlobalTransactional
    public TwOutputUtil submitCheck(@PathVariable Long id) {
        service.submitCheck(id);
        return TwOutputUtil.ok();
    }

    /**
     * 修改
     *
     * @param payload po类
     * @return result
     */
    @PutMapping("/plan/update")
    //@GlobalTransactional
    public TwOutputUtil update(CrmActPlanPayload payload) {
        service.update(payload);
        return TwOutputUtil.ok();
    }

    /**
     * 删除计划
     *
     * @param key 主键
     * @return result
     */
    @DeleteMapping("/plan/deleteSoft")
    public TwOutputUtil deleteSoft(Long key) {
        service.deleteSoft(key);
        return TwOutputUtil.ok();
    }


    /**
     * 打开变更模式，返回结果
     *
     * @param planId 计划id
     * @return
     */
    @GetMapping("/plan/changeModeData/{planId}")
    public TwOutputUtil getChangeModeData(@PathVariable Long planId) {
        return TwOutputUtil.ok(service.changeLogDetailByPlanId(planId));
    }


    /**
     * 查询变更历史
     */
    @GetMapping("/plan/changeLogList/{planId}")
    public TwOutputUtil changeLogList(@PathVariable Long planId) {
        return TwOutputUtil.ok(service.changeLogList(planId));
    }


    /**
     * 分页查询
     *
     * @param query 查询条件
     * @return result
     */
    @GetMapping("/plan/paging")
    public TwOutputUtil paging(CrmActPlanQuery query) {
        return TwOutputUtil.ok(service.paging(query));
    }

    /**
     * 批量导出计划excel
     *
     * @param query 查询条件
     * @return result
     */
    @GetMapping("/plan/downloadPlans")
    public void downloadPlans(HttpServletResponse response, CrmActPlanQuery query) {
        service.downloadPlans(response, query);
    }

    /**
     * 批量导出计划明细excel
     *
     * @param planId 市场计划id
     * @return result
     */
    @GetMapping("/plan/downloadPlanDetails")
    public void downloadPlanDetails(HttpServletResponse response, Long planId) {
        service.downloadPlanDetails(response, planId);
    }

    /**
     * 根据开始时间列表查询
     *
     * @param newDate 查询开始时间
     * @return result
     */
    @GetMapping("/plan/queryPlans")
    public TwOutputUtil queryPlans(@DateTimeFormat(pattern = "yyyy-MM-dd HH:mm:ss") LocalDateTime newDate) {
        return TwOutputUtil.ok(service.queryPlans(newDate));
    }

    /**
     * 详情查询
     *
     * @param id 主键
     * @return result
     */
    @GetMapping("/plan/queryByKey/{id}")
    public TwOutputUtil queryByKey(@PathVariable Long id) {
        CrmActPlanVO vo = service.queryByKey(id);
        return TwOutputUtil.ok(vo);
    }

//    /**
//     * 根据当前单据查询当前版本的变更记录并比较
//     * @param id 主键查询
//     * @return ApiResult
//     */
//    @GetMapping("/plan/findByKeyWithChange/{id}")
//    public TwOutputUtil findChangeDataById(@PathVariable("id") Long id){
//        Map<String,Object> result = service.findChangeDataById(id);
//        return TwOutputUtil.ok(result);
//    }

    /**
     * 计划关闭
     *
     * @param key        主键
     * @param closeReson 关闭原因
     * @return result
     */
    @GetMapping("/plan/closePlan")
    public TwOutputUtil closePlan(Long key, String closeReson) {
        service.closePlan(key, closeReson);
        return TwOutputUtil.ok();
    }

    /**
     * 新建明细
     *
     * @param payload po类
     * @return result
     */
    @PostMapping("/plan/insertDetail")
    public TwOutputUtil insertDetail(CrmActPlanDetailPayload payload) {
        CrmActPlanDetailVO vo = service.insertDetail(payload);
        return TwOutputUtil.ok(vo);
    }

    /**
     * 修改明细
     *
     * @param payload po类
     * @return result
     */
    @PutMapping("/plan/updateDetail")
    public TwOutputUtil updateDetail(CrmActPlanDetailPayload payload) {
        service.updateDetail(payload);
        return TwOutputUtil.ok();
    }

    /**
     * 删除明细
     *
     * @param planId      市场活动计划id
     * @param key         市場计划明细主键id
     * @param changeLogId 更新日志的id(当数据尚存在于日志中时，处理其删除问题)
     * @return result
     */
    @DeleteMapping("/plan/deleteDetailSoft")
    public TwOutputUtil deleteDetailSoft(Long planId, Long key, Long changeLogId) {
        service.deleteDetailSoft(planId, key, changeLogId);
        return TwOutputUtil.ok();
    }

    /**
     * 市場计划明细列表查询
     *
     * @param planId  市场活动计划id
     * @param newDate 活动开始时间
     * @return result
     */
    @GetMapping("/plan/queryDetails")
    public TwOutputUtil queryDetails(Long planId, @DateTimeFormat(pattern = "yyyy-MM-dd HH:mm:ss") LocalDateTime newDate) {
        return TwOutputUtil.ok(service.queryDetails(planId, newDate));
    }

//    /**
//     * 根据当前单据查询当市场预算详情前版本的变更记录并比较
//     * @param id 主键查询
//     * @return ApiResult
//     */
//    @GetMapping("/plan/findDetailsByKeyWithChange/{id}")
//    public TwOutputUtil findDetailChangeDataById(@PathVariable("id") Long id,@DateTimeFormat(pattern = "yyyy-MM-dd HH:mm:ss") LocalDateTime newDate){
//        List<Map<String,Object>> result = service.findDetailsChangeDataById(id,newDate);
//        return TwOutputUtil.ok(result);
//    }

    /**
     * 市場计划工种查询
     *
     * @return result
     */
    @GetMapping("/plan/queryWorkTypes")
    public TwOutputUtil queryWorkTypes() {
        return TwOutputUtil.ok(service.queryWorkTypes());
    }


}
