package com.elitesland.tw.tw5.server.prd.crm.controller;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.write.builder.ExcelWriterSheetBuilder;
import com.alibaba.excel.write.style.column.LongestMatchColumnWidthStyleStrategy;
import com.elitesland.tw.tw5.api.prd.crm.payload.CrmCustomerOperationExcelExport;
import com.elitesland.tw.tw5.api.prd.crm.payload.CrmCustomerOperationPayload;
import com.elitesland.tw.tw5.api.prd.crm.query.CrmCustomerOperationBusinessQuery;
import com.elitesland.tw.tw5.api.prd.crm.query.CrmCustomerOperationQuery;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmCustomerOperationService;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmCustomerDataVO;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmCustomerOperationSimpleVO;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmCustomerOperationVO;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmFollowVO;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemLogVO;
import com.elitesland.tw.tw5.server.common.TwOutputUtil;
import com.elitesland.tw.tw5.server.common.excel.ExcelUtil;
import com.elitesland.tw.tw5.server.log.constant.ApiRequestLogTypeEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.RoleEnum;
import com.elitesland.tw.tw5.server.prd.qixin.service.QiXinService;
import com.elitesland.tw.tw5.server.prd.system.dao.PrdSystemRoleDAO;
import com.elitesland.tw.tw5.server.prd.tianyancha.service.TianYanChaService;
import com.elitesland.tw.tw5.server.udc.UdcNameClass;
import com.elitesland.tw.tw5.server.udc.UdcUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.crm.convert.CrmCustomerOperationConvert;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.task.TaskExecutor;
import org.springframework.http.MediaType;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.URLEncoder;
import java.util.*;
import java.util.concurrent.CompletableFuture;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toCollection;

/**
 * 客户经营
 *
 * @author duwh
 * @date 2022/11/15
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/api/crm")
@Slf4j
public class CrmCustomerOperationController {

    private final CrmCustomerOperationService service;
    private final TianYanChaService tianYanChaService;
    private final QiXinService qiXinService;
    private final PrdSystemRoleDAO systemRoleDAO;
    private final UdcUtil udcUtil;
    private final TaskExecutor taskExecutor;

    /**
     * 新增
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PostMapping("/customerOperation")
    public TwOutputUtil insert(@RequestBody CrmCustomerOperationPayload payload) {
        return TwOutputUtil.ok(service.insert(payload));
    }

    /**
     * excel导入
     *
     * @param file     文件
     * @param force    力
     * @param response 响应
     * @return {@link TwOutputUtil}
     * @throws IOException ioexception
     */
    @PostMapping(value = "/customerOperation/excelImport", consumes = MediaType.MULTIPART_FORM_DATA_VALUE)
    public TwOutputUtil excelImport(MultipartFile file, Boolean force, HttpServletResponse response) throws IOException {
        Map<String, Object> resultMap = service.excelImport(file, force, response);
        //if (resultMap.get("ok").equals("ok")) {
        //    return TwOutputUtil.ok();
        //} else if (resultMap.get("ok").equals("warn")) {
        //    return TwOutputUtil.warn("", "", resultMap);
        //} else {
        //    return TwOutputUtil.error("", "", resultMap);
        //}
        return TwOutputUtil.ok(resultMap);
    }

    @GetMapping("/customerOperation/excelExport")
    @UdcNameClass
    public void excelExport(CrmCustomerOperationQuery query, HttpServletRequest request, HttpServletResponse response) throws IOException {
        //定义文件名称
        String sheetName = "客户经营";
        //导出顺序是反的
        OrderItem orderItem = OrderItem.asc("createTime");
        query.defaultOrder(orderItem);
        //获取想要导出的数据
        List<CrmCustomerOperationVO> vos = queryListCommon(query);
        List<CrmCustomerOperationExcelExport> resultList = CrmCustomerOperationConvert.INSTANCE.voListVoExcelExport(vos);
        resultList = udcUtil.translateList(resultList);
        //对文件名进行固定格式编码
        String fileName = URLEncoder.encode(sheetName + System.currentTimeMillis() + ".xlsx", "UTF-8");
        //设置请求响应内容类型
        //作用:使客户端浏览器，区分不同种类的数据，并根据不同的MIME调用浏览器内不同的程序嵌入模块来处理相应的数据。
        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        //设置请求响应内容编码方式
        response.setCharacterEncoding("utf-8");
        //文件下载，指定默认名
        response.addHeader("Content-Disposition", "attachment;filename=" + fileName);

        final ExcelWriterSheetBuilder sheet = EasyExcel.write(response.getOutputStream(), CrmCustomerOperationVO.class)
                .registerWriteHandler(new LongestMatchColumnWidthStyleStrategy())
                .sheet(sheetName);
        // 列
        ExcelUtil.excelHelper(sheet, CrmCustomerOperationExcelExport.class, null);
        //写入
        sheet.doWrite(resultList);
    }

    /**
     * 更新
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PutMapping("/customerOperation")
    public TwOutputUtil update(CrmCustomerOperationPayload payload) {
        return TwOutputUtil.ok(service.update(payload));
    }

    /**
     * 修改客户经营状态
     *
     * @param key            主键
     * @param custOperStatus 状态
     * @return result
     */
    @GetMapping("/customerOperation/updateCustOperStatus")
    public TwOutputUtil updateCustOperStatus(Long key, String custOperStatus) {
        service.updateCustOperStatus(key, custOperStatus);
        return TwOutputUtil.ok();
    }

    /**
     * 关联客户
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PutMapping("/customerOperation/relationCustomer")
    public TwOutputUtil relationCustomer(CrmCustomerOperationPayload payload) {
        service.relationCustomer(payload);
        return TwOutputUtil.ok();
    }

    /**
     * 主键查询
     *
     * @param key 关键
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/customerOperation/{key}")
    @UdcNameClass
    public TwOutputUtil queryOneByKey(@PathVariable Long key) {
        CrmCustomerOperationVO prdFileVO = service.queryByKey(key);
        return TwOutputUtil.ok(prdFileVO);
    }

    /**
     * 检查 名称 唯一性
     *
     * @param custName cust名字
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/customerOperation/checkCustNameUnique")
    public TwOutputUtil checkCustNameUnique(String custName) {
        return TwOutputUtil.ok(service.checkCustNameUnique(custName));
    }

    /**
     * 操作记录日志列表
     *
     * @param key 客户经营主键
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/customerOperation/log/list/{key}")
    @UdcNameClass
    public TwOutputUtil logList(@PathVariable Long key) {
        final List<PrdSystemLogVO> prdSystemLogVOS = service.queryLogList(key);
        return TwOutputUtil.ok(prdSystemLogVOS);
    }

    /**
     * 分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/customerOperation/paging")
    @UdcNameClass
    //@ReSubmitCheck(argExpressions = {"[0].current","[0].size"}, interval = 500, message = "请勿重复提交")
    public TwOutputUtil paging(CrmCustomerOperationQuery query) {
        final Long loginUserId = GlobalUtil.getLoginUserId();
        query.setUserId(loginUserId);
        // 判断 管理员
        final boolean isSystemAdmin = GlobalUtil.getLoginGeneralUser().isSystemAdmin();
        PagingVO<CrmCustomerOperationVO> paging;
        if (isSystemAdmin) {
            paging = service.paging(query);
        } else {
            // 判断当前登录人是否是系统管理员or 拥有角色【客户经营管理员	CUST_OPER_MANAGER】
            List<Long> userIdsByRole = systemRoleDAO.queryUserIdByRoleCodes(Arrays.asList(RoleEnum.SYS.getCode(), RoleEnum.CUST_OPER_MANAGER.getCode()));
            if (!CollectionUtils.isEmpty(userIdsByRole) && userIdsByRole.contains(loginUserId)) {
                paging = service.paging(query);
            } else {
                // 数据权限 创建人 、客户经营团队成员、经营计划执行者
                paging = service.pagingDataFilter(query);
            }
        }
        return TwOutputUtil.ok(paging);
    }

    /**
     * 查询列表
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/customerOperation/list")
    @UdcNameClass
    public TwOutputUtil queryList(CrmCustomerOperationQuery query) {
        List<CrmCustomerOperationVO> vos = queryListCommon(query);
        return TwOutputUtil.ok(vos);
    }

    private List<CrmCustomerOperationVO> queryListCommon(CrmCustomerOperationQuery query) {
        //数据处理
        final String idListStr = query.getIdListStr();
        if (StringUtils.hasText(idListStr)) {
            final String[] split = idListStr.split(",");
            List<Long> idList = new ArrayList<>();
            for (String s : split) {
                idList.add(Long.valueOf(s));
            }
            query.setIdList(idList);
        }

        final Long loginUserId = GlobalUtil.getLoginUserId();
        query.setUserId(loginUserId);
        // 判断 管理员
        final boolean isSystemAdmin = GlobalUtil.getLoginGeneralUser().isSystemAdmin();
        List<CrmCustomerOperationVO> vos;
        if (isSystemAdmin) {
            vos = service.queryList(query);
        } else {
            // 判断当前登录人是否是系统管理员or 拥有角色【客户经营管理员	CUST_OPER_MANAGER】
            List<Long> userIdsByRole = systemRoleDAO.queryUserIdByRoleCodes(Arrays.asList(RoleEnum.SYS.getCode(), RoleEnum.CUST_OPER_MANAGER.getCode()));
            if (!CollectionUtils.isEmpty(userIdsByRole) && userIdsByRole.contains(loginUserId)) {
                vos = service.queryList(query);
            } else {
                // 数据权限 创建人 、客户经营团队成员、经营计划执行者
                vos = service.queryListDataFilter(query);
            }
        }
        // 级联查询 查到第一层
        if (null != query.getTreeSearch() && query.getTreeSearch()) {
            if (!CollectionUtils.isEmpty(vos)) {
                //vos.addAll();
                final List<CrmCustomerOperationVO> collection = treeDataSearch(vos);
                if (!CollectionUtils.isEmpty(collection)) {
                    vos.addAll(collection);
                }
                // 数据去重
                vos = vos.stream().collect(
                        collectingAndThen(
                                toCollection(() -> new TreeSet<>(Comparator.comparing(CrmCustomerOperationVO::getId))), ArrayList::new)
                );
            }
        }
        return vos;
    }

    /**
     * 级联查询母公司数据
     *
     * @param vos
     * @return {@link List}<{@link CrmCustomerOperationVO}>
     */
    private List<CrmCustomerOperationVO> treeDataSearch(List<CrmCustomerOperationVO> vos) {
        List<CrmCustomerOperationVO> result = new ArrayList<>();
        if (!CollectionUtils.isEmpty(vos)) {
            vos.forEach(crmCustomerOperationVO -> {
                final Long parentId = crmCustomerOperationVO.getParentId();
                if (parentId != null) {
                    final CrmCustomerOperationVO operationVO = service.queryByKey(parentId);
                    result.add(operationVO);
                }
            });
            if (!CollectionUtils.isEmpty(treeDataSearch(result))) {
                result.addAll(treeDataSearch(result));
            }
        }
        return result;
    }

    /**
     * 公司列表
     * 可用于公司下拉
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/customerOperation/select/list")
    public TwOutputUtil selectList(CrmCustomerOperationQuery query) {
        List<CrmCustomerOperationSimpleVO> vos = service.selectList(query);
        return TwOutputUtil.ok(vos);
    }

    /**
     * 删除
     *
     * @param keys 主键id
     * @return result
     */
    @DeleteMapping("/customerOperation/deleteSoft")
    public TwOutputUtil deleteSoft(Long[] keys) {
        service.deleteSoft(Arrays.asList(keys));
        return TwOutputUtil.ok();
    }

    /**
     * 查询客户下面的经营往来
     *
     * @param operId 查询
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/queryOperateFollowList/{operId}")
    public TwOutputUtil queryOperateFollowList(@PathVariable Long operId) {
        List<CrmFollowVO> vos = service.queryOperateFollowList(operId);
        return TwOutputUtil.ok(vos);
    }

    /**
     * 分页获取业务往来
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/queryBusinessFollowPaging/paging")
    @UdcNameClass
    public TwOutputUtil queryBusinessFollowPaging(CrmCustomerOperationBusinessQuery query) {
        Object paging = service.queryBusinessFollowPaging(query);
        return TwOutputUtil.ok(paging);
    }

    /**
     * 查询客户信息
     *
     * @param operId 查询
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/queryCustomerByOperId/{operId}")
    public TwOutputUtil queryCustomerByOperId(@PathVariable Long operId) {
        CrmCustomerDataVO customerDataVO = service.queryCustomerByOperId(operId);
        return TwOutputUtil.ok(customerDataVO);
    }

    /**
     * 客户情况概览
     *
     * @param operId 查询
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/queryCustomerView/{operId}")
    public TwOutputUtil queryCustomerView(@PathVariable Long operId) {
        Object view = service.queryCustomerView(operId);
        return TwOutputUtil.ok(view);
    }

    /**
     * 第三方接口 天眼查 -公司搜索
     *
     * @param word     关键词
     * @param pageSize 页面大小
     * @param pageNum  页面num
     * @param type     TYC  天眼查  QXB 启信宝
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/customerOperation/search")
    @Deprecated
    public TwOutputUtil search(String word, Integer pageSize, Integer pageNum, String type) {
        String result = "";
        if (!StringUtils.hasText(type)) {
            type = ApiRequestLogTypeEnum.QXB.getCode();
        }
        if (type.equals(ApiRequestLogTypeEnum.QXB.getCode())) {
            result = qiXinService.search(null, null, word, pageNum + "");
        } else if (type.equals(ApiRequestLogTypeEnum.TYC.getCode())) {
            result = tianYanChaService.search(word, pageSize, pageNum);
        }
        return TwOutputUtil.ok(result);
    }


    /**
     * 第三方接口 启信宝 -公司搜索
     * see QiXinController
     *
     * @param matchType 匹配类型（可选），keyword 需要匹配的 字段，可以输入以下一个或者几个类型， 多个类型使用|分隔： partner：股东 oper：法人 4 member：高管 contact：联系方式 scope：经营范围 name：公司名称 patent：专利 copyright：著作权作品名称 software：软件著作权名称 trademark：商标 domain：网址 product：产品
     * @param region    地区编码（可选），输入参数为空返回全 部区域匹配记录。可传入 省/直辖市，城市，区县的国家区划码。 只支持传入一个区划码。 省/直辖市：传入国家区划码前两位数字， 如搜索“江苏省”则传入 32，各省/直辖 市编码可参考本文档附录 城市：传入国家区划码前四位数字，如搜 索“苏州市”，则传入 3205 区县：传入国家区划码六位数字，如搜索 “苏州市吴中区”，则传入 320506 可以通过下面网页查询国家区划编码 http://www.mca.gov.cn/article/sj/tj bz/a/2017/2018
     * @param keyword   关键字（必填） 企业全名/注册号/统一社会信用代码
     * @param skip      跳过(可选) 跳过条目数（默认为0，单页返回10条数据）
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/customerOperation/searchPro")
    @Deprecated
    public TwOutputUtil searchByQxb(String matchType, String region, String keyword, String skip) {
        String result = qiXinService.search(matchType, region, keyword, skip);
        return TwOutputUtil.ok(result);
    }

    /**
     * 补充启信宝信息
     * @return 失败的客户经营
     */
    @PostMapping("/customerOperation/fillQxbInfo")
    public TwOutputUtil fillQxbInfo(){
        List<CrmCustomerOperationVO> operVOList = service.findOperationEnterpriseInfoAbsent();
        if (CollectionUtils.isEmpty(operVOList)){
            return TwOutputUtil.ok();
        }

        StringBuffer fails = new StringBuffer();
        CompletableFuture<Void>[] futures = new CompletableFuture[operVOList.size()];
        for (int i = 0; i < operVOList.size(); ++i){
            var operVO = operVOList.get(i);
            futures[i] = CompletableFuture.runAsync(() -> {
                try {
                    boolean success = service.fillQxbInfo(operVO.getCustName(), operVO.getId());
                    if (!success){
                        fails.append(operVO.getCustName()+";");
                    }
                } catch (Exception e){
                    log.error(e.getMessage(), e);
                    fails.append(operVO.getCustName()+";");
                }
            }, taskExecutor);
        }
        try {
            CompletableFuture.allOf(futures).get();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }

        if (fails.isEmpty()){
            return TwOutputUtil.ok();
        } else {
            fails.setLength(fails.length() - 1);
            return TwOutputUtil.error("fail", "处理失败的客户经营："+fails, null);
        }
    }
}
