package com.elitesland.tw.tw5.server.prd.crm.controller;

import com.elitesland.tw.tw5.api.prd.crm.payload.CrmFollowPayload;
import com.elitesland.tw.tw5.api.prd.crm.payload.CrmLeadsMembersPayload;
import com.elitesland.tw.tw5.api.prd.crm.payload.CrmLeadsPayload;
import com.elitesland.tw.tw5.api.prd.crm.payload.CrmOpportunityPayload;
import com.elitesland.tw.tw5.api.prd.crm.query.CrmLeadsOffshoreQuery;
import com.elitesland.tw.tw5.api.prd.crm.query.CrmLeadsQuery;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmFollowService;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmLeadsService;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmLeadsVO;
import com.elitesland.tw.tw5.api.prd.system.query.PrdSystemLogQuery;
import com.elitesland.tw.tw5.server.common.TwOutputUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.CrmFollowObjectEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.PrdSystemLogEnum;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.MediaType;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.*;

/**
 * @author zoey
 * 线索管理
 * @date 2022/5/20 - 16:16
 */
@Api("线索管理")
@RestController
@RequiredArgsConstructor
@RequestMapping("/api/crm")
@Slf4j
public class CrmLeadsController {

    private final CrmLeadsService service;
    private final CrmFollowService followService;

    /**
     * 新建或修改
     *
     * @param payload po类
     * @return result
     */
    @PostMapping("/leads/saveOne")
    public TwOutputUtil insert(@RequestBody CrmLeadsPayload payload) {
        CrmLeadsVO vo = service.saveOne(payload);
        return TwOutputUtil.ok(vo);
    }


    /**
     * 分配
     *
     * @param payload po类
     * @return result
     */
    @PostMapping("/leads/distribute")
    public TwOutputUtil distributeLeads(@RequestBody CrmLeadsPayload payload) {
//        service.distributeLeads(payload);
        return TwOutputUtil.ok(service.distributeLeads(payload));
    }

    /**
     * 批量分配
     *
     * @param payloads po类
     * @return result
     */
    @PostMapping("/leads/batchDistribute")
    public TwOutputUtil distributeLeads(@RequestBody List<CrmLeadsPayload> payloads) {
        service.batchDistributeLeads(payloads);
        return TwOutputUtil.ok();
    }

    /**
     * 领取
     *
     * @param id 主键id
     * @return result
     */
    @GetMapping("/leads/receive/{id}")
    public TwOutputUtil receiveLeads(@PathVariable Long id) {
        service.receiveLeads(id);
        return TwOutputUtil.ok();
    }


    /**
     * 收回
     *
     * @param payload po类
     * @return result
     */
    @PostMapping("/leads/withdraw")
    public TwOutputUtil withdrawLeads(@RequestBody CrmLeadsPayload payload) {
        service.withdrawLeads(payload);
        return TwOutputUtil.ok();
    }

    /**
     * 退回
     *
     * @param payload po类
     * @return result
     */
    @PostMapping("/leads/back")
    public TwOutputUtil backLeads(@RequestBody CrmLeadsPayload payload) {
        service.backLeads(payload);
        return TwOutputUtil.ok();
    }

    /**
     * 转移给人
     *
     * @param payload po类
     * @return result
     */
    @PostMapping("/leads/transfer/user")
    public TwOutputUtil transferLeadsUser(@RequestBody CrmLeadsPayload payload) {
        service.transferLeadsUser(payload);
        return TwOutputUtil.ok();
    }

    /**
     * 转移到近海
     *
     * @param payload po类
     * @return result
     */
    @PostMapping("/leads/transfer/offshore")
    public TwOutputUtil transferLeadsOffshore(@RequestBody CrmLeadsPayload payload) {
        service.transferLeadsOffshore(payload);
        return TwOutputUtil.ok();
    }

    /**
     * 批量转移到近海
     *
     * @param payloads po类
     * @return result
     */
    @PostMapping("/leads/batchTransfer/offshore")
    public TwOutputUtil transferLeadsOffshoreBatch(@RequestBody List<CrmLeadsPayload> payloads) {
        service.transferLeadsOffshoreBatch(payloads);
        return TwOutputUtil.ok();
    }

    /**
     * 标记
     *
     * @param payload po类
     * @return result
     */
    @PostMapping("/leads/remark")
    public TwOutputUtil remarkLeads(@RequestBody CrmLeadsPayload payload) {
        service.remarkLeads(payload);
        return TwOutputUtil.ok();
    }

    /**
     * 批量标记
     *
     * @param payloads po类
     * @return result
     */
    @PostMapping("/leads/batchRemark")
    public TwOutputUtil batchRemarkLeads(@RequestBody List<CrmLeadsPayload> payloads) {
        service.batchRemarkLeads(payloads);
        return TwOutputUtil.ok();
    }

    /**
     * 转商机
     *
     * @param opportunityPayload 线索id
     * @param opportunityPayload po类
     * @return result
     */
    @PostMapping("/leads/toOppo/{id}")
    public TwOutputUtil toOpportunity(@PathVariable Long id, @RequestBody CrmOpportunityPayload opportunityPayload) {
        Long opportunityId = service.toOpportunity(id, opportunityPayload);
        return TwOutputUtil.ok(opportunityId);
    }


    /**
     * 关联客户
     *
     * @return result
     */
    @GetMapping("/leads/bindCustomer/{id}")
    public TwOutputUtil bindCustomer(@PathVariable Long id) {
        Map<String, Object> result = new HashMap<>();
        try {
            result = service.bindCustomer(id);
        } catch (Exception e) {
            //删除
            service.deleteSoft(Collections.singletonList(id));
            return TwOutputUtil.error("", "绑定失败，请重试！", "");
        }
        return TwOutputUtil.ok(result);
    }

    /**
     * 批量关联客户
     *
     * @return result
     */
    @GetMapping("/leads/bindCustomerBatch")
    public TwOutputUtil bindCustomer(Long[] ids) {
        service.bindCustomerBatch(ids);
        return TwOutputUtil.ok();
    }
    /**
     * 批量关联未关联客户的线索
     *
     * @return result
     */
    @GetMapping("/leads/bindCustomerNotBind")
    public TwOutputUtil bindCustomerNotBind() {
        service.bindCustomerNotBind();
        return TwOutputUtil.ok();
    }

    /**
     * 对线索历史数据绑定客户
     */
    @GetMapping("/leads/bindCustomers")
    public TwOutputUtil bindCustomers() {
        return TwOutputUtil.ok(service.bindCustomers());
    }


    /**
     * 获取offshore表的字段及名称
     *
     * @return result
     */
    @ApiOperation("获取offshore表的字段及名称")
    @GetMapping("/leads/fields")
    public TwOutputUtil queryFields() {
        Map<String, String> fields = service.queryFields();
        return TwOutputUtil.ok(fields);
    }

    /**
     * 局部字段更新
     *
     * @param payload po类
     * @return result
     */
    @PostMapping("/leads/dynamicUpdate")
    public TwOutputUtil dynamicUpdate(@RequestBody CrmLeadsPayload payload) {
        service.dynamicUpdate(payload, PrdSystemLogEnum.UPDATE.getDesc());
        return TwOutputUtil.ok();
    }


    /**
     * 批量添加标签
     *
     * @param keys        批量修改的列表
     * @param leadsTagIds 要添加的标签列表
     * @return result
     */
    @GetMapping("/leads/batchAddTags")
    public TwOutputUtil batchAddTags(Long[] keys, String leadsTagIds) {
        service.batchAddTags(keys, leadsTagIds);
        return TwOutputUtil.ok();
    }

    /**
     * 批量删除标签
     *
     * @param keys        批量修改的列表
     * @param leadsTagIds 要添加的标签列表
     * @return result
     */
    @GetMapping("/leads/batchDelTags")
    public TwOutputUtil batchDelTags(Long[] keys, String leadsTagIds) {
        service.batchDelTags(keys, leadsTagIds);
        return TwOutputUtil.ok();
    }


    /**
     * 更新成员(增删)
     *
     * @param members po类
     * @return result
     */
    @PostMapping("/leads/updateMembers")
    public TwOutputUtil updateMembers(@RequestBody List<CrmLeadsMembersPayload> members) {
        service.updateMembers(members);
        return TwOutputUtil.ok();
    }


    /**
     * 更新跟进
     *
     * @param payload po类
     * @return result
     */
    @PostMapping("/leads/updateFollow")
    public TwOutputUtil updateFollow(@RequestBody CrmFollowPayload payload) {
        service.updateFollow(payload);
        return TwOutputUtil.ok();
    }


    /**
     * 待领取线索列表查询分页
     *
     * @return result
     */
    @GetMapping("/leads/paging/receive")
    public TwOutputUtil queryListNotReceivedLeads(CrmLeadsOffshoreQuery query) {
        return TwOutputUtil.ok(service.queryListNotReceivedLeadsByUserId(query));
    }

    /**
     * 待分配线索列表查询分页
     *
     * @return result
     */
    @GetMapping("/leads/paging/distribute")
    public TwOutputUtil queryListNotDistributedLeadsByUserId(CrmLeadsOffshoreQuery query) {
        return TwOutputUtil.ok(service.queryListNotDistributedLeadsByUserId(query));
    }


    /**
     * 市场活动线索列表查询分页
     *
     * @param query 查询参数
     * @return result
     */
    @GetMapping("/leads/page/market")
    public TwOutputUtil pageListMarket(CrmLeadsQuery query) {
        return TwOutputUtil.ok(service.pageListMarketLeads(query));
    }


    /**
     * 市场活动线索列表查询不分页
     *
     * @param query 查询参数
     * @return result
     */
    @GetMapping("/leads/list/market")
    public TwOutputUtil queryListMarket(CrmLeadsQuery query) {
        return TwOutputUtil.ok(service.queryListMarketLeads(query));
    }


    /**
     * 近海线索列表查询不分页
     *
     * @param query 查询参数
     * @return result
     */
    @GetMapping("/leads/list/offshore")
    public TwOutputUtil queryListOffshore(CrmLeadsOffshoreQuery query) {
        return TwOutputUtil.ok(service.queryListOffshoreLeads(query));
    }


    /**
     * 近海线索分页查询
     *
     * @param query 查询参数
     * @return
     */
    @GetMapping("/leads/paging/offshore")
    public TwOutputUtil pagingOffshore(CrmLeadsOffshoreQuery query) {
        return TwOutputUtil.ok(service.pagingOffshoreLeads(query));
    }


    /**
     * 近海线索列表的关联近海(与近海分页查询联合使用)
     *
     * @param query
     * @return
     */
    @GetMapping("/leads/paging/offshore/filter")
    public TwOutputUtil getOffshoreList(CrmLeadsOffshoreQuery query) {
        return TwOutputUtil.ok(service.getOffshoreList(query));
    }

    /**
     * 角色判断，当前用户是不是近海管理员(与近海分页查询联合使用)
     *
     * @return
     */
    @GetMapping("/leads/isOffshoreAdmin")
    public TwOutputUtil isOffshoreAdmin() {
        return TwOutputUtil.ok(service.isOffshoreAdmin());
    }

    /**
     * 线索列表查询不分页
     *
     * @param query 查询参数
     * @return result
     */
    @GetMapping("/leads/list")
    public TwOutputUtil queryList(CrmLeadsOffshoreQuery query) {
        return TwOutputUtil.ok(service.queryList(query));
    }

    /**
     * 线索分页查询
     *
     * @return result
     */
    @GetMapping("/leads/paging")
    public TwOutputUtil paging(CrmLeadsQuery query) {
        return TwOutputUtil.ok(service.paging(query));
    }

    /**
     * 登陆人报备的线索查询
     *
     * @return result
     */
    @GetMapping("/leads/paging/personal")
    public TwOutputUtil pagingPersonal(CrmLeadsQuery query) {
        return TwOutputUtil.ok(service.pagingPersonalLeads(query));
    }

    /**
     * 线索成员列表
     *
     * @param leadsId 主键id
     * @return result
     */
    @GetMapping("/leads/members/list/{leadsId}")
    public TwOutputUtil queryMembersList(@PathVariable Long leadsId) {
        return TwOutputUtil.ok(service.queryMembersList(leadsId));
    }

    /**
     * 线索日志列表
     *
     * @param leadsId 主键id
     * @return result
     */
    @GetMapping("/leads/log/list/{leadsId}")
    public TwOutputUtil queryLogList(@PathVariable Long leadsId, String stage) {
        return TwOutputUtil.ok(service.queryLogList(leadsId, stage));
    }

    /**
     * 线索日志列表
     *
     * @param query
     * @return result
     */
    @GetMapping("/leads/log/paging")
    public TwOutputUtil queryLogList(PrdSystemLogQuery query) {
        return TwOutputUtil.ok(service.pageLog(query));
    }

    /**
     * 线索跟进列表
     *
     * @param leadsId 主键id
     * @return result
     */
    @GetMapping("/leads/follow/list/{leadsId}")
    public TwOutputUtil queryFollowList(@PathVariable Long leadsId) {
        return TwOutputUtil.ok(followService.queryFollowList(leadsId, CrmFollowObjectEnum.Leads.name()));
    }


//    /**
//     * 可领取线索列表
//     *
//     * @return result
//     */
//    @GetMapping("/leads/paging/recieve")
//    public TwOutputUtil pagingReceive(CrmLeadsQuery query) {
//        return TwOutputUtil.ok(service.pagingReceive(query));
//    }


    /**
     * 详情查询
     *
     * @param id 主键
     * @return result
     */
    @GetMapping("/leads/detail")
    public TwOutputUtil queryDetail(@RequestParam Long offshoreId, @RequestParam Long id) {
        CrmLeadsVO vo = service.queryDetail(offshoreId, id);
        return TwOutputUtil.ok(vo);
    }


    /**
     * 逻辑删除
     *
     * @param keys 主键
     * @return result
     */
    @DeleteMapping("/leads/delete")
    public TwOutputUtil delete(Long[] keys) {
        service.deleteSoft(Arrays.asList(keys));
        return TwOutputUtil.ok();
    }


    /**
     * 关闭线索
     *
     * @param payload
     * @return result
     */
    @PostMapping("/leads/close")
    public TwOutputUtil close(@RequestBody CrmLeadsPayload payload) {
        service.closeLeads(payload);
        return TwOutputUtil.ok();
    }


//    /**
//     * 模板下载
//     *
//     * @return result
//     */
//    @ApiOperation("批量导出近海线索excel")
//    @PostMapping("/leads/downloadModel")
//    public void batchExport(HttpServletResponse response) {
//        service.downloadModel(response);
//    }


    /**
     * 批量导出excel
     *
     * @return result
     */
    @ApiOperation("批量导出近海线索excel")
    @GetMapping("/leads/batchExport/offshore")
    public void batchExport(HttpServletResponse response, CrmLeadsOffshoreQuery query) {
        service.downloadBatchOffshore(response, query);
    }

    /**
     * 批量导出excel
     *
     * @return result
     */
    @ApiOperation("批量导出线索")
    @GetMapping("/leads/batchExport")
    public void batchExport(HttpServletResponse response, CrmLeadsQuery query) {
        service.downloadBatch(response, query);
    }

    /**
     * 批量导入excel
     *
     * @return result
     */
    @ApiOperation("批量导入excel")
    @PostMapping(value = "/leads/batchImport", consumes = MediaType.MULTIPART_FORM_DATA_VALUE)
    @Transactional
    public TwOutputUtil batchImport(HttpServletResponse response, MultipartFile file, Boolean force) {
        Map<String, Object> resultMap = service.importBatch(response, file, force);
        if (resultMap.get("ok").equals("ok")) {
            return TwOutputUtil.ok(resultMap);
        } else if (resultMap.get("ok").equals("warn")) {
            return TwOutputUtil.warn("", "", resultMap);
        } else {
            return TwOutputUtil.error("", "", resultMap);
        }
    }

    /**
     * 测试数据
     *
     * @param nnum 主键
     * @return result
     */
    @GetMapping("/leads/test")
    public TwOutputUtil test(Integer nnum) {
        service.leadsNotReceiveRemind(nnum);
        return TwOutputUtil.ok();
    }

}
