package com.elitesland.tw.tw5.server.prd.crm.service;

import com.elitesland.tw.tw5.api.prd.crm.payload.CrmCustomerOperationMemberListPayload;
import com.elitesland.tw.tw5.api.prd.crm.payload.CrmCustomerOperationMemberPayload;
import com.elitesland.tw.tw5.api.prd.crm.query.CrmCustomerOperationMemberQuery;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmCustomerOperationMemberService;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmCustomerOperationMemberVO;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemLogService;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.util.PageUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.PrdSystemLogEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.PrdSystemObjectEnum;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.crm.constant.CustomerOperationMemberTypeEnum;
import com.elitesland.tw.tw5.server.prd.crm.convert.CrmCustomerOperationMemberConvert;
import com.elitesland.tw.tw5.server.prd.crm.entity.CrmCustomerOperationMemberDO;
import com.elitesland.tw.tw5.server.prd.crm.repo.CrmCustomerOperationMemberRepo;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * 客户经营 - 经营团队成员
 *
 * @author duwh
 * @date 2022/11/18
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class CrmCustomerOperationMemberServiceImpl implements CrmCustomerOperationMemberService {

    private final CrmCustomerOperationMemberRepo repo;
    private final PrdSystemLogService logService;
    private final CacheUtil cacheUtil;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public CrmCustomerOperationMemberVO insert(CrmCustomerOperationMemberPayload payload) {
        // 校验
        check(payload);
        // 默认值
        initDefaultValue(payload);

        CrmCustomerOperationMemberDO entityDo = CrmCustomerOperationMemberConvert.INSTANCE.toDo(payload);
        repo.save(entityDo);
        return CrmCustomerOperationMemberConvert.INSTANCE.toVo(entityDo);
    }

    @Override
    public List<CrmCustomerOperationMemberVO> saveAll(CrmCustomerOperationMemberListPayload payload) {
        if (null == payload.getOperId()) {
            throw TwException.error("", "operId不能为空");
        }
        List<CrmCustomerOperationMemberVO> list = new ArrayList<>();
        // 先清空，在新增；
        deleteSoftByOperId(payload.getOperId());
        if (!CollectionUtils.isEmpty(payload.getMemberList())) {
            payload.getMemberList().forEach(crmCustomerOperationMemberPayload -> {
                list.add(insert(crmCustomerOperationMemberPayload));
            });
            logService.saveNewLog(payload.getOperId(),
                PrdSystemObjectEnum.CUSTOMER_OPERATION.getCode(),
                PrdSystemLogEnum.CREATE.getDesc() + PrdSystemObjectEnum.CUSTOMER_OPERATION_MEMBER.getDesc());
        }
        return list;
    }

    @Override
    public void deleteSoftByOperId(Long operId) {
        CrmCustomerOperationMemberQuery query = new CrmCustomerOperationMemberQuery();
        query.setOperId(operId);
        final List<CrmCustomerOperationMemberDO> planTempDetailDOList = repo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder));
        final List<Long> keys = planTempDetailDOList.stream().map(CrmCustomerOperationMemberDO::getId).collect(Collectors.toList());
        deleteSoftNoLog(keys);
    }

    /**
     * 初始化默认值
     *
     * @param payload 有效载荷
     */
    private void initDefaultValue(CrmCustomerOperationMemberPayload payload) {
        if (!StringUtils.hasText(payload.getUserName())) {
            payload.setUserName(cacheUtil.getUserName(payload.getUserId()));
        }
        if (null == payload.getStartTime()) {
            payload.setStartTime(LocalDateTime.now());
        }
        if (!StringUtils.hasText(payload.getType())) {
            payload.setType(CustomerOperationMemberTypeEnum.TEMP.getCode());
            payload.setTypeName(CustomerOperationMemberTypeEnum.TEMP.getDesc());
        }

    }

    /**
     * 检查
     *
     * @param payload 有效载荷
     */
    private void check(CrmCustomerOperationMemberPayload payload) {
        if (null == payload.getOperId()) {
            throw TwException.error("", "operId不能为空");
        }
        if (null == payload.getUserId()) {
            throw TwException.error("", "userId不能为空");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public CrmCustomerOperationMemberVO update(CrmCustomerOperationMemberPayload payload) {
        if (null != payload.getUserId()) {
            if (!StringUtils.hasText(payload.getUserName())) {
                payload.setUserName(cacheUtil.getUserName(payload.getUserId()));
            }
        }
        CrmCustomerOperationMemberDO entity = repo.findById(payload.getId()).orElseGet(CrmCustomerOperationMemberDO::new);
        Assert.notNull(entity.getId(), "不存在");
        CrmCustomerOperationMemberDO entityDo = CrmCustomerOperationMemberConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return CrmCustomerOperationMemberConvert.INSTANCE.toVo(repo.save(entity));
    }

    @Override
    public CrmCustomerOperationMemberVO queryByKey(Long key) {
        CrmCustomerOperationMemberDO entity = repo.findById(key).orElseGet(CrmCustomerOperationMemberDO::new);
        Assert.notNull(entity.getId(), "不存在");
        final CrmCustomerOperationMemberVO crmCustomerOperationVO = CrmCustomerOperationMemberConvert.INSTANCE.toVo(entity);
        return crmCustomerOperationVO;
    }

    @Override
    public List<CrmCustomerOperationMemberVO> queryList(CrmCustomerOperationMemberQuery query) {
        //默认按照时间倒叙排序
        OrderItem orderItem = OrderItem.desc("createTime");
        query.defaultOrder(orderItem);
        return CrmCustomerOperationMemberConvert.INSTANCE.toVoList(repo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder)));
    }

    @Override
    public PagingVO<CrmCustomerOperationMemberVO> paging(CrmCustomerOperationMemberQuery query) {
        //默认按照时间倒叙排序
        OrderItem orderItem = OrderItem.desc("createTime");
        query.defaultOrder(orderItem);
        Page<CrmCustomerOperationMemberDO> page = repo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder), query.getPageRequest());

        //TODO 查询超期的授权，修改禁用状态

        return PageUtil.toPageVo(page.map(CrmCustomerOperationMemberConvert.INSTANCE::toVo));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            Optional<CrmCustomerOperationMemberDO> optional = repo.findById(keys.get(0));
            if (!optional.isEmpty()) {
                CrmCustomerOperationMemberDO entity = optional.get();
                final Long operId = entity.getOperId();
                logService.saveNewLog(operId,
                    PrdSystemObjectEnum.CUSTOMER_OPERATION.getCode(),
                    PrdSystemLogEnum.DELETE.getDesc() + PrdSystemObjectEnum.CUSTOMER_OPERATION_MEMBER.getDesc());
            }
            deleteSoftNoLog(keys);
        }
    }

    public void deleteSoftNoLog(List<Long> keys) {
        if (!keys.isEmpty()) {
            keys.stream().forEach(id -> {
                Optional<CrmCustomerOperationMemberDO> optional = repo.findById(id);
                if (!optional.isEmpty()) {
                    CrmCustomerOperationMemberDO entity = optional.get();
                    entity.setDeleteFlag(1);
                    repo.save(entity);
                }
            });
        }
    }


}
