package com.elitesland.tw.tw5.server.prd.crm.service;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.io.IoUtil;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.poi.excel.ExcelUtil;
import cn.hutool.poi.excel.ExcelWriter;
import com.alibaba.excel.EasyExcel;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitesland.tw.tw5.api.prd.ab.service.PrdAbAddressService;
import com.elitesland.tw.tw5.api.prd.ab.vo.PrdAbAddressVO;
import com.elitesland.tw.tw5.api.prd.crm.payload.*;
import com.elitesland.tw.tw5.api.prd.crm.service.*;
import com.elitesland.tw.tw5.api.prd.crm.vo.*;
import com.elitesland.tw.tw5.api.prd.crm.query.CrmCustomerOperationBusinessQuery;
import com.elitesland.tw.tw5.api.prd.crm.query.CrmCustomerOperationQuery;
import com.elitesland.tw.tw5.api.prd.crm.query.CrmOperationPlanTempDetailQuery;
import com.elitesland.tw.tw5.api.prd.crm.query.CrmOperationPlanTempQuery;
import com.elitesland.tw.tw5.api.prd.my.service.PrdUserService;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgEmployeeRefVO;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgOrganizationRefVO;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemLogService;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemLogVO;
import com.elitesland.tw.tw5.server.common.ExcelEntityDataListener;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.service.BusinessFollowServiceImpl;
import com.elitesland.tw.tw5.server.common.util.ChangeFieldLogUtil;
import com.elitesland.tw.tw5.server.common.util.PageUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.FunctionSelectionEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.PrdSystemObjectEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.SystemDefaultEnum;
import com.elitesland.tw.tw5.server.prd.org.dao.PrdOrgOrganizationDAO;
import com.elitesland.tw.tw5.server.prd.org.entity.PrdOrgOrganizationDO;
import com.elitesland.tw.tw5.server.prd.qixin.dto.*;
import com.elitesland.tw.tw5.server.prd.qixin.service.QiXinService;
import com.elitesland.tw.tw5.server.udc.UdcUtil;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.crm.constant.CustomerOperationTypeEnum;
import com.elitesland.tw.tw5.server.prd.crm.constant.SystemBasicStatusEnum;
import com.elitesland.tw.tw5.server.prd.crm.convert.CrmCustomerOperationConvert;
import com.elitesland.tw.tw5.server.prd.crm.dao.CrmCustomerOperationDAO;
import com.elitesland.tw.tw5.server.prd.crm.entity.CrmCustomerOperationDO;
import com.elitesland.tw.tw5.server.prd.crm.entity.CrmCustomerOperationMemberDO;
import com.elitesland.tw.tw5.server.prd.crm.entity.CrmOperationPlanDetailDO;
import com.elitesland.tw.tw5.server.prd.crm.repo.CrmCustomerOperationRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.persistence.criteria.Join;
import javax.persistence.criteria.JoinType;
import javax.persistence.criteria.Predicate;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 客户经营
 *
 * @author duwh
 * @date 2022/11/15
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class CrmCustomerOperationServiceImpl extends BaseServiceImpl implements CrmCustomerOperationService {
    private final CacheUtil cacheUtil;
    private final CrmCustomerOperationRepo repo;
    private final CrmCustomerOperationDAO dao;
    private final ChangeFieldLogUtil changeFieldLogUtil;
    private final PrdSystemLogService logService;
    private final CrmCustomerService customerService;
    private final PrdAbAddressService serviceAbAddress;

    private final CrmOperationPlanTempDetailService crmOperationPlanTempDetailService;
    private final CrmOperationPlanTempService crmOperationPlanTempService;
    private final CrmOperationPlanDetailService crmOperationPlanDetailService;
    private final CrmOperationPlanDetailMemberService crmOperationPlanDetailMemberService;
    private final PrdUserService prdUserService;
    private final CrmCustomerService crmCustomerService;
    private final UdcUtil udcUtil;
    // @DubboReference(version = "${provider.service.version}")
//    @Autowired
    //private SysNumberRuleService numberRuleService;
    private final PrdOrgOrganizationDAO prdOrgOrganizationDAO;
    private final CrmCustomerOperationEnterpriseInfoService enterpriseInfoService;
    private final CrmCustomerOperationSitesService sitesService;
    private final CrmCustomerOperationPartnersService partnersService;
    private final CrmCustomerOperationEmployeesService employeesService;
    private final CrmPeopleService peopleService;

    private final BusinessFollowServiceImpl globalUtilService;
    private final CrmCustomerOperationChanceService operationChanceService;
    private final QiXinService qiXinService;
    /**
     * 特殊 客户级别  非企业的时候 用于查询经营计划模板
     */
    private final String SPECIAL_CUST_GRADE = "99";
    /**
     * 客户经营编号
     */
    private final String CUST_OPER_NO = "CUST_OPER_NO";
    @Value("${tw5.sys_default.dimensionId}")
    private Long default_dimension_id;
    /**
     * 启信宝type_new字段翻译
     */
    private static final Map<String, String> qxbTypeNewMap = new HashMap<>() {{
        put("01", "大陆企业");
        put("02", "社会组织");
        put("03", "机关及事业单位");
        put("04", "港澳台及国外企业");
        put("05", "律所及其他组织机构");
    }};


    @Override
    @Transactional(rollbackFor = Exception.class)
    public CrmCustomerOperationVO insert(CrmCustomerOperationPayload payload) {
        // 数据校验
        check(payload);
        //"CUST_OPER_NO"
        String code = generateSeqNum(CUST_OPER_NO);
        payload.setCustNo(code);
        CrmCustomerOperationDO entityDo = CrmCustomerOperationConvert.INSTANCE.toDo(payload);
        entityDo.setCustOperStatus(SystemBasicStatusEnum.ACTIVE.getCode());
        repo.save(entityDo);    //如果customerId有值，就自然绑定

        // 初始化 经营计划
        initOperationPlan(entityDo);


        final Long operId = entityDo.getId();
        saveQxbInfo(payload, operId);

        // 获取变更日志
        final String fieldsCreateLog = changeFieldLogUtil.getFieldsCreateLog(entityDo);
        logService.saveNewLog(operId, PrdSystemObjectEnum.CUSTOMER_OPERATION.getCode(), fieldsCreateLog);
        return CrmCustomerOperationConvert.INSTANCE.toVo(entityDo);
    }

    private void saveQxbInfo(CrmCustomerOperationPayload payload, Long operId) {
        // 工商照面信息保存 - 启信宝
        saveEnterpriseInfo(payload, operId);
        // 企业年报网址 保存- 启信宝
        saveSites(payload, operId);
        // 工商股东（含历史） 保存- 启信宝
        savePartners(payload, operId);
        // 企业工商主要人员  保存- 启信宝
        saveEmployees(payload, operId);
        // 同步 企业工商主要人员 到经营人脉
        syncToPeople(payload, operId);
    }

    /**
     * 同步创建 经营人脉
     *
     * @param payload 有效载荷
     * @param operId  ③id
     */
    private void syncToPeople(CrmCustomerOperationPayload payload, Long operId) {
        final List<CrmCustomerOperationEmployeesPayload> employeesList = payload.getEmployeesList();
        if (!CollectionUtils.isEmpty(employeesList)) {
            employeesList.forEach(employeesPayload -> {
                CrmPeoplePayload peoplePayload = new CrmPeoplePayload();
                peoplePayload.setOperId(operId);
                peoplePayload.setCustomerId(payload.getCustomerId());
                peoplePayload.setPeopleName(employeesPayload.getName());
                peoplePayload.setJobs(employeesPayload.getTitle());
                peopleService.insertOrUpdate(peoplePayload);
            });
        }

    }

    /**
     * 1.45 主要人员 保存
     *
     * @param payload 有效载荷
     * @param operId  ③id
     */
    private void saveEmployees(CrmCustomerOperationPayload payload, Long operId) {
        final List<CrmCustomerOperationEmployeesPayload> employeesList = payload.getEmployeesList();
        if (!CollectionUtils.isEmpty(employeesList)) {
            employeesList.forEach(employeesPayload -> {
                employeesPayload.setOperId(operId);
                employeesService.insert(employeesPayload);
            });
        }
    }

    /**
     * 1.43 工商股东（含历史） 保存
     *
     * @param payload 有效载荷
     * @param operId  客户经营主键
     */
    private void savePartners(CrmCustomerOperationPayload payload, Long operId) {
        final List<CrmCustomerOperationPartnersPayload> partnersList = payload.getPartnersList();
        if (!CollectionUtils.isEmpty(partnersList)) {
            partnersList.forEach(partnersPayload -> {
                partnersPayload.setOperId(operId);
                partnersService.insert(partnersPayload);
            });
        }
    }

    /**
     * 1.53 企业年报网址 保存
     *
     * @param payload 有效载荷
     * @param operId  客户经营主键
     */
    private void saveSites(CrmCustomerOperationPayload payload, Long operId) {
        final List<CrmCustomerOperationSitesPayload> sitesList = payload.getSitesList();
        if (!CollectionUtils.isEmpty(sitesList)) {
            sitesList.forEach(site -> {
                site.setOperId(operId);
                sitesService.insert(site);
            });
        }
    }

    /**
     * 工商照面信息保存
     *
     * @param payload 有效载荷
     * @param operId  客户经营主键
     */
    private void saveEnterpriseInfo(CrmCustomerOperationPayload payload, Long operId) {
        final CrmCustomerOperationEnterpriseInfoPayload enterpriseInfo = payload.getEnterpriseInfo();
        if (null != enterpriseInfo) {
            enterpriseInfo.setOperId(operId);
            enterpriseInfoService.insert(enterpriseInfo);
        }
    }

    /**
     * 检查
     *
     * @param payload 有效载荷
     */
    private void check(CrmCustomerOperationPayload payload) {
        if (StringUtils.hasText(payload.getCustName())) {
            int count = repo.countByCustName(payload.getCustName());
            if (count > 0) {
                throw TwException.error("", "请勿重复创建客户！");
            }
        } else {
            throw TwException.error("", "名称必填！");
        }
        String custType = payload.getCustType();
        if (!StringUtils.hasText(custType)) {
            throw TwException.error("", "请选择类型");
        }
        // 类型为 企业、企业及生态伙伴  客户类别必填
        if (custType.equals(CustomerOperationTypeEnum.ENTERPRISE.getCode())
                || custType.equals(CustomerOperationTypeEnum.ENTERPRISE_ECO_PARTNER.getCode())) {
            if (!StringUtils.hasText(payload.getCustGrade())) {
                throw TwException.error("", "客户级别必填！");
            }
        }

    }

    /**
     * 检查 名称 唯一性
     *
     * @param custName 名称
     * @return int
     */
    @Override
    public int checkCustNameUnique(String custName) {
        return repo.countByCustName(custName);
    }

    /**
     * 根据经营计划模板 初始化经营计划
     *
     * @param entity 有效载荷
     */
    private void initOperationPlan(CrmCustomerOperationDO entity) {
        final String custType = entity.getCustType();
        String custGrade = entity.getCustGrade();
        final boolean isEnterprise = custType.equals(CustomerOperationTypeEnum.ENTERPRISE.getCode())
                || custType.equals(CustomerOperationTypeEnum.ENTERPRISE_ECO_PARTNER.getCode());
        if (!isEnterprise) {
            custGrade = SPECIAL_CUST_GRADE;
        }
        CrmOperationPlanTempQuery planTempQuery = new CrmOperationPlanTempQuery(custType, custGrade);
        final List<CrmOperationPlanTempVO> planTempVOList = crmOperationPlanTempService.queryList(planTempQuery);
        if (!CollectionUtils.isEmpty(planTempVOList)) {
            if (planTempVOList.size() > 1) {
                log.error("客户经营-经营计划模板有重复数据，类型：{},级别：{}", custType, custGrade);
            }
            final CrmOperationPlanTempVO planTempVO = planTempVOList.get(0);
            CrmOperationPlanTempDetailQuery planTempDetailQuery = new CrmOperationPlanTempDetailQuery(planTempVO.getId());
            final List<CrmOperationPlanTempDetailVO> planTempDetailVOList = crmOperationPlanTempDetailService.queryList(planTempDetailQuery);
            List<CrmOperationPlanDetailPayload> planDetailDOS = new ArrayList<>();
            planTempDetailVOList.forEach(crmOperationPlanTempDetailVO -> {
                CrmOperationPlanDetailPayload planDetailDO = new CrmOperationPlanDetailPayload();
                BeanUtils.copyProperties(crmOperationPlanTempDetailVO, planDetailDO);
                planDetailDO.setOperId(entity.getId());
                planDetailDO.setReadFlag(0);
                planDetailDOS.add(planDetailDO);
            });
            crmOperationPlanDetailService.saveAll(planDetailDOS);
        } else {
            log.warn("客户经营-经营计划模板未配置，类型：{},级别：{}", custType, custGrade);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public CrmCustomerOperationVO update(CrmCustomerOperationPayload payload) {
        CrmCustomerOperationDO entity = repo.findById(payload.getId()).orElseGet(CrmCustomerOperationDO::new);
        Assert.notNull(entity.getId(), "不存在");
        // 变更日志用
        CrmCustomerOperationDO entityLog = new CrmCustomerOperationDO();
        BeanUtils.copyProperties(entity, entityLog);

        CrmCustomerOperationDO entityDo = CrmCustomerOperationConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        // 修改NULL值特殊处理
        final StringBuilder fieldsUpdateLog = changeFieldLogUtil.nullFieldsProcess(payload, entityLog, entity);
        final CrmCustomerOperationDO save = repo.save(entity);

        //获取变更日志
        fieldsUpdateLog.append(changeFieldLogUtil.getFieldsUpdateLog(entityDo, entityLog));
        if (StringUtils.hasText(fieldsUpdateLog)) {
            logService.saveNewLog(entityDo.getId(), PrdSystemObjectEnum.CUSTOMER_OPERATION.getCode(), fieldsUpdateLog.toString());
        }
        return CrmCustomerOperationConvert.INSTANCE.toVo(save);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean updateCustOperStatus(Long key, String custOperStatus) {
        dao.updateCustOperStatus(key, custOperStatus);
        logService.saveNewLog(key, PrdSystemObjectEnum.CUSTOMER_OPERATION.getCode(), cacheUtil.transferSystemSelection(FunctionSelectionEnum.SystemStatus.getCode(), custOperStatus) + PrdSystemObjectEnum.CUSTOMER_OPERATION.getDesc());
        return true;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void relationCustomer(CrmCustomerOperationPayload payload) {
        CrmCustomerOperationDO entity = repo.findById(payload.getId()).orElseGet(CrmCustomerOperationDO::new);
        Assert.notNull(entity.getId(), "不存在");
        if (entity.getCustomerId() != null) {
            throw TwException.error("", "已关联客户不可重复操作，请核验！");
        }
        CrmCustomerOperationVO crmCustomerOperationVO = dao.queryCrmCustomerOperationVOByCustomerId(payload.getCustomerId());
        if (crmCustomerOperationVO != null) {
            throw TwException.error("", "该客户已被【" + crmCustomerOperationVO.getCustName() + "】关联，请核验！");
        }
        //客户经营关联客户
        dao.relationCustomer(payload.getId(), payload.getCustomerId());
        //客户经营下的人脉关联客户
        peopleService.relationCustomer(payload.getId(), payload.getCustomerId());
    }

    @Override
    public CrmCustomerOperationVO queryByKey(Long key) {
        CrmCustomerOperationDO entity = repo.findById(key).orElseGet(CrmCustomerOperationDO::new);
        Assert.notNull(entity.getId(), "不存在");
        final CrmCustomerOperationVO crmCustomerOperationVO = CrmCustomerOperationConvert.INSTANCE.toVo(entity);
        return crmCustomerOperationVO;
    }

    @Override
    public List<PrdSystemLogVO> queryLogList(Long id) {
        return logService.queryLogList(id, PrdSystemObjectEnum.CUSTOMER_OPERATION.getCode());
    }

    @Override
    public List<CrmCustomerOperationVO> queryList(CrmCustomerOperationQuery query) {
        final List<CrmCustomerOperationDO> list = repo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder));
        return CrmCustomerOperationConvert.INSTANCE.toVoList(list);
    }

    @Override
    public List<CrmCustomerOperationVO> queryListDataFilter(CrmCustomerOperationQuery query) {
        //默认按照时间倒叙排序
        OrderItem orderItem = OrderItem.desc("createTime");
        query.defaultOrder(orderItem);
        final Specification<CrmCustomerOperationDO> specification = where(query);
        return CrmCustomerOperationConvert.INSTANCE.toVoList(repo.findAll(specification));
    }

    @Override
    //@ReSubmitCheck(argExpressions = {"[0].custType","[0].current","[0].size"},conditionExpressions = "[0].custName != null", interval = 1, message = "请勿重复提交")
    public PagingVO<CrmCustomerOperationVO> pagingDataFilter(CrmCustomerOperationQuery query) {
        final Specification<CrmCustomerOperationDO> specification = where(query);
        Page<CrmCustomerOperationDO> page = repo.findAll(specification, query.getPageRequest());
        return PageUtil.toPageVo(page.map(CrmCustomerOperationConvert.INSTANCE::toVo));
    }

    @Override
    //@ReSubmitCheck(argExpressions = {"[0].custName","[0].current","[0].size"}, interval = 1, message = "请勿重复提交")
    public PagingVO<CrmCustomerOperationVO> paging(CrmCustomerOperationQuery query) {
        Page<CrmCustomerOperationDO> page = repo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder), query.getPageRequest());
        return PageUtil.toPageVo(page.map(CrmCustomerOperationConvert.INSTANCE::toVo));
    }

    /**
     * 条件封装
     *
     * @param query 查询
     * @return {@link Specification}<{@link CrmCustomerOperationDO}>
     */
    private Specification<CrmCustomerOperationDO> where(CrmCustomerOperationQuery query) {
        final Specification<CrmCustomerOperationDO> specification = (root, criteriaQuery, criteriaBuilder) -> {
            final Predicate predicate = QueryHelp.getPredicate(root, query, criteriaBuilder);
            List<Predicate> predicates = new ArrayList<>();
            predicates.add(predicate);
            // 使用左连接查询 关联经营团队成员表
            Join<CrmCustomerOperationMemberDO, CrmCustomerOperationDO> memberJoin = root.join("memberList", JoinType.LEFT);
            Join<CrmOperationPlanDetailDO, CrmCustomerOperationDO> planDetailJoin = root.join("planDetailList", JoinType.LEFT);
            final Long userId = query.getUserId();
            if (!ObjectUtils.isEmpty(userId)) {
                // 查出来userId的所有下级userId
                final List<PrdOrgEmployeeRefVO> empRef = prdUserService.queryLowListByKey(null);
                final Set<Long> empRefUserIdList = empRef.stream().map(prdOrgEmployeeRefVO -> prdOrgEmployeeRefVO.getUserId()).collect(Collectors.toSet());
                // 组织的组织负责人是当前登录人的 组织集合结果
                final List<PrdOrgOrganizationDO> organizationDOList = prdOrgOrganizationDAO.queryByManagerId(userId);

                // 经营团队成员
                Predicate predMember = criteriaBuilder.equal(memberJoin.get("userId"), userId);
                // 经营计划 执行者
                Predicate predPlanDetail = criteriaBuilder.equal(planDetailJoin.get("performerId"), userId);
                //predicates.add(predMember);

                Predicate pred2 = criteriaBuilder.equal(root.get("createUserId"), userId);
                // 销售经营部负责人
                Predicate pred3 = criteriaBuilder.equal(root.get("saleOperManagerId"), userId);
                // 客户经营部负责人
                Predicate pred4 = criteriaBuilder.equal(root.get("custOperManagerId"), userId);
                // 渠道经营负责人
                Predicate pred5 = criteriaBuilder.equal(root.get("channelUserId"), userId);
                // 产品负责人
                Predicate pred6 = criteriaBuilder.equal(root.get("productUserId"), userId);
                // 服务负责人（四大角色之一）
                Predicate pred7 = criteriaBuilder.equal(root.get("serviceUserId"), userId);
                // 商务负责人 （四大角色之一）
                Predicate pred8 = criteriaBuilder.equal(root.get("businessUserId"), userId);
                // 关怀负责人（四大角色之一）
                Predicate pred9 = criteriaBuilder.equal(root.get("careUserId"), userId);
                // 运维售后负责人 （四大角色之一）
                Predicate pred10 = criteriaBuilder.equal(root.get("operationUserId"), userId);

                List<Predicate> dataFilterQuery = new ArrayList<>();
                // 数据权限固定条件
                Predicate[] predicatesList = {predMember, predPlanDetail, pred2, pred3, pred4, pred5, pred6, pred7, pred8, pred9, pred10};
                dataFilterQuery.addAll(Arrays.asList(predicatesList));
                // 数据权限 上下级权限
                if (!CollectionUtils.isEmpty(empRefUserIdList)) {
                    // 存在下级的时候
                    // 服务负责人（四大角色之一）
                    final Predicate pred11 = root.get("serviceUserId").in(empRefUserIdList);
                    dataFilterQuery.add(pred11);
                    // 商务负责人 （四大角色之一）
                    final Predicate pred12 = root.get("businessUserId").in(empRefUserIdList);
                    dataFilterQuery.add(pred12);
                    // 关怀负责人（四大角色之一）
                    final Predicate pred13 = root.get("careUserId").in(empRefUserIdList);
                    dataFilterQuery.add(pred13);
                    // 运维售后负责人 （四大角色之一）
                    final Predicate pred14 = root.get("operationUserId").in(empRefUserIdList);
                    dataFilterQuery.add(pred11);
                    dataFilterQuery.add(pred14);
                }
                // 数据权限 组织负责人视角
                if (!CollectionUtils.isEmpty(organizationDOList)) {
                    Set<Long> orgIdList = organizationDOList.stream().map(PrdOrgOrganizationDO::getId).collect(Collectors.toSet());
                    // 客户经营部门
                    final Predicate pred15 = root.get("custOperBu").in(orgIdList);
                    // 销售经营部门
                    final Predicate pred16 = root.get("saleOperBu").in(orgIdList);
                    dataFilterQuery.add(pred15);
                    dataFilterQuery.add(pred16);
                }

                Predicate[] predicatesListResult = dataFilterQuery.toArray(new Predicate[dataFilterQuery.size()]);
                //  组合条件  or查询
                predicates.add(criteriaBuilder.or(predicatesListResult));
            }
            criteriaQuery.distinct(true);
            return criteriaBuilder.and(predicates.toArray(new Predicate[predicates.size()]));
            //return predicate;
        };
        return specification;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> operIdList) {
        if (!operIdList.isEmpty()) {
            operIdList.stream().forEach(id -> {
                Optional<CrmCustomerOperationDO> optional = repo.findById(id);
                if (!optional.isEmpty()) {
                    CrmCustomerOperationDO entity = optional.get();
                    entity.setDeleteFlag(1);
                    repo.save(entity);
                }
            });

            // 删除经营计划
            crmOperationPlanDetailService.deleteSoftByOperIds(operIdList);
            // 删除经营计划参与人
            crmOperationPlanDetailMemberService.deleteSoftByOperIds(operIdList);

            // 删除经营人脉
            peopleService.deleteSoftByOperIds(operIdList);
            //删除经营机会点
            operationChanceService.deleteSoftByOperIds(operIdList);
            //TODO 删除xxx
        }
    }

    @Override
    public List<CrmCustomerOperationSimpleVO> selectList(CrmCustomerOperationQuery query) {
        return CrmCustomerOperationConvert.INSTANCE.toListVoSimple(repo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder)));

    }

    @Override
    public List<CrmFollowVO> queryOperateFollowList(Long operId) {
        CrmCustomerOperationVO operationVO = dao.queryById(operId);
        //校验数据
        checkeRelation(operationVO);
        List<CrmFollowVO> crmFollowVOS = customerService.queryFollowList(operationVO.getCustomerId(), true);
        return crmFollowVOS;
    }

    @Override
    public CrmCustomerDataVO queryCustomerByOperId(Long operId) {
        CrmCustomerOperationVO operationVO = dao.queryById(operId);
        //校验数据
        checkeRelation(operationVO);
        CrmCustomerDataVO customerDataVO = dao.queryByCustomerId(operationVO.getCustomerId());
        List<PrdAbAddressVO> addressVOS = serviceAbAddress.queryList(customerDataVO.getBookId());
        customerDataVO.setPrdAbAddressVOs(addressVOS);

        return customerDataVO;
    }

    @Override
    public Object queryCustomerView(Long operId) {
        CrmCustomerOperationVO operationVO = dao.queryById(operId);
        //校验数据
        // checkeRelation(operationVO);
        if (operationVO == null) {
            throw TwException.error("", "数据不存在，请核验！");
        }
        Object contractCount = 0L;
        Object projectCount = 0;
        long opportunityCount = 0;
        if (operationVO.getCustomerId() != null) {
//            //获取线索数量
//            CrmLeadsQuery leadsQuery = new CrmLeadsQuery();
//            leadsQuery.setFormalCustomerId(operationVO.getCustomerId());
//            leadsQuery.setIsPermission(false);
//            PagingVO<CrmLeadsListVO> leadsPagingVO = leadService.paging(leadsQuery);
//            //获取线索数量
//            leadsCount = leadsPagingVO.getTotal();
            //获取合同数量
            contractCount = globalUtilService.queryContractNum(operationVO.getCustomerId());
            //获取项目数量
            projectCount = globalUtilService.queryProjectNum(operationVO.getCustomerId());
            //获取商机数量
            opportunityCount = dao.queryOpportunityCount(operationVO.getCustomerId());
        }
        //人脉数量
        long peopleCount = peopleService.queryList(operId).size();
        //long peopleCount = dao.queryOperationPeopleCount(operId);
        Map<String, Object> map = new HashMap<>();
        map.put("contractCount", contractCount);
        map.put("projectCount", projectCount);
        map.put("opportunityCount", opportunityCount);
        map.put("peopleCount", peopleCount);
        return map;
    }

    @Override
    public Map<String, Object> excelImport(MultipartFile file, Boolean force, HttpServletResponse response) throws IOException {
        if (force == null) {
            force = false;
        }
        ExcelEntityDataListener<CrmCustomerOperationExcel> dataListener = new ExcelEntityDataListener<>();
        try {
            EasyExcel.read(file.getInputStream(), CrmCustomerOperationExcel.class, dataListener).sheet(0).headRowNumber(4).doRead();
        } catch (Exception e) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "导入数据异常，请检查导入 excel的列值是否与模板对应");
        }
        List<CrmCustomerOperationExcel> excelDataList = dataListener.getDatas();
        if (CollectionUtils.isEmpty(excelDataList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "导入数据不能为空！");
        }
        // udc反向翻译
        excelDataList = udcUtil.translateListReverse(excelDataList);
        // 数据库
        final List<CrmCustomerOperationVO> dataDbList = queryList(new CrmCustomerOperationQuery());
        // 查询crm客户
        final List<CrmCustomerDataVO> crmCustomerDataVOS = crmCustomerService.queryList();
        // 组织数据
        List<PrdOrgOrganizationRefVO> organizationRefVOS = prdOrgOrganizationDAO.queryOrgList(default_dimension_id, SystemDefaultEnum.DefaultOrgStatus.getCode());
        StringBuilder error = new StringBuilder();
        StringBuilder warn = new StringBuilder();
        for (CrmCustomerOperationExcel payload : excelDataList) {
            StringBuilder importError = new StringBuilder();
            StringBuilder importWarn = new StringBuilder();
            final String custName = payload.getCustName();
            if (org.apache.commons.lang3.StringUtils.isEmpty(custName)) {
                warn.append("客户名称为空，跳过此数据；\n");
                continue;
                //importWarn.append("客户名称不能为空;");
            }
            // 校验名称重复性
            final long count = dataDbList.stream().filter(crmCustomerOperationVO -> crmCustomerOperationVO.getCustName().equals(custName)).count();
            if (count > 0) {
                importError.append("客户名称【" + custName + "】重复");
                error.append("客户名称【" + custName + "】重复;\n");
            }
            final Optional<Long> first = crmCustomerDataVOS.stream()
                    .filter(crmCustomerDataVO -> crmCustomerDataVO.getCustomerName().equals(custName))
                    .map(CrmCustomerDataVO::getId).findFirst();
            if (first.isPresent()) {
                final Long customerId = first.get();
                payload.setCustomerId(customerId);
            }
            // 提交组织  创建人 为提交组织的 组织负责人
            String excelCommitBuName = payload.getExcelCommitBuName();
            if (StringUtils.hasText(excelCommitBuName)) {
                final List<Long> managerIdList = organizationRefVOS.stream()
                        .filter(prdOrgOrganizationRefVO -> prdOrgOrganizationRefVO.getOrgName().equals(excelCommitBuName))
                        .map(PrdOrgOrganizationRefVO::getManageId)
                        .collect(Collectors.toList());
                if (!CollectionUtils.isEmpty(managerIdList)) {
                    final Long managerId = managerIdList.get(0);
                    payload.setCreateUserId(managerId);
                }
            }
            // 金额数值处理
            final String saleScaleStr = payload.getSaleScaleStr();
            if (StringUtils.hasText(saleScaleStr)) {
                final String trim = saleScaleStr.trim();
                if (NumberUtil.isNumber(trim)) {
                    payload.setSaleScale(new BigDecimal(trim));
                } else {
                    importWarn.append("\"客户名称【\" + custName + \"】:销售规模【" + saleScaleStr + "】不是数字类型");
                    warn.append("\"客户名称【\" + custName + \"】: 销售规模【" + saleScaleStr + "】不是数字类型;+\n");
                }
            }
            final String historyCooOutputStr = payload.getHistoryCooOutputStr();
            if (StringUtils.hasText(historyCooOutputStr)) {
                final String trim = historyCooOutputStr.trim();
                if (NumberUtil.isNumber(trim)) {
                    payload.setHistoryCooOutput(new BigDecimal(trim));
                } else {
                    importWarn.append("\"客户名称【\" + custName + \"】: 历史合作产出【" + saleScaleStr + "】不是数字类型");
                    warn.append("\"客户名称【\" + custName + \"】: 历史合作产出【" + saleScaleStr + "】不是数字类型;\n");
                }
            }

            payload.setImoprtError(importError.toString());
            payload.setImportWarn(importWarn.toString());
        }

        int warnNum = 0;
        int errorNum = 0;
        for (CrmCustomerOperationExcel operationExcel : excelDataList) {
            if (StringUtils.hasText(operationExcel.getImoprtError()) && !force) {
                errorNum++;
            }
            if ((StringUtils.hasText(operationExcel.getImportWarn())) && !force) {
                warnNum++;
            }
        }

        Map<String, Object> resultMap = new HashMap<>();
        if (errorNum > 0) {
            resultMap.put("ok", "error");
            resultMap.put("errorNum", errorNum);
            resultMap.put("error", error);
            resultMap.put("warnNum", warnNum);
            resultMap.put("warn", warn);
            //downloadErrorData(response, excelDataList);
            //resultMap.put("downloadUrl", downloadModelWithData(crmPotentialCustomerPayloadList));
        } /*else if (warnNum > 0) {
            resultMap.put("ok", "warn");
            resultMap.put("warnNum", warnNum);
            resultMap.put("warn", warn);
            //downloadErrorData(response, excelDataList);
            //resultMap.put("downloadUrl", downloadModelWithData(crmPotentialCustomerPayloadList));
        } */ else {
            for (CrmCustomerOperationExcel payload : excelDataList) {
                payload.setCustType(CustomerOperationTypeEnum.ENTERPRISE.getCode());
                // TODO
                payload.setDeleteFlag(0);
                payload.setRemark("excel导入");
                String code = generateSeqNum(CUST_OPER_NO);
                payload.setCustNo(code);
                CrmCustomerOperationDO entityDo = CrmCustomerOperationConvert.INSTANCE.excelToDo(payload);
                entityDo.setCustOperStatus(SystemBasicStatusEnum.ACTIVE.getCode());   //先写死，后面再改
                repo.save(entityDo);

                //TODO  初始化 经营计划
                initOperationPlan(entityDo);

                // TODO  获取变更日志
                final String fieldsCreateLog = changeFieldLogUtil.getFieldsCreateLog(entityDo);
                logService.saveNewLog(entityDo.getId(), PrdSystemObjectEnum.CUSTOMER_OPERATION.getCode(), fieldsCreateLog, payload.getCreateUserId());
            }
            resultMap.put("ok", "ok");
        }
        return resultMap;
    }

    private void downloadErrorData(HttpServletResponse response, List<CrmCustomerOperationExcel> excelDataList) throws IOException {
        List<CrmCustomerOperationExcel> rows = CollUtil.newArrayList(excelDataList);
        // 通过工具类创建writer，默认创建xls格式
        ExcelWriter writer = ExcelUtil.getWriter(true);
        // 一次性写出内容，使用默认样式，强制输出标题
        writer.write(rows, true);
        //out为OutputStream，需要写出到的目标流

        //response为HttpServletResponse对象
        response.setContentType("application/vnd.ms-excel;charset=utf-8");
        //test.xls是弹出下载对话框的文件名，不能为中文，中文请自行编码
        response.setHeader("Content-Disposition", "attachment;filename=error.xls");
        ServletOutputStream out = response.getOutputStream();
        writer.flush(out, true);
        // 关闭writer，释放内存
        writer.close();
        //此处记得关闭输出Servlet流
        IoUtil.close(out);
    }

    @Override
    public Object queryBusinessFollowPaging(CrmCustomerOperationBusinessQuery query) {

        Object resultData = new Object();
        CrmCustomerOperationVO operationVO = dao.queryById(query.getOperId());
        //校验数据
        checkeRelation(operationVO);
        query.setCustomerId(operationVO.getCustomerId());
        //合同业务往来
        if ("1".equals(query.getBusinessType())) {
            resultData = globalUtilService.queryContractFollow(query, operationVO.getCustomerId());
        }
        //商机业务往来
        if ("2".equals(query.getBusinessType())) {
            resultData = globalUtilService.queryOppoFollow(query, operationVO.getCustomerId());
        }
        //线索业务往来
        if ("3".equals(query.getBusinessType())) {
            resultData = globalUtilService.queryLeadsFollow(query, operationVO.getCustomerId());
        }
        return resultData;
    }

//    /**
//     * 获取线索数据
//     *
//     * @param query
//     * @return
//     */
//    @Override
//    public PagingVO<CrmLeadsListVO> queryLeadsFollow(TwQueryParam query, Long customerId) {
//        CrmLeadsQuery leadsQuery = new CrmLeadsQuery();
//        leadsQuery.setCurrent(query.getCurrent());
//        leadsQuery.setSize(query.getSize());
//        leadsQuery.setFormalCustomerId(customerId);
//        leadsQuery.setIsPermission(false);
//
//        return leadService.paging(leadsQuery);
//    }
//
//
//    /**
//     * 获取商机数据
//     *
//     * @param query
//     * @return
//     */
//    @Override
//    public PagingVO<CrmOpportunityVO> queryOppoFollow(TwQueryParam query, Long customerId) {
//        CrmOpportunityQuery oppoQuery = new CrmOpportunityQuery();
//        oppoQuery.setCurrent(query.getCurrent());
//        oppoQuery.setSize(query.getSize());
//        oppoQuery.setFormalCustomerId(customerId);
//        oppoQuery.setIsPermission(false);
//        return opportunityService.paging(oppoQuery);
//    }
//
//    /**
//     * 获取合同数量
//     *
//     * @param customerId
//     * @return
//     */
//    Object queryContractNum(Long customerId) {
//        String url = "api/openReport/v1/sync/contractCount";
//        PrdAbVO prdAbVO = daoCustomer.queryAbVOBykey(customerId);
//        if (ObjectUtils.isEmpty(prdAbVO)) {
//            throw TwException.error("", "关联客户数据不存在，请核验！");
//        }
//        Map<String, Object> map = new HashMap<>();
//        map.put("bookId", prdAbVO.getBookIdV4());
//        String result = httpUtil.sendSyncGet(url, map);
//        Map<String, Object> data = (Map) JSON.parse(result);
//        if ((data.get("ok") + "").equals("true")) {
//            if (!ObjectUtils.isEmpty(data.get("datum"))) {
//                return data.get("datum");
//            }
//        }
//        return 0;
//    }
//
//    /**
//     * 获取项目数量
//     *
//     * @param customerId
//     * @return
//     */
//    Object queryProjectNum(Long customerId) {
//        String url = "api/openReport/v1/sync/countProjectNumByCustId";
//        PrdAbVO prdAbVO = daoCustomer.queryAbVOBykey(customerId);
//        if (ObjectUtils.isEmpty(prdAbVO)) {
//            throw TwException.error("", "关联客户数据不存在，请核验！");
//        }
//        Map<String, Object> map = new HashMap<>();
//        map.put("bookId", prdAbVO.getBookIdV4());
//        String result = httpUtil.sendSyncGet(url, map);
//        Map<String, Object> data = (Map) JSON.parse(result);
//        if ((data.get("ok") + "").equals("true")) {
//            if (!ObjectUtils.isEmpty(data.get("datum"))) {
//                return data.get("datum");
//            }
//        }
//        return 0;
//    }
//
//    /**
//     * 获取4.0的合同数据
//     *
//     * @param query
//     * @return
//     */
//    @Override
//    public Map<String, Object> queryContractFollow(TwQueryParam query, Long customerId) {
//        Map<String, Object> resultMap = new HashMap<>();
//        resultMap.put("total", 0);
//        resultMap.put("records", new ArrayList<>());
//        String url = "api/openReport/v1/sync/contract";
//        PrdAbVO prdAbVO = daoCustomer.queryAbVOBykey(customerId);
//        if (ObjectUtils.isEmpty(prdAbVO)) {
//            throw TwException.error("", "关联客户数据不存在，请核验！");
//        }
//        Map<String, Object> map = new HashMap<>();
//        map.put("bookId", prdAbVO.getBookIdV4());
//        map.put("offset", query.getPageRequest().getOffset());
//        map.put("limit", query.getPageRequest().getPageSize());
//        String result = httpUtil.sendSyncGet(url, map);
//        Map<String, Object> data = (Map) JSON.parse(result);
//        if ((data.get("ok") + "").equals("true")) {
//            if (!ObjectUtils.isEmpty(data.get("datum"))) {
//                Map<String, Object> datas = (Map<String, Object>) data.get("datum");
//                resultMap.put("total", datas.get("total"));
//                resultMap.put("records", datas.get("rows"));
//            }
//        }
//        return resultMap;
//    }

    /**
     * 核验是否关联客户
     *
     * @param operationVO
     */
    void checkeRelation(CrmCustomerOperationVO operationVO) {
        if (ObjectUtils.isEmpty(operationVO)) {
            throw TwException.error("", "数据不存在，请核验！");
        }
        if (operationVO.getCustomerId() == null) {
            throw TwException.error("", "暂未关联客户，请核验！");
        }
    }

    @Override
    public List<CrmCustomerOperationVO> findOperationEnterpriseInfoAbsent() {
        return dao.findOperationEnterpriseInfoAbsent();
    }

    /**
     * 为一个客户经营补充启信宝信息
     *
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean fillQxbInfo(String custName, long operId) {
        //保存数据载体
        CrmCustomerOperationPayload operationPayload = new CrmCustomerOperationPayload();

        //基本信息
        String qxbBasicInfoStr = qiXinService.getBasicInfo(custName);
        log.info("根据关键字[{}]查询启信宝企业基本信息结果:{}", custName, qxbBasicInfoStr);
        QxbRootDTO qxbBasicInfoRoot = JSON.parseObject(qxbBasicInfoStr, QxbRootDTO.class);
        if (!QxbRootDTO.SUCCESS.equals(qxbBasicInfoRoot.getStatus())) {
            //基本信息接口失败，就认为所有接口都失败
            return false;
        }
        QxbBasicInfoDTO qxbBasicInfoDTO = qxbBasicInfoRoot.getData().toJavaObject(QxbBasicInfoDTO.class);
        CrmCustomerOperationEnterpriseInfoPayload operationEnterpriseInfoPayload = new CrmCustomerOperationEnterpriseInfoPayload();
        operationEnterpriseInfoPayload.setOrgNo(qxbBasicInfoDTO.getOrgNo());
        operationEnterpriseInfoPayload.setStartDate(qxbBasicInfoDTO.getStartDate());
        operationEnterpriseInfoPayload.setOperName(qxbBasicInfoDTO.getOperName());
        operationEnterpriseInfoPayload.setCreditNo(qxbBasicInfoDTO.getCreditNo());
        operationEnterpriseInfoPayload.setNewStatus(qxbBasicInfoDTO.getNew_status());
        operationEnterpriseInfoPayload.setActualCapi(qxbBasicInfoDTO.getActualCapi());
        operationEnterpriseInfoPayload.setTitle(qxbBasicInfoDTO.getTitle());
        operationEnterpriseInfoPayload.setCheckDate(qxbBasicInfoDTO.getCheckDate());
        if (StringUtils.hasText(qxbBasicInfoDTO.getType_new())) {
            String typeNew = qxbTypeNewMap.get(qxbBasicInfoDTO.getType_new());
            operationEnterpriseInfoPayload.setTypeNew(typeNew);
        }
        operationEnterpriseInfoPayload.setScope(qxbBasicInfoDTO.getScope());

        //联系方式
        String contactInfoStr = qiXinService.getContactInfo(custName);
        log.info("根据关键字[{}]查询启信宝企业联系方式结果:{}", custName, contactInfoStr);
        QxbRootDTO qxbContactInfoRoot = JSON.parseObject(contactInfoStr, QxbRootDTO.class);
        if (QxbRootDTO.SUCCESS.equals(qxbContactInfoRoot.getStatus())) {
            QxbContactInfoDTO qxbContactInfoDTO = qxbContactInfoRoot.getData().toJavaObject(QxbContactInfoDTO.class);
            CrmCustomerOperationDO operationDO = repo.findById(operId).get();
            boolean changed = false;
            if (StringUtils.hasText(qxbContactInfoDTO.getAddress())) {
                operationDO.setCustAddress(qxbContactInfoDTO.getAddress());
                changed = true;
            }
            if (StringUtils.hasText(qxbContactInfoDTO.getTelephone())) {
                operationDO.setLandline(qxbContactInfoDTO.getTelephone());
                changed = true;
            }
            if (StringUtils.hasText(qxbContactInfoDTO.getEmail())) {
                operationDO.setEmail(qxbContactInfoDTO.getEmail());
                changed = true;
            }
            if (changed) {
                repo.save(operationDO);
            }
        }

        //企业简介
        String entBriefStr = qiXinService.getEntBriefByName(custName);
        log.info("根据关键字[{}]查询启信宝企业简介结果:{}", custName, entBriefStr);
        QxbRootDTO qxbEntBriefRoot = JSON.parseObject(entBriefStr, QxbRootDTO.class);
        if (QxbRootDTO.SUCCESS.equals(qxbEntBriefRoot.getStatus())) {
            QxbEntBriefDTO qxbEntBriefDTO = qxbEntBriefRoot.getData().toJavaObject(QxbEntBriefDTO.class);
            operationEnterpriseInfoPayload.setEnterpriseDesc(qxbEntBriefDTO.getBrief());
        }
        operationPayload.setEnterpriseInfo(operationEnterpriseInfoPayload);
        //年报网址
        String qxbGetWebsitesStr = qiXinService.getGetWebsites(custName, null);
        log.info("根据关键字[{}]查询启信宝企业年报网址结果:{}", custName, qxbGetWebsitesStr);
        QxbRootDTO qxbGetWebsitesRoot = JSON.parseObject(qxbGetWebsitesStr, QxbRootDTO.class);
        if (QxbRootDTO.SUCCESS.equals(qxbGetWebsitesRoot.getStatus())) {
            JSONObject data = qxbGetWebsitesRoot.getData();
            JSONArray items = data.getJSONArray("items");
            List<QxbGetWebSiteDTO> qxbGetWebSiteDTOList = items.toJavaList(QxbGetWebSiteDTO.class);
            List<CrmCustomerOperationSitesPayload> sitesList = qxbGetWebSiteDTOList.stream().map(qxbGetWebSiteDTO -> {
                CrmCustomerOperationSitesPayload sitesPayload = new CrmCustomerOperationSitesPayload();
                sitesPayload.setDate(qxbGetWebSiteDTO.getDate());
                sitesPayload.setUrl(qxbGetWebSiteDTO.getUrl());
                return sitesPayload;
            }).collect(Collectors.toList());
            operationPayload.setSitesList(sitesList);
        }
        //股东
        String qxbPartnersStr = qiXinService.getPartners(custName, null);
        log.info("根据关键字[{}]查询启信宝股东信息结果:{}", custName, qxbPartnersStr);
        QxbRootDTO qxbPartnersDTO = JSON.parseObject(qxbPartnersStr, QxbRootDTO.class);
        if (QxbRootDTO.SUCCESS.equals(qxbPartnersDTO.getStatus())) {
            JSONObject data = qxbPartnersDTO.getData();
            JSONArray items = data.getJSONArray("items");
            List<QxbPartnersDTO> qxbPartnersDTOList = items.toJavaList(QxbPartnersDTO.class);
            List<CrmCustomerOperationPartnersPayload> partnersList = qxbPartnersDTOList.stream().map(qxbPartnersDTO1 -> {
                CrmCustomerOperationPartnersPayload partnersPayload = new CrmCustomerOperationPartnersPayload();
                partnersPayload.setName(qxbPartnersDTO1.getName());
                partnersPayload.setStockPercent(qxbPartnersDTO1.getStockPercent());
                partnersPayload.setTotalShouldCapi(qxbPartnersDTO1.getTotalShouldCapi());
                partnersPayload.setTotalRealCapi(qxbPartnersDTO1.getTotalRealCapi());
                return partnersPayload;
            }).collect(Collectors.toList());
            operationPayload.setPartnersList(partnersList);
        }

        //主要人员
        String qxbGetEmployeesStr = qiXinService.getGetEmployees(custName, null);
        log.info("根据关键字[{}]查询启信宝主要人员字段结果:{}", custName, qxbGetEmployeesStr);
        QxbRootDTO qxbGetEmployeesRoot = JSON.parseObject(qxbGetEmployeesStr, QxbRootDTO.class);
        if (QxbRootDTO.SUCCESS.equals(qxbGetEmployeesRoot.getStatus())) {
            JSONObject data = qxbGetEmployeesRoot.getData();
            JSONArray items = data.getJSONArray("items");
            List<QxbGetEmployeeDTO> qxbGetEmployeeDTOList = items.toJavaList(QxbGetEmployeeDTO.class);
            List<CrmCustomerOperationEmployeesPayload> employeesList = qxbGetEmployeeDTOList.stream().map(qxbGetEmployeeDTO -> {
                CrmCustomerOperationEmployeesPayload employeesPayload = new CrmCustomerOperationEmployeesPayload();
                employeesPayload.setName(qxbGetEmployeeDTO.getName());
                employeesPayload.setTitle(qxbGetEmployeeDTO.getTitle());
                employeesPayload.setIsHistory(qxbGetEmployeeDTO.getIs_history());
                return employeesPayload;
            }).collect(Collectors.toList());
            operationPayload.setEmployeesList(employeesList);
        }

        //保存
        saveQxbInfo(operationPayload, operId);
        return true;
    }
}
