package com.elitesland.tw.tw5.server.prd.crm.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitesland.tw.tw5.api.prd.crm.payload.CrmOperationPlanDetailMemberPayload;
import com.elitesland.tw.tw5.api.prd.crm.payload.CrmOperationPlanDetailPayload;
import com.elitesland.tw.tw5.api.prd.crm.query.CrmOperationPlanDetailMemberQuery;
import com.elitesland.tw.tw5.api.prd.crm.query.CrmOperationPlanDetailQuery;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmOperationPlanDetailMemberService;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmOperationPlanDetailService;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmOperationPlanDetailVO;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgEmployeeRefVO;
import com.elitesland.tw.tw5.api.prd.system.payload.PrdFsmFileRefPayload;
import com.elitesland.tw.tw5.api.prd.system.service.PrdFsmFileRefService;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemLogService;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemLogVO;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.QyWx.service.QyWxCommunicationService;
import com.elitesland.tw.tw5.server.common.util.ChangeFieldLogUtil;
import com.elitesland.tw.tw5.server.common.util.PageUtil;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.PrdSystemLogEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.PrdSystemObjectEnum;
import com.elitesland.tw.tw5.server.prd.crm.entity.*;
import com.elitesland.tw.tw5.server.prd.org.dao.PrdOrgOrganizationDAO;
import com.elitesland.tw.tw5.server.prd.org.entity.PrdOrgOrganizationDO;
import com.elitesland.tw.tw5.server.prd.system.constant.PrdCommentObjTypeEnum;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.crm.convert.CrmOperationPlanDetailConvert;
import com.elitesland.tw.tw5.server.prd.crm.repo.CrmCustomerOperationRepo;
import com.elitesland.tw.tw5.server.prd.crm.repo.CrmOperationPlanDetailRepo;
import com.elitesland.tw.tw5.server.prd.my.dao.PrdUserDAO;
import com.querydsl.core.types.dsl.BooleanExpression;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.core.task.TaskExecutor;
import org.springframework.data.domain.Page;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.persistence.criteria.Join;
import javax.persistence.criteria.JoinType;
import javax.persistence.criteria.Predicate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * 客户经营 - 经营计划明细
 *
 * @author duwh
 * @date 2022/11/17
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class CrmOperationPlanDetailServiceImpl implements CrmOperationPlanDetailService {

    private final CrmOperationPlanDetailRepo repo;
    private final PrdSystemLogService logService;
    private final ChangeFieldLogUtil changeFieldLogUtil;
    private final QyWxCommunicationService qyWxCommunicationService;
    private final CrmOperationPlanDetailMemberService planDetailMemberService;
    private final JPAQueryFactory jpaQueryFactory;
    private final PrdUserDAO prdUserDAO;
    private final PrdOrgOrganizationDAO prdOrgOrganizationDAO;
    private final CrmCustomerOperationRepo customerOperationRepo;
    private final PrdFsmFileRefService fsmFileRefService;
    private final TaskExecutor taskExecutor;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public CrmOperationPlanDetailVO insert(CrmOperationPlanDetailPayload payload) {
        if (null == payload.getReadFlag()) {
            payload.setReadFlag(0);
        }
        CrmOperationPlanDetailDO entityDo = CrmOperationPlanDetailConvert.INSTANCE.toDo(payload);
        long id = repo.save(entityDo).getId();
        // 发送企业通知 消息给执行者
        // sendQwxMessage(payload);

        //附件保存
        if (StringUtils.hasText(payload.getFileCodes())) {
            for (String fileCode : payload.getFileCodes().split(",")) {
                PrdFsmFileRefPayload fsmFileRefPayload = new PrdFsmFileRefPayload();
                fsmFileRefPayload.setFileCode(fileCode);
                fsmFileRefPayload.setObjId(id);
                fsmFileRefPayload.setObjType(PrdCommentObjTypeEnum.OPER_PLAN_MAIN.getCode());
                fsmFileRefService.insert(fsmFileRefPayload);
            }
        }

        // 参与者保存
        List<CrmOperationPlanDetailMemberPayload> memberList = payload.getMemberList();
        if (!CollectionUtils.isEmpty(memberList)) {
            memberList.stream().filter(member -> member.getUserId() != null).forEach(member -> {
                if (null == member.getOperId()) {
                    member.setOperId(entityDo.getOperId());
                    member.setPlanDetailId(entityDo.getId());
                }
                planDetailMemberService.insert(member);
            });
        }

        logService.saveNewLog(entityDo.getOperId(),
                PrdSystemObjectEnum.CUSTOMER_OPERATION.getCode(),
                PrdSystemLogEnum.CREATE.getDesc() + PrdSystemObjectEnum.CUSTOMER_OPERATION_PLAN_DETAIL.getDesc());
        logService.saveNewLog(entityDo.getId(),
                PrdSystemObjectEnum.CUSTOMER_OPERATION_PLAN_DETAIL.getCode(),
                PrdSystemLogEnum.CREATE.getDesc() + PrdSystemObjectEnum.CUSTOMER_OPERATION_PLAN_DETAIL.getDesc());
        return CrmOperationPlanDetailConvert.INSTANCE.toVo(entityDo);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public CrmOperationPlanDetailVO update(CrmOperationPlanDetailPayload payload) {
        CrmOperationPlanDetailDO entity = repo.findById(payload.getId()).orElseGet(CrmOperationPlanDetailDO::new);
        Assert.notNull(entity.getId(), "不存在");

        // 变更日志用
        CrmOperationPlanDetailDO entityLog = new CrmOperationPlanDetailDO();
        BeanUtils.copyProperties(entity, entityLog);

        CrmOperationPlanDetailDO entityDo = CrmOperationPlanDetailConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        // 修改NULL值特殊处理
        final StringBuilder fieldsUpdateLog = changeFieldLogUtil.nullFieldsProcess(payload, entityLog, entity);
        final CrmOperationPlanDetailDO save = repo.save(entity);
        // 发送企业通知 消息给执行者
        //sendQwxMessage(payload);

        //保存附件，全删全插
        fsmFileRefService.deleteSoftByObjIdAndObjType(payload.getId(), PrdCommentObjTypeEnum.OPER_PLAN_MAIN.getCode());
        if (StringUtils.hasText(payload.getFileCodes())) {
            for (String fileCode : payload.getFileCodes().split(",")) {
                PrdFsmFileRefPayload fsmFileRefPayload = new PrdFsmFileRefPayload();
                fsmFileRefPayload.setFileCode(fileCode);
                fsmFileRefPayload.setObjId(payload.getId());
                fsmFileRefPayload.setObjType(PrdCommentObjTypeEnum.OPER_PLAN_MAIN.getCode());
                fsmFileRefService.insert(fsmFileRefPayload);
            }
        }

        logService.saveNewLog(entityDo.getOperId(),
                PrdSystemObjectEnum.CUSTOMER_OPERATION.getCode(),
                PrdSystemLogEnum.UPDATE.getDesc() + PrdSystemObjectEnum.CUSTOMER_OPERATION_PLAN_DETAIL.getDesc());
        // 经营计划信息的修改：（张三）更新（状态/执行者/时间/内容/优先级/内容）为（字段新值）
        // 获取变更日志
        fieldsUpdateLog.append(changeFieldLogUtil.getFieldsUpdateLog(entityDo, entityLog));
        if (StringUtils.hasText(fieldsUpdateLog)) {
            logService.saveNewLog(entityDo.getId(), PrdSystemObjectEnum.CUSTOMER_OPERATION_PLAN_DETAIL.getCode(), fieldsUpdateLog.toString());
        }

        return CrmOperationPlanDetailConvert.INSTANCE.toVo(save);
    }

    /**
     * 发送企业微信消息
     *
     * @param payload 有效载荷
     */
    private void sendQwxMessage(CrmOperationPlanDetailPayload payload) {
        // 发送企业通知 消息给执行者
        if (null != payload.getPerformerId()) {
            final String loginUserName = GlobalUtil.getLoginUserName();
            String planName = payload.getPlanName();
            // a标签不行，企业微信会跳转
            //planName = "<a href='javascript:void(0);' οnclick='return false'>" + planName + "</a>";
            qyWxCommunicationService.sendMessageToUser(payload.getPerformerId(), loginUserName + "在“" + planName + "”将你设为执行者");
        }
    }

    @Override
    public CrmOperationPlanDetailVO queryByKey(Long key) {
        CrmOperationPlanDetailDO entity = repo.findById(key).orElseGet(CrmOperationPlanDetailDO::new);
        Assert.notNull(entity.getId(), "不存在");
        final CrmOperationPlanDetailVO crmCustomerOperationVO = CrmOperationPlanDetailConvert.INSTANCE.toVo(entity);
        //前端需要使用客户经营状态控制编辑
        if (entity.getOperId() != null) {
            Optional<CrmCustomerOperationDO> operOptional = customerOperationRepo.findById(entity.getOperId());
            if (operOptional.isPresent()) {
                crmCustomerOperationVO.setCustOperStatus(operOptional.get().getCustOperStatus());
            }
        }
        return crmCustomerOperationVO;
    }

    @Override
    public List<CrmOperationPlanDetailVO> queryList(CrmOperationPlanDetailQuery query) {
        return CrmOperationPlanDetailConvert.INSTANCE.toVoList(repo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder)));
    }

    @Override
    public List<CrmOperationPlanDetailVO> queryListDataFilter(CrmOperationPlanDetailQuery query) {
        final Specification<CrmOperationPlanDetailDO> specification = where(query);
        return CrmOperationPlanDetailConvert.INSTANCE.toVoList(repo.findAll(specification));
    }

    @Override
    public long count(CrmOperationPlanDetailQuery query) {
        return repo.count((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder));
    }

    @Override
    public PagingVO<CrmOperationPlanDetailVO> paging(CrmOperationPlanDetailQuery query) {
        Page<CrmOperationPlanDetailDO> page = repo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder), query.getPageRequest());
        return PageUtil.toPageVo(page.map(CrmOperationPlanDetailConvert.INSTANCE::toVo));
    }

    /**
     * 分页 数据权限
     * <p>
     * ● 管理员：客户经营管理员角色，可看全量；（菜单+数据）
     * ● 普通：客户经营普通角色； （菜单）
     * ● 有该客户经营的权限就有其下面的经营计划权限；
     * ● 经营计划的负责人、参与者有该经营计划权限；
     *
     * @param query 查询
     * @return {@link PagingVO}<{@link CrmOperationPlanDetailVO}>
     */
    @Override
    public PagingVO<CrmOperationPlanDetailVO> pagingDataFilter(CrmOperationPlanDetailQuery query) {
        final Specification<CrmOperationPlanDetailDO> specification = where(query);
        Page<CrmOperationPlanDetailDO> page = repo.findAll(specification, query.getPageRequest());
        Page<CrmOperationPlanDetailVO> mapedPage = page.map(CrmOperationPlanDetailConvert.INSTANCE::toVo);
        //补充客户经营和计划参与者数据
        if (!CollectionUtils.isEmpty(mapedPage.getContent())) {
            CompletableFuture<Void>[] futures = new CompletableFuture[mapedPage.getContent().size()];
            for (int i = 0; i < mapedPage.getContent().size(); i++) {
                final CrmOperationPlanDetailVO planDetailVO = mapedPage.getContent().get(i);
                futures[i] = CompletableFuture.runAsync(() -> {
                    Optional<CrmCustomerOperationDO> operationDOOptional = customerOperationRepo.findById(planDetailVO.getOperId());
                    if (operationDOOptional.isPresent()) {
                        var operationDO = operationDOOptional.get();
                        planDetailVO.setCustOperBu(operationDO.getCustOperBu());
                        planDetailVO.setCustOperManagerId(operationDO.getCustOperManagerId());
                        planDetailVO.setSaleOperBu(operationDO.getSaleOperBu());
                        planDetailVO.setSaleOperManagerId(operationDO.getSaleOperManagerId());
                    }
                    CrmOperationPlanDetailMemberQuery memberQuery = new CrmOperationPlanDetailMemberQuery();
                    memberQuery.setPlanDetailId(planDetailVO.getId());
                    memberQuery.setOperId(planDetailVO.getOperId());
                    planDetailVO.setMemberList(planDetailMemberService.queryList(memberQuery));
                }, taskExecutor);
            }
            try {
                CompletableFuture.allOf(futures).get();
            } catch (Exception e) {
                log.error(e.getMessage(), e);
                throw new BusinessException("查询客户经营和计划参与者数据时出现异常");
            }
        }
        return PageUtil.toPageVo(mapedPage);
    }

    private Specification<CrmOperationPlanDetailDO> where(CrmOperationPlanDetailQuery query) {
        Long userId = GlobalUtil.getLoginUserId();
        final Specification<CrmOperationPlanDetailDO> specification = (root, criteriaQuery, criteriaBuilder) -> {
            final Predicate predicate = QueryHelp.getPredicate(root, query, criteriaBuilder);
            List<Predicate> predicates = new ArrayList<>();
            predicates.add(predicate);

            List<Predicate> dataFilterQuery = new ArrayList<>();
            // 使用左连接查询 关联客户经营数据
            Join<CrmCustomerOperationDO, CrmOperationPlanDetailDO> operObj = root.join("operObj", JoinType.LEFT);
            // 使用左连接查询 关联经营计划参与者
            Join<CrmOperationPlanDetailMemberDO, CrmOperationPlanDetailDO> memberJoin = root.join("memberList", JoinType.LEFT);
            if (!query.isManagerFlag()) {

                // 经营计划参与者
                Predicate predMember = criteriaBuilder.equal(memberJoin.get("userId"), userId);

                // 经营计划 执行者
                Predicate performerId = criteriaBuilder.equal(root.get("performerId"), userId);

                // 创建人
                Predicate createUserId = criteriaBuilder.equal(root.get("createUserId"), userId);

                // 数据权限固定条件
                Predicate[] predicatesList = {predMember, performerId, createUserId};
                dataFilterQuery.addAll(Arrays.asList(predicatesList));

                // 使用左连接查询 关联经营团队成员表
                Join<CrmCustomerOperationMemberDO, CrmCustomerOperationDO> operMemberJoin = operObj.join("memberList", JoinType.LEFT);
                // 经营团队成员
                Predicate operPredMember = criteriaBuilder.equal(operMemberJoin.get("userId"), userId);
                // 查出来userId的所有下级userId
                final List<PrdOrgEmployeeRefVO> empRef = prdUserDAO.queryLowListByKey(null, GlobalUtil.getLoginUserId());
                final Set<Long> empRefUserIdList = empRef.stream().map(prdOrgEmployeeRefVO -> prdOrgEmployeeRefVO.getUserId()).collect(Collectors.toSet());
                // 组织的组织负责人是当前登录人的 组织集合结果
                final List<PrdOrgOrganizationDO> organizationDOList = prdOrgOrganizationDAO.queryByManagerId(userId);

                // 客户经营里的数据权限
                Predicate pred2 = criteriaBuilder.equal(operObj.get("createUserId"), userId);
                // 销售经营部负责人
                Predicate pred3 = criteriaBuilder.equal(operObj.get("saleOperManagerId"), userId);
                // 客户经营部负责人
                Predicate pred4 = criteriaBuilder.equal(operObj.get("custOperManagerId"), userId);
                // 渠道经营负责人
                Predicate pred5 = criteriaBuilder.equal(operObj.get("channelUserId"), userId);
                // 产品负责人
                Predicate pred6 = criteriaBuilder.equal(operObj.get("productUserId"), userId);
                // 服务负责人（四大角色之一）
                Predicate pred7 = criteriaBuilder.equal(operObj.get("serviceUserId"), userId);
                // 商务负责人 （四大角色之一）
                Predicate pred8 = criteriaBuilder.equal(operObj.get("businessUserId"), userId);
                // 关怀负责人（四大角色之一）
                Predicate pred9 = criteriaBuilder.equal(operObj.get("careUserId"), userId);
                // 运维售后负责人 （四大角色之一）
                Predicate pred10 = criteriaBuilder.equal(operObj.get("operationUserId"), userId);
                // 数据权限固定条件
                Predicate[] predicatesListOper = {operPredMember, pred2, pred3, pred4, pred5, pred6, pred7, pred8, pred9, pred10};
                dataFilterQuery.addAll(Arrays.asList(predicatesListOper));

                // 数据权限 上下级权限
                if (!CollectionUtils.isEmpty(empRefUserIdList)) {
                    // 存在下级的时候
                    // 服务负责人（四大角色之一）
                    final Predicate pred11 = operObj.get("serviceUserId").in(empRefUserIdList);
                    dataFilterQuery.add(pred11);
                    // 商务负责人 （四大角色之一）
                    final Predicate pred12 = operObj.get("businessUserId").in(empRefUserIdList);
                    dataFilterQuery.add(pred12);
                    // 关怀负责人（四大角色之一）
                    final Predicate pred13 = operObj.get("careUserId").in(empRefUserIdList);
                    dataFilterQuery.add(pred13);
                    // 运维售后负责人 （四大角色之一）
                    final Predicate pred14 = operObj.get("operationUserId").in(empRefUserIdList);
                    dataFilterQuery.add(pred11);
                    dataFilterQuery.add(pred14);
                }
                // 数据权限 组织负责人视角
                if (!CollectionUtils.isEmpty(organizationDOList)) {
                    Set<Long> orgIdList = organizationDOList.stream().map(PrdOrgOrganizationDO::getId).collect(Collectors.toSet());
                    // 客户经营部门
                    final Predicate pred15 = operObj.get("custOperBu").in(orgIdList);
                    // 销售经营部门
                    final Predicate pred16 = operObj.get("saleOperBu").in(orgIdList);
                    dataFilterQuery.add(pred15);
                    dataFilterQuery.add(pred16);
                }

            }

            if (!dataFilterQuery.isEmpty()) {
                Predicate[] predicatesListResult = dataFilterQuery.toArray(new Predicate[dataFilterQuery.size()]);
                //  组合条件  or查询
                predicates.add(criteriaBuilder.or(predicatesListResult));
            }
            if (StringUtils.hasText(query.getCustName())) {
                // 根据 客户经营 名称筛选
                Predicate custName = criteriaBuilder.like(operObj.get("custName"), SqlUtil.toSqlLikeString(query.getCustName()));
                predicates.add(custName);
            }
            if (query.getCustOperBu() != null) {
                predicates.add(criteriaBuilder.equal(operObj.get("custOperBu"), query.getCustOperBu()));
            }
            if (query.getCustOperManagerId() != null) {
                predicates.add(criteriaBuilder.equal(operObj.get("custOperManagerId"), query.getCustOperManagerId()));
            }
            if (query.getSaleOperBu() != null) {
                predicates.add(criteriaBuilder.equal(operObj.get("saleOperBu"), query.getSaleOperBu()));
            }
            if (query.getSaleOperManagerId() != null) {
                predicates.add(criteriaBuilder.equal(operObj.get("saleOperManagerId"), query.getSaleOperManagerId()));
            }
            if (StringUtils.hasText(query.getMemberUserIds())) {
                predicates.add(memberJoin.get("userId").in(query.getMemberUserIds().split(",")));
            }
            if (query.getStartDate() != null) {
                LocalDateTime startTimeFrom = LocalDateTime.of(query.getStartDate(), LocalTime.MIN);
                LocalDateTime startTimeTo = LocalDateTime.of(query.getStartDate().plusDays(1), LocalTime.MIN);
                predicates.add(criteriaBuilder.greaterThanOrEqualTo(root.get("startTime"), startTimeFrom));
                predicates.add(criteriaBuilder.lessThan(root.get("startTime"), startTimeTo));
            }
            if (query.getEndDate() != null) {
                LocalDateTime endTimeFrom = LocalDateTime.of(query.getEndDate(), LocalTime.MIN);
                LocalDateTime endTimeTo = LocalDateTime.of(query.getEndDate().plusDays(1), LocalTime.MIN);
                predicates.add(criteriaBuilder.greaterThanOrEqualTo(root.get("endTime"), endTimeFrom));
                predicates.add(criteriaBuilder.lessThan(root.get("endTime"), endTimeTo));
            }

            criteriaQuery.distinct(true);
            return criteriaBuilder.and(predicates.toArray(new Predicate[predicates.size()]));
        };
        return specification;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            AtomicReference<Long> operId = new AtomicReference<>();
            keys.stream().forEach(id -> {
                Optional<CrmOperationPlanDetailDO> optional = repo.findById(id);
                if (!optional.isEmpty()) {
                    CrmOperationPlanDetailDO entity = optional.get();
                    operId.set(entity.getOperId());
                    entity.setDeleteFlag(1);
                    repo.save(entity);
                }
            });
            // 删除参与者
            planDetailMemberService.deleteSoftByPlanDetailIds(keys);

            logService.saveNewLog(operId.get(),
                    PrdSystemObjectEnum.CUSTOMER_OPERATION.getCode(),
                    PrdSystemLogEnum.DELETE.getDesc() + PrdSystemObjectEnum.CUSTOMER_OPERATION_PLAN_DETAIL.getDesc());
        }
    }

    /**
     * 根据客户经营主键id集合 逻辑删除数据
     *
     * @param operIds 客户经营主键
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoftByOperIds(List<Long> operIds) {
        if (!operIds.isEmpty()) {
            QCrmOperationPlanDetailDO qdo = QCrmOperationPlanDetailDO.crmOperationPlanDetailDO;
            JPAUpdateClause update = jpaQueryFactory.update(qdo)
                    .set(qdo.deleteFlag, 1)
                    .where(qdo.operId.in(operIds));
            update.execute();
        }
    }

    @Override
    public void saveAll(List<CrmOperationPlanDetailPayload> planDetailPayloadS) {
        List<CrmOperationPlanDetailDO> list = CrmOperationPlanDetailConvert.INSTANCE.toDoList(planDetailPayloadS);
        repo.saveAll(list);
    }

    @Override
    public List<PrdSystemLogVO> queryLogList(Long key) {
        return logService.queryLogList(key, PrdSystemObjectEnum.CUSTOMER_OPERATION_PLAN_DETAIL.getCode());
    }

    /**
     * 工时选择经营计划列表
     *
     * @return {@link List}<{@link CrmOperationPlanDetailVO}>
     */
    @Override
    public List<CrmOperationPlanDetailVO> listForTimeSheet() {
        CrmOperationPlanDetailQuery query = new CrmOperationPlanDetailQuery();
        final Long userId = GlobalUtil.getLoginUserId();
        query.setPerformerId(userId);
        QCrmOperationPlanDetailDO qdo = QCrmOperationPlanDetailDO.crmOperationPlanDetailDO;
        final QCrmOperationPlanDetailMemberDO qMemberDo = QCrmOperationPlanDetailMemberDO.crmOperationPlanDetailMemberDO;

        BooleanExpression expression = qdo.deleteFlag.eq(0);
        expression = expression.and(qMemberDo.deleteFlag.eq(0));
        expression = expression.and(qdo.performerId.eq(userId).or(qMemberDo.userId.eq(userId)));
        final List<CrmOperationPlanDetailDO> list = jpaQueryFactory.select(qdo)
                .from(qdo)
                .leftJoin(qMemberDo)
                .on(qdo.id.eq(qMemberDo.planDetailId))
                .where(expression)
                .fetch();
        return CrmOperationPlanDetailConvert.INSTANCE.toVoList(list);
    }

}
