package com.elitesland.tw.tw5.server.prd.my.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.my.payload.TimesheetBiweeklyDetailPayload;
import com.elitesland.tw.tw5.api.prd.my.payload.TimesheetBiweeklyPayload;
import com.elitesland.tw.tw5.api.prd.my.query.TimesheetBiweeklyDetailQuery;
import com.elitesland.tw.tw5.api.prd.my.query.TimesheetBiweeklyQuery;
import com.elitesland.tw.tw5.api.prd.my.query.TimesheetPlanQuery;
import com.elitesland.tw.tw5.api.prd.my.query.TimesheetQuery;
import com.elitesland.tw.tw5.api.prd.my.service.*;
import com.elitesland.tw.tw5.api.prd.my.vo.TimesheetBiweeklyDetailVO;
import com.elitesland.tw.tw5.api.prd.my.vo.TimesheetBiweeklyVO;
import com.elitesland.tw.tw5.api.prd.my.vo.TimesheetPlanVO;
import com.elitesland.tw.tw5.api.prd.my.vo.TimesheetVO;
import com.elitesland.tw.tw5.api.prd.org.service.PrdOrgEmployeeService;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgDataRefVO;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.util.DateUtil;
import com.elitesland.tw.tw5.server.common.util.FileUtil;
import com.elitesland.tw.tw5.server.common.util.PageUtil;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.RoleEnum;
import com.elitesland.tw.tw5.server.prd.my.constant.TimesheetBiweeklyDetailTypeEnum;
import com.elitesland.tw.tw5.server.prd.my.constant.TimesheetStatus;
import com.elitesland.tw.tw5.server.prd.my.convert.TimesheetBiweeklyConvert;
import com.elitesland.tw.tw5.server.prd.system.dao.PrdSystemRoleDAO;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.my.entity.TimesheetBiweeklyDO;
import com.elitesland.tw.tw5.server.prd.my.repo.TimesheetBiweeklyRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.persistence.criteria.Predicate;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.time.LocalDate;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 双周计划
 *
 * @author duwh
 * @date 2022-12-21
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class TimesheetBiweeklyServiceImpl implements TimesheetBiweeklyService {

    private final TimesheetBiweeklyRepo timesheetBiweeklyRepo;
    private final PrdUserService prdUserService;
    private final TimesheetService timesheetService;
    private final TimesheetPlanService timesheetPlanService;
    private final TimesheetBiweeklyDetailService timesheetBiweeklyDetailService;
    private final PrdOrgEmployeeService prdOrgEmployeeService;
    private final PrdSystemRoleDAO systemRoleDAO;

    @Override
    public PagingVO<TimesheetBiweeklyVO> paging(TimesheetBiweeklyQuery query) {
        final Specification<TimesheetBiweeklyDO> spec = (root, criteriaQuery, criteriaBuilder) -> {
            final Predicate predicate = QueryHelp.getPredicate(root, query, criteriaBuilder);

            List<Predicate> predicates = new ArrayList<>();
            predicates.add(predicate);
            criteriaQuery.distinct(true);
            List<Predicate> dataFilterQuery = new ArrayList<>();
            // 判断当前登录人是否是系统管理员
            List<Long> userIdsByPlatRole = systemRoleDAO.queryUserIdByRoleCodes(Arrays.asList(RoleEnum.SYS.getCode()));
            if (CollectionUtils.isEmpty(userIdsByPlatRole) || !userIdsByPlatRole.contains(query.getUserId())) {
                if (null != query.getUserId() && null == query.getCreateUserId()) {
                    Predicate tsUserId = criteriaBuilder.equal(root.get("tsUserId"), query.getUserId());
                    Predicate receiveUserIds = criteriaBuilder.like(root.get("receiveUserIds"), SqlUtil.toSqlLikeString(query.getUserId() + ""));
                    dataFilterQuery.add(tsUserId);
                    dataFilterQuery.add(receiveUserIds);
                }
                if (!CollectionUtils.isEmpty(dataFilterQuery)) {
                    Predicate[] predicatesListResult = dataFilterQuery.toArray(new Predicate[dataFilterQuery.size()]);
                    //  组合条件  or查询
                    predicates.add(criteriaBuilder.or(predicatesListResult));
                }
            }
            return criteriaBuilder.and(predicates.toArray(new Predicate[predicates.size()]));
        };
        Page<TimesheetBiweeklyDO> page = timesheetBiweeklyRepo.findAll(spec, query.getPageRequest());
        return PageUtil.toPageVo(page.map(TimesheetBiweeklyConvert.INSTANCE::toVo));
    }

    @Override
    public List<TimesheetBiweeklyVO> queryList(TimesheetBiweeklyQuery query) {
        return TimesheetBiweeklyConvert.INSTANCE.toVoList(timesheetBiweeklyRepo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder)));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public TimesheetBiweeklyVO queryByKey(Long key) {
        TimesheetBiweeklyDO entity = timesheetBiweeklyRepo.findById(key).orElseGet(TimesheetBiweeklyDO::new);
        Assert.notNull(entity.getId(), "不存在");
        TimesheetBiweeklyVO vo = TimesheetBiweeklyConvert.INSTANCE.toVo(entity);

        TimesheetBiweeklyDetailQuery query = new TimesheetBiweeklyDetailQuery();
        query.setTsbId(vo.getId());
        final List<TimesheetBiweeklyDetailVO> timesheetBiweeklyDetailVOS = timesheetBiweeklyDetailService.queryList(query);
        // 明细数据
        vo.setDetailList(timesheetBiweeklyDetailVOS);

        //如果是审批人查看 更改已读
        if (entity.getReceiveUserIds().contains(GlobalUtil.getLoginUserId() + "")) {
            TimesheetBiweeklyPayload update = new TimesheetBiweeklyPayload();
            update.setId(key);
            update.setIsRead(1);
            update(update);
        }
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public TimesheetBiweeklyVO insert(TimesheetBiweeklyPayload payload) {
        // 生成周报日期 后台可以判断是哪个周, 不传值默认今天
        if (payload.getGenerateDate() == null) {
            payload.setGenerateDate(LocalDate.now());
        }
        final Long userId = GlobalUtil.getLoginUserId();
        // 生成周报需要校验，上传周报时间范围的下周工作计划不能为空，本周工时不可为空；
        //下周工作计划不填写，点击【生成周报】，需提醒“请填写下周工作计划”
        //本周工时没写，点击【生成周报】，需提醒“请填写本周工作日志”
        //二者都没填写，需优先进行工时校验
        final LocalDate startWeekDay = DateUtil.getStartWeekDay(payload.getGenerateDate());
        // 周五
        LocalDate endWeekDay = DateUtil.getEndWeekDay(payload.getGenerateDate());
        // 周日
        endWeekDay = endWeekDay.plusDays(2);

        // 校验本周工作计划
        TimesheetPlanQuery timesheetPlanQuery = new TimesheetPlanQuery();
        List<LocalDate> workDatePlanList = new ArrayList<>();
        workDatePlanList.add(startWeekDay);
        workDatePlanList.add(endWeekDay);
        timesheetPlanQuery.setWorkDate(workDatePlanList);
        timesheetPlanQuery.setTsUserId(userId);
        // 值无所谓
        timesheetPlanQuery.setWorkDescNotNull("null");
        final List<TimesheetPlanVO> timesheetPlanVOList = timesheetPlanService.queryList(timesheetPlanQuery);
        //if (CollectionUtils.isEmpty(timesheetPlanVOList)) {
        //    throw TwException.error("", "请填写本周工作计划");
        //}

        // 校验本周工时填写情况
        TimesheetQuery timesheetQuery = new TimesheetQuery();
        List<LocalDate> workDateList = new ArrayList<>();
        workDateList.add(startWeekDay);
        workDateList.add(endWeekDay);
        timesheetQuery.setWorkDate(workDateList);
        timesheetQuery.setTsUserId(userId);
        List<String> tsStatusList = new ArrayList<>();
        tsStatusList.add(TimesheetStatus.APPROVING.getCode());
        tsStatusList.add(TimesheetStatus.APPROVED.getCode());
        tsStatusList.add(TimesheetStatus.SETTLED.getCode());
        timesheetQuery.setTsStatusList(tsStatusList);
        final List<TimesheetVO> timesheetVOList = timesheetService.queryList(timesheetQuery);
        if (CollectionUtils.isEmpty(timesheetVOList)) {
            throw TwException.error("", "请填写本周工作日志，确保工时状态为审批中、已审批");
        }

        // 校验下周工作计划
        final LocalDate nextStartWeekDay = startWeekDay.plusDays(7);
        final LocalDate nextEndWeekDay = endWeekDay.plusDays(7);
        TimesheetPlanQuery timesheetPlanNextQuery = new TimesheetPlanQuery();
        List<LocalDate> workDatePlanNextList = new ArrayList<>();
        workDatePlanNextList.add(nextStartWeekDay);
        workDatePlanNextList.add(nextEndWeekDay);
        timesheetPlanNextQuery.setWorkDate(workDatePlanNextList);
        timesheetPlanNextQuery.setTsUserId(userId);
        // 值无所谓
        timesheetPlanNextQuery.setWorkDescNotNull("null");
        final List<TimesheetPlanVO> timesheetPlanVONextList = timesheetPlanService.queryList(timesheetPlanNextQuery);
        //if (CollectionUtils.isEmpty(timesheetPlanVONextList)) {
        //    throw TwException.error("", "请填写下周工作计划");
        //}

        // 初始化数据
        initData(payload);

        //（2） 点击生成周报后，撤回一条工作日志，重新点击生成周报进行数据覆盖
        //（3） 修改工作计划以后需要重新生成周报，选择周报所属时间段进行数据覆盖

        // 先判断有没有生成周报，有的话 ，提示不能重复提交
        TimesheetBiweeklyQuery timesheetBiweeklyQuery = new TimesheetBiweeklyQuery();
        timesheetBiweeklyQuery.setStartWeekDateQuery(startWeekDay);
        timesheetBiweeklyQuery.setTsUserId(userId);
        final List<TimesheetBiweeklyVO> timesheetBiweeklyVOS = queryList(timesheetBiweeklyQuery);
        List<TimesheetBiweeklyDetailPayload> timesheetBiweeklyDetailPayloadList = new ArrayList<>();
        TimesheetBiweeklyVO timesheetBiweeklyVOResult;
        if (!CollectionUtils.isEmpty(timesheetBiweeklyVOS)) {
            timesheetBiweeklyVOResult = timesheetBiweeklyVOS.get(0);
            //throw TwException.error("", "请勿重复提交");
            final List<Long> delIdArr = timesheetBiweeklyVOS.stream().map(timesheetBiweeklyVO -> timesheetBiweeklyVO.getId()).collect(Collectors.toList());
            //deleteSoft(delIdArr);
            //// 删除 周报明细
            //timesheetBiweeklyDetailService.deleteSoftByTsbIds(delIdArr);

            // 只更新工时周报情况
            timesheetBiweeklyDetailService.deleteSoftByTsbIds(delIdArr, TimesheetBiweeklyDetailTypeEnum.TIMESHEET.getCode());
            // 保存明细 本周工时
            timesheetVOList.forEach(timesheetVO -> {
                TimesheetBiweeklyDetailPayload week = new TimesheetBiweeklyDetailPayload();
                week.setTsbId(timesheetBiweeklyVOResult.getId());
                week.setExt1(StringUtils.hasText(timesheetVO.getType()) ? timesheetVO.getType() : "PROJ");
                week.setType(TimesheetBiweeklyDetailTypeEnum.TIMESHEET.getCode());
                week.setYearWeek(timesheetVO.getYearWeek());
                week.setProjId(timesheetVO.getProjId());
                week.setProjectName(timesheetVO.getProjName());
                week.setTaskId(timesheetVO.getTaskId());
                week.setTaskName(timesheetVO.getTaskName());
                week.setActId(timesheetVO.getActId());
                week.setActName(timesheetVO.getActName());
                week.setTsbDate(timesheetVO.getWorkDate());
                week.setWorkDesc(timesheetVO.getWorkDesc());
                timesheetBiweeklyDetailPayloadList.add(week);
            });

        } else {
            // 插入新数据
            TimesheetBiweeklyDO entityDo = TimesheetBiweeklyConvert.INSTANCE.toDo(payload);
            final TimesheetBiweeklyVO timesheetBiweeklyVO = TimesheetBiweeklyConvert.INSTANCE.toVo(timesheetBiweeklyRepo.save(entityDo));
            timesheetBiweeklyVOResult = timesheetBiweeklyVO;

            // 保存明细 本周工作计划
            timesheetPlanVOList.forEach(timesheetPlanVO -> {
                TimesheetBiweeklyDetailPayload week = new TimesheetBiweeklyDetailPayload();
                week.setTsbId(timesheetBiweeklyVO.getId());
                week.setType(TimesheetBiweeklyDetailTypeEnum.TIMESHEET_PLAN.getCode());
                week.setExt1(StringUtils.hasText(timesheetPlanVO.getType()) ? timesheetPlanVO.getType() : "PROJ");
                week.setYearWeek(timesheetPlanVO.getYearWeek());
                week.setProjId(timesheetPlanVO.getProjId());
                week.setProjectName(timesheetPlanVO.getProjName());
                week.setTaskId(timesheetPlanVO.getTaskId());
                week.setTaskName(timesheetPlanVO.getTaskName());
                week.setActId(timesheetPlanVO.getActId());
                week.setActName(timesheetPlanVO.getActName());
                week.setTsbDate(timesheetPlanVO.getWorkDate());
                week.setWorkDesc(timesheetPlanVO.getWorkDesc());
                timesheetBiweeklyDetailPayloadList.add(week);
            });

            // 保存明细 下周工作计划
            timesheetPlanVONextList.forEach(timesheetPlanVO -> {
                TimesheetBiweeklyDetailPayload week = new TimesheetBiweeklyDetailPayload();
                week.setTsbId(timesheetBiweeklyVO.getId());
                week.setType(TimesheetBiweeklyDetailTypeEnum.TIMESHEET_PLAN_NEXT_WEEK.getCode());
                week.setExt1(StringUtils.hasText(timesheetPlanVO.getType()) ? timesheetPlanVO.getType() : "PROJ");
                week.setYearWeek(timesheetPlanVO.getYearWeek());
                week.setProjId(timesheetPlanVO.getProjId());
                week.setProjectName(timesheetPlanVO.getProjName());
                week.setTaskId(timesheetPlanVO.getTaskId());
                week.setTaskName(timesheetPlanVO.getTaskName());
                week.setActId(timesheetPlanVO.getActId());
                week.setActName(timesheetPlanVO.getActName());
                week.setTsbDate(timesheetPlanVO.getWorkDate());
                week.setWorkDesc(timesheetPlanVO.getWorkDesc());
                timesheetBiweeklyDetailPayloadList.add(week);
            });
            // 保存明细 本周工时
            timesheetVOList.forEach(timesheetVO -> {
                TimesheetBiweeklyDetailPayload week = new TimesheetBiweeklyDetailPayload();
                week.setTsbId(timesheetBiweeklyVO.getId());
                week.setExt1(StringUtils.hasText(timesheetVO.getType()) ? timesheetVO.getType() : "PROJ");
                week.setType(TimesheetBiweeklyDetailTypeEnum.TIMESHEET.getCode());
                week.setYearWeek(timesheetVO.getYearWeek());
                week.setProjId(timesheetVO.getProjId());
                week.setProjectName(timesheetVO.getProjName());
                week.setTaskId(timesheetVO.getTaskId());
                week.setTaskName(timesheetVO.getTaskName());
                week.setActId(timesheetVO.getActId());
                week.setActName(timesheetVO.getActName());
                week.setTsbDate(timesheetVO.getWorkDate());
                week.setWorkDesc(timesheetVO.getWorkDesc());
                timesheetBiweeklyDetailPayloadList.add(week);
            });

        }
        timesheetBiweeklyDetailPayloadList.forEach(timesheetBiweeklyDetailPayload -> {
            timesheetBiweeklyDetailService.insert(timesheetBiweeklyDetailPayload);
        });
        return timesheetBiweeklyVOResult;
    }


    /**
     * 初始化数据
     *
     * @param payload 有效载荷
     */
    private void initData(TimesheetBiweeklyPayload payload) {
        final Long loginUserId = GlobalUtil.getLoginUserId();
        payload.setTsUserId(loginUserId);
        // 审批人赋值
        final Set<Long> parentIdsByUserId = prdOrgEmployeeService.queryParentIdsByUserId(loginUserId);
        final String receiveUserIds = org.apache.commons.lang3.StringUtils.join(parentIdsByUserId.toArray(), ",");
        payload.setReceiveUserIds(receiveUserIds);
        if (!StringUtils.hasText(payload.getTsbType())) {
            payload.setTsbType("week");
        }
        if (!StringUtils.hasText(payload.getTitle())) {
            String title = "";
            if (payload.getTsbType().equals("week")) {
                title = "周报";
            }
            payload.setTitle(GlobalUtil.getLoginUserName() + title);
        }
        if (null == payload.getIsRead()) {
            payload.setIsRead(0);
        }
        // TODO 获取用户上级、上级的上级....
        if (null == payload.getStartWeekDate()) {
            final LocalDate startWeekDay = DateUtil.getStartWeekDay(payload.getGenerateDate());
            if (null == startWeekDay) {
                throw TwException.error("", "参数异常，请指定生成周报日期");
            }
            payload.setStartWeekDate(startWeekDay);
            payload.setEndWeekDate(startWeekDay.plusDays(7));
            payload.setYearWeek(Integer.parseInt(DateUtil.getYearWeek(startWeekDay)));
        }
        // TODO 获取当前用户的组织信息
        final List<PrdOrgDataRefVO> prdOrgDataRefVOS = prdUserService.queryOrgList();
        if (!CollectionUtils.isEmpty(prdOrgDataRefVOS)) {
            final List<PrdOrgDataRefVO> list = prdOrgDataRefVOS.stream()
                    .filter(prdOrgDataRefVO -> prdOrgDataRefVO.getIsDefault().equals(0))
                    .collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(list)) {
                final Long orgId = list.get(0).getOrgId();
                payload.setTsbBuId(orgId);
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public TimesheetBiweeklyVO update(TimesheetBiweeklyPayload payload) {
        TimesheetBiweeklyDO entity = timesheetBiweeklyRepo.findById(payload.getId()).orElseGet(TimesheetBiweeklyDO::new);
        Assert.notNull(entity.getId(), "不存在");
        TimesheetBiweeklyDO entityDo = TimesheetBiweeklyConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return TimesheetBiweeklyConvert.INSTANCE.toVo(timesheetBiweeklyRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            keys.stream().forEach(id -> {
                Optional<TimesheetBiweeklyDO> optional = timesheetBiweeklyRepo.findById(id);
                if (!optional.isEmpty()) {
                    TimesheetBiweeklyDO entity = optional.get();
                    entity.setDeleteFlag(1);
                    timesheetBiweeklyRepo.save(entity);
                }
            });
        }
    }

    @Override
    public void download(List<TimesheetBiweeklyVO> all, HttpServletResponse response) throws IOException {
        List<Map<String, Object>> list = new ArrayList<>();
        for (TimesheetBiweeklyVO timesheetBiweekly : all) {
            Map<String, Object> map = new LinkedHashMap<>();
            map.put("标题", timesheetBiweekly.getTitle());
            map.put("双周计划类型", timesheetBiweekly.getTsbType());
            map.put("双周计划Buid", timesheetBiweekly.getTsbBuId());
            map.put("发送对象", timesheetBiweekly.getReceiveUserIds());
            map.put("是否已读", timesheetBiweekly.getIsRead());
            map.put("拓展1", timesheetBiweekly.getExt1());
            map.put("拓展2", timesheetBiweekly.getExt2());
            map.put("拓展3", timesheetBiweekly.getExt3());
            map.put("本周开始日期", timesheetBiweekly.getStartWeekDate());
            list.add(map);
        }
        FileUtil.downloadExcel(list, response);
    }
}
