package com.elitesland.tw.tw5.server.prd.my.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.my.payload.TimesheetPlanTempListPayload;
import com.elitesland.tw.tw5.api.prd.my.payload.TimesheetPlanTempPayload;
import com.elitesland.tw.tw5.api.prd.my.query.TimesheetPlanTempQuery;
import com.elitesland.tw.tw5.api.prd.my.service.ProjectService;
import com.elitesland.tw.tw5.api.prd.my.service.ResActivityService;
import com.elitesland.tw.tw5.api.prd.my.service.TaskService;
import com.elitesland.tw.tw5.api.prd.my.service.TimesheetPlanTempService;
import com.elitesland.tw.tw5.api.prd.my.vo.ProjectVO;
import com.elitesland.tw.tw5.api.prd.my.vo.ResActivityVO;
import com.elitesland.tw.tw5.api.prd.my.vo.TaskVO;
import com.elitesland.tw.tw5.api.prd.my.vo.TimesheetPlanTempVO;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.util.FileUtil;
import com.elitesland.tw.tw5.server.common.util.PageUtil;
import com.elitesland.tw.tw5.server.prd.my.constant.ProjectEnum;
import com.elitesland.tw.tw5.server.prd.my.constant.TaskStatusEnum;
import com.elitesland.tw.tw5.server.prd.my.convert.TimesheetPlanTempConvert;
import com.elitesland.tw.tw5.server.prd.my.entity.TimesheetPlanTempDO;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.my.repo.TimesheetPlanTempRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.*;

/**
 * 工作计划模板
 *
 * @author duwh
 * @date 2022-12-14
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class TimesheetPlanTempServiceImpl implements TimesheetPlanTempService {

    private final TimesheetPlanTempRepo timesheetPlanTempRepo;
    private final ProjectService projectService;
    private final TaskService taskService;
    private final ResActivityService resActivityService;

    @Override
    public PagingVO<TimesheetPlanTempVO> paging(TimesheetPlanTempQuery query) {
        Page<TimesheetPlanTempDO> page = timesheetPlanTempRepo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder), query.getPageRequest());
        return PageUtil.toPageVo(page.map(TimesheetPlanTempConvert.INSTANCE::toVo));
    }

    @Override
    public List<TimesheetPlanTempVO> queryList(TimesheetPlanTempQuery query) {
        return TimesheetPlanTempConvert.INSTANCE.toVoList(timesheetPlanTempRepo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder)));
    }

    public long count(TimesheetPlanTempQuery query) {
        return timesheetPlanTempRepo.count((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder));
    }

    @Override
    public TimesheetPlanTempVO queryByKey(Long key) {
        TimesheetPlanTempDO entity = timesheetPlanTempRepo.findById(key).orElseGet(TimesheetPlanTempDO::new);
        Assert.notNull(entity.getId(), "不存在");
        TimesheetPlanTempVO vo = TimesheetPlanTempConvert.INSTANCE.toVo(entity);
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public TimesheetPlanTempVO insert(TimesheetPlanTempPayload payload) {
        // 参数校验
        check(payload);
        // 冗余数据处理
        initData(payload);

        TimesheetPlanTempDO entityDo = TimesheetPlanTempConvert.INSTANCE.toDo(payload);
        return TimesheetPlanTempConvert.INSTANCE.toVo(timesheetPlanTempRepo.save(entityDo));
    }

    private void check(TimesheetPlanTempPayload payload) {
        final Long loginUserId = GlobalUtil.getLoginUserId();
        if (null == payload.getProjId()) {
            throw TwException.error("", "参数异常：projId");
        }
        if (null == payload.getTaskId()) {
            throw TwException.error("", "参数异常：taskId");
        }
        if (!StringUtils.hasText(payload.getType())) {
            throw TwException.error("", "参数异常：type");
        }
        // 校验重复数据
        // 1、 运营类、运维类根据projId校验唯一性
        if (payload.getType().equals(ProjectEnum.OPER.getCode()) || payload.getType().equals(ProjectEnum.DEV_OPS.getCode())) {
            TimesheetPlanTempQuery queryOper = new TimesheetPlanTempQuery();
            queryOper.setProjId(payload.getProjId());
            queryOper.setCreateUserId(loginUserId);
            // 编辑情况
            if (null != payload.getId() && payload.getId() > 0) {
                queryOper.setIdNotEqual(payload.getId());
            }
            final long countOper = count(queryOper);
            if (countOper > 0) {
                throw TwException.error("", "请勿重复添加！");
            }
        } else {
            // 2、 交付类根据taskId校验唯一性
            TimesheetPlanTempQuery query = new TimesheetPlanTempQuery();
            query.setTaskId(payload.getTaskId());
            query.setCreateUserId(loginUserId);
            // 编辑情况
            if (null != payload.getId() && payload.getId() > 0) {
                query.setIdNotEqual(payload.getId());
            }
            final long count = count(query);
            if (count > 0) {
                throw TwException.error("", "请勿重复添加！");
            }
        }


    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<TimesheetPlanTempVO> batchInsert(TimesheetPlanTempListPayload payload) {
        final List<TimesheetPlanTempPayload> timesheetPlanTempList = payload.getTimesheetPlanTempList();
        List<TimesheetPlanTempVO> list = new ArrayList<>();
        if (!CollectionUtils.isEmpty(timesheetPlanTempList)) {
            timesheetPlanTempList.forEach(timesheetPlanTempPayload -> {
                list.add(insert(timesheetPlanTempPayload));
            });
        }
        return list;
    }

    private void initData(TimesheetPlanTempPayload payload) {
        if (null != payload.getProjId()) {
            final ProjectVO projectVO = projectService.queryByProjIdV4(payload.getProjId());
            if (!StringUtils.hasText(payload.getProjNo())) {
                payload.setProjNo(null != projectVO ? projectVO.getProjNo() : "");
            }
            if (!StringUtils.hasText(payload.getProjName())) {
                payload.setProjName(null != projectVO ? projectVO.getProjName() : "");
            }
        }

        if (null != payload.getTaskId()) {
            final TaskVO taskVO = taskService.queryByTaskIdV4(payload.getTaskId());
            if (!StringUtils.hasText(payload.getTaskNo())) {
                payload.setTaskNo(null != taskVO ? taskVO.getTaskNo() : "");
            }
            if (!StringUtils.hasText(payload.getTaskName())) {
                payload.setTaskName(null != taskVO ? taskVO.getTaskName() : "");
            }
            if (null == payload.getEqvaQty()) {
                payload.setEqvaQty(null != taskVO ? taskVO.getEqvaQty() : null);
            }
        }
        if (null != payload.getActId()) {
            final ResActivityVO resActivityVO = resActivityService.queryByActivityIdV4(payload.getActId());
            if (!StringUtils.hasText(payload.getActNo())) {
                payload.setActNo(null != resActivityVO ? resActivityVO.getActNo() : "");
            }
            if (!StringUtils.hasText(payload.getActName())) {
                payload.setActName(null != resActivityVO ? resActivityVO.getActName() : "");
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public TimesheetPlanTempVO update(TimesheetPlanTempPayload payload) {
        TimesheetPlanTempDO entity = timesheetPlanTempRepo.findById(payload.getId()).orElseGet(TimesheetPlanTempDO::new);
        Assert.notNull(entity.getId(), "不存在");
        initData(payload);
        TimesheetPlanTempDO entityDo = TimesheetPlanTempConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return TimesheetPlanTempConvert.INSTANCE.toVo(timesheetPlanTempRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            keys.stream().forEach(id -> {
                Optional<TimesheetPlanTempDO> optional = timesheetPlanTempRepo.findById(id);
                if (!optional.isEmpty()) {
                    TimesheetPlanTempDO entity = optional.get();
                    //#6849 审批中（任务包状态为派发中）不可以删除数据，必须要在4.0中【我提交的流程中】进行撤回，删除以后才可以在5.0中删除
                    //退回后（任务包状态为新建）可以删除
                    //审批通过（任务包状态为激活）可以删除
                    final String taskApplyStatus = entity.getTaskApplyStatus();
                    if (StringUtils.hasText(taskApplyStatus) && taskApplyStatus.equals(TaskStatusEnum.DISPATCH.getCode())) {
                        throw TwException.error("", "审批中不可以删除数据，必须要在4.0中【我提交的流程中】进行撤回，删除以后才可以在5.0中删除");
                    }
                    entity.setDeleteFlag(1);
                    timesheetPlanTempRepo.save(entity);
                }
            });
        }
    }

    @Override
    public void download(List<TimesheetPlanTempVO> all, HttpServletResponse response) throws IOException {
        List<Map<String, Object>> list = new ArrayList<>();
        for (TimesheetPlanTempVO timesheetPlanTemp : all) {
            Map<String, Object> map = new LinkedHashMap<>();
            map.put("任务id", timesheetPlanTemp.getTaskId());
            map.put("工作计划主键", timesheetPlanTemp.getPlanId());
            map.put("状态", timesheetPlanTemp.getStatus());
            map.put("拓展1", timesheetPlanTemp.getExt1());
            map.put("拓展2", timesheetPlanTemp.getExt2());
            map.put("拓展3", timesheetPlanTemp.getExt3());
            map.put("拓展4", timesheetPlanTemp.getExt4());
            map.put("拓展5", timesheetPlanTemp.getExt5());
            list.add(map);
        }
        FileUtil.downloadExcel(list, response);
    }
}
