package com.elitesland.tw.tw5.server.prd.system.service;

import cn.hutool.core.thread.ThreadUtil;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmOperationPlanDetailService;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmCustomerOperationChanceVO;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmCustomerOperationVO;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmOperationPlanDetailVO;
import com.elitesland.tw.tw5.api.prd.system.payload.PrdCommentPayload;
import com.elitesland.tw.tw5.api.prd.system.query.PrdCommentQuery;
import com.elitesland.tw.tw5.api.prd.system.service.PrdCommentService;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdCommentVO;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.QyWx.service.QyWxCommunicationService;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.util.PageUtil;
import com.elitesland.tw.tw5.server.prd.system.constant.PrdCommentObjTypeEnum;
import com.elitesland.tw.tw5.server.prd.system.convert.PrdCommentConvert;
import com.elitesland.tw.tw5.server.prd.system.entity.PrdCommentDO;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.crm.dao.CrmCustomerOperationChanceDAO;
import com.elitesland.tw.tw5.server.prd.crm.dao.CrmCustomerOperationDAO;
import com.elitesland.tw.tw5.server.prd.system.repo.PrdCommentRepo;
import com.elitescloud.cloudt.common.base.PagingVO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * prd评论服务impl
 *
 * @author duwh
 * @date 2022/11/22
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class PrdCommentServiceImpl implements PrdCommentService {

    private final PrdCommentRepo repo;
    private final QyWxCommunicationService qyWxCommunicationService;
    private final CrmOperationPlanDetailService crmOperationPlanDetailService;
    private final CrmCustomerOperationChanceDAO customerOperationChanceDAO;
    private final CrmCustomerOperationDAO crmCustomerOperationDAO;
    private final CacheUtil cacheUtil;
    /**
     * 一级评论标识
     */
    private final static Long REPLY_ID_DEFAULT = 0L;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PrdCommentVO insert(PrdCommentPayload payload) {
        // 校验
        check(payload);
        // 内容不为空才执行
        if (StringUtils.hasText(payload.getContent())) {
            // 数据处理
            dataProcess(payload);

            PrdCommentDO entityDo = PrdCommentConvert.INSTANCE.toDo(payload);
            repo.save(entityDo);
            final String loginUserName = GlobalUtil.getLoginUserName();

            // 经营计划评论 发送企业消息 通知被安特@用户
            if (StringUtils.hasText(payload.getAnnUserIds())) {
                ThreadUtil.execute(() -> {
                    if (payload.getObjType().equals(PrdCommentObjTypeEnum.OPER_PLAN.getCode())) {
                        // 经营计划评论 发送企业消息 通知被安特@用户
                        sendMessageOperPlan(payload, loginUserName);
                    }
                    if (payload.getObjType().equals(PrdCommentObjTypeEnum.OPER_CHANCE.getCode())) {
                        // 经营机会点评论 发送企业消息 通知被安特@用户
                        sendMessageOperChange(payload, loginUserName);
                    }
                });
            }

            return PrdCommentConvert.INSTANCE.toVo(entityDo);
        }
        return PrdCommentConvert.INSTANCE.payloadToVo(payload);
    }

    /**
     * 经营机会点 发送企业消息 通知被安特@用户
     *
     * @param payload 有效载荷
     */
    private void sendMessageOperChange(PrdCommentPayload payload, String currentUserName) {
        if (StringUtils.hasText(payload.getAnnUserIds())) {
            final List<Long> annUserIdsList = Arrays.stream(payload.getAnnUserIds().split(","))
                    .map(s -> Long.valueOf(s)).collect(Collectors.toList());

            // 经营机会点
            final CrmCustomerOperationChanceVO changeVo = customerOperationChanceDAO.queryByKey(payload.getObjId());
            final Long operId = changeVo.getOperId();
            CrmCustomerOperationVO customerOperationVO = crmCustomerOperationDAO.queryById(operId);
            // 公司名称
            String custName = customerOperationVO.getCustName();
            // 机会点-标题
            String planName = changeVo.getTitle();
            //TODO 增加链接调整到 系统
            final String temp = custName + "-" + planName;
            String content = currentUserName + "在“" + temp + "”中@了你";
            qyWxCommunicationService.sendMessageToUser(annUserIdsList, content);
        }
    }

    /**
     * 经营计划评论 发送企业消息 通知被安特@用户
     *
     * @param payload 有效载荷
     */
    private void sendMessageOperPlan(PrdCommentPayload payload, String currentUserName) {
        if (StringUtils.hasText(payload.getAnnUserIds())) {
            final List<Long> annUserIdsList = Arrays.stream(payload.getAnnUserIds().split(","))
                    .map(s -> Long.valueOf(s)).collect(Collectors.toList());

            // 经营计划详情
            CrmOperationPlanDetailVO detailVO = crmOperationPlanDetailService.queryByKey(payload.getObjId());
            final Long operId = detailVO.getOperId();
            CrmCustomerOperationVO customerOperationVO = crmCustomerOperationDAO.queryById(operId);
            // 公司名称
            String custName = customerOperationVO.getCustName();
            // 活动名称
            String planName = detailVO.getPlanName();
            //TODO 增加链接调整到 系统
            final String temp = custName + "-" + planName;
            String content = currentUserName + "在“" + temp + "”中@了你";
            qyWxCommunicationService.sendMessageToUser(annUserIdsList, content);
        }
    }

    /**
     * 数据处理
     *
     * @param payload 有效载荷
     */
    private void dataProcess(PrdCommentPayload payload) {
        // 评论人为空，默认当前登录人
        if (null == payload.getUserId()) {
            payload.setUserId(GlobalUtil.getLoginUserId());
            payload.setUserName(GlobalUtil.getLoginUserName());
            //payload.setUserLogo();
        }
        if (!StringUtils.hasText(payload.getUserName())) {
            payload.setUserName(cacheUtil.getUserName(payload.getUserId()));
            //payload.setUserLogo();
        }
        // 代表一级评论  新增评论场景
        if (null == payload.getReplyId()) {
            payload.setReplyId(REPLY_ID_DEFAULT);
        } else {
            // 一、 无限极的情况
            // 被评论 评论
            final PrdCommentVO commentVO = queryByKey(payload.getReplyId());
            // 记录回复人信息  评论人->回复人
            if (null == payload.getReplyUserId()) {
                payload.setReplyUserId(commentVO.getUserId());
                payload.setReplyUserName(commentVO.getUserName());
                payload.setReplyUserLogo(commentVO.getUserLogo());
            }

            // 二级评论  replyId 前端传入 一级评论的id
            // 二级评论、通过前端传入 replyUserId（回复二级评论的userId）
            if (!StringUtils.hasText(payload.getReplyUserName())) {
                payload.setReplyUserName(cacheUtil.getUserName(payload.getReplyUserId()));
                //payload.setReplyUserLogo();
            }
        }
    }

    /**
     * 检查
     *
     * @param payload 有效载荷
     */
    private void check(PrdCommentPayload payload) {
        if (!StringUtils.hasText(payload.getObjType())) {
            throw TwException.error("", "objType不能为空");
        }
        if (null == payload.getObjId()) {
            throw TwException.error("", "objId不能为空");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PrdCommentVO update(PrdCommentPayload payload) {
        PrdCommentDO entity = repo.findById(payload.getId()).orElseGet(PrdCommentDO::new);
        final Long tempId = entity.getId();
        Assert.notNull(tempId, "不存在");
        PrdCommentDO entityDo = PrdCommentConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        final PrdCommentDO save = repo.save(entity);

        return PrdCommentConvert.INSTANCE.toVo(save);
    }

    @Override
    public PrdCommentVO queryByKey(Long key) {
        PrdCommentDO entity = repo.findById(key).orElseGet(PrdCommentDO::new);
        Assert.notNull(entity.getId(), "不存在");
        final PrdCommentVO crmCustomerOperationVO = PrdCommentConvert.INSTANCE.toVo(entity);
        return crmCustomerOperationVO;
    }

    @Override
    public List<PrdCommentVO> queryList(PrdCommentQuery query) {
        // 校验参数必传
        checkQuery(query);

        return PrdCommentConvert.INSTANCE.toVoList(repo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder)));
    }

    /**
     * 统计
     *
     * @param query 查询
     * @return long
     */
    @Override
    public long count(PrdCommentQuery query) {
        // 校验参数必传
        checkQuery(query);
        final long count = repo.count((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder));
        return count;
    }

    /**
     * 统计
     *
     * @param objTypeEnum obj枚举类型
     * @param objId       obj id
     * @return long
     */
    @Override
    public long count(String objTypeEnum, Long objId) {
        PrdCommentQuery query = new PrdCommentQuery();
        query.setObjType(objTypeEnum);
        query.setObjId(objId);
        return count(query);
    }

    @Override
    public PagingVO<PrdCommentVO> paging(PrdCommentQuery query) {
        // 校验参数必传
        checkQuery(query);

        Page<PrdCommentDO> page = repo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder), query.getPageRequest());
        return PageUtil.toPageVo(page.map(PrdCommentConvert.INSTANCE::toVo));
    }

    /**
     * 检查查询
     *
     * @param query 查询
     */
    private void checkQuery(PrdCommentQuery query) {
        if (!StringUtils.hasText(query.getObjType())) {
            throw TwException.error("", "objType不能为空");
        }
        if (null == query.getObjId()) {
            throw TwException.error("", "objId不能为空");
        }
    }

    @Override
    public PagingVO<PrdCommentVO> pagingPro(PrdCommentQuery query) {
        // 校验参数必传
        checkQuery(query);

        // 默认查第一级
        if (null == query.getReplyId()) {
            query.setReplyId(REPLY_ID_DEFAULT);
        }
        Page<PrdCommentDO> pageDo = repo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder), query.getPageRequest());
        final Page<PrdCommentVO> pageVo = pageDo.map(PrdCommentConvert.INSTANCE::toVo);

        // 二级评论
        final List<PrdCommentVO> content = pageVo.getContent();
        content.forEach(commentVO -> {
            PrdCommentQuery commentQuery = new PrdCommentQuery();
            commentQuery.setObjId(query.getObjId());
            commentQuery.setObjType(query.getObjType());
            // 一级评论主键
            commentQuery.setReplyId(commentVO.getId());
            final List<PrdCommentVO> prdCommentVOS = queryList(commentQuery);
            commentVO.setChildren(prdCommentVOS);
        });
        return PageUtil.toPageVo(pageVo);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            keys.stream().forEach(id -> {
                Optional<PrdCommentDO> optional = repo.findById(id);
                if (!optional.isEmpty()) {
                    PrdCommentDO entity = optional.get();
                    entity.setDeleteFlag(1);
                    repo.save(entity);
                }
            });
        }
    }

    @Override
    public List<PrdCommentVO> queryListPro(PrdCommentQuery query) {
        // 默认查第一级
        if (null == query.getReplyId()) {
            // 先查第一级
            query.setReplyId(REPLY_ID_DEFAULT);
        }
        final List<PrdCommentVO> prdCommentVOList = queryList(query);
        // 二级评论
        prdCommentVOList.forEach(commentVO -> {
            PrdCommentQuery commentQuery = new PrdCommentQuery();
            commentQuery.setObjId(query.getObjId());
            commentQuery.setObjType(query.getObjType());
            // 一级评论主键
            commentQuery.setReplyId(commentVO.getId());
            final List<PrdCommentVO> prdCommentVOS = queryList(commentQuery);
            commentVO.setChildren(prdCommentVOS);
        });
        return prdCommentVOList;
    }

}
