package com.elitesland.tw.tw5crm.server.common.util;

import com.elitesland.tw.tw5.server.common.HttpUtil;
import lombok.RequiredArgsConstructor;
import org.gavaghan.geodesy.Ellipsoid;
import org.gavaghan.geodesy.GeodeticCalculator;
import org.gavaghan.geodesy.GlobalCoordinates;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author carl
 */
@Component
@RequiredArgsConstructor
public class GeodesyUtil {
    @Value("${tw5crm.gps.web_key}")
    private String web_key;

    private final HttpUtil httpUtil;

    /**
     * 根据经纬度获取两个位置距离
     *
     * @param longitudeFrom 经度1
     * @param latitudeFrom  纬度1
     * @param longitudeTo   经度2
     * @param latitudeTo    纬度2
     * @return 公里数
     */
    public double getDistance(double longitudeFrom, double latitudeFrom, double longitudeTo, double latitudeTo) {
        GlobalCoordinates source = new GlobalCoordinates(latitudeFrom, longitudeFrom);
        GlobalCoordinates target = new GlobalCoordinates(latitudeTo, longitudeTo);
        double ellipsoidalDistance = new GeodeticCalculator().calculateGeodeticCurve(Ellipsoid.WGS84, source, target).getEllipsoidalDistance();
        return Math.round(ellipsoidalDistance) / 1000d;
    }

    /**
     * 根据地址获取经纬度
     *
     * @param address 详细地址
     * @return
     */
    public String getLongitudeAndLatitudeByAddress(String address) {
        String url = "https://restapi.amap.com/v3/geocode/geo";

        Map<String, Object> map = new HashMap<>();
        map.put("address", address);
        map.put("key", web_key);
        map.put("output", "JSON");

        Map result = httpUtil.sendGetCallMap(url, map);

        if (result.get("status").toString().equals("1")) {
            List<Map> geocodes = (List<Map>) result.get("geocodes");
            if (geocodes.size() > 0) {
                return geocodes.get(0).get("location").toString();
            }

        }
        return "";
    }

    /**
     * 根据经纬度获取地址
     *
     * @param location
     * @return
     */
    public String getAddressByLocation(String location) {
        String url = "https://restapi.amap.com/v3/geocode/regeo";

        Map<String, Object> map = new HashMap<>();
        map.put("location", location);
        map.put("key", web_key);
        map.put("output", "JSON");
        map.put("radius", 1000);
        map.put("extensions", "base");
        map.put("batch", "false");
        map.put("roadlevel", 1);

        Map result = httpUtil.sendGetCallMap(url, map);

        if (result.get("status").toString().equals("1")) {
            Map regeocode = (Map) result.get("regeocode");
            if (!ObjectUtils.isEmpty(regeocode)) {
                final List<String> formatted_address = (List<String>) regeocode.get("formatted_address");
                if (!CollectionUtils.isEmpty(formatted_address)) {
                    return formatted_address.toString();//大致地址不精确
                }
//                String address = addressComponent.get("province").toString();//省
//                if (!ObjectUtils.isEmpty(addressComponent.get("city"))) {
//                    address += addressComponent.get("city").toString();//市
//                }
//                address += addressComponent.get("district").toString();//区
//
//                Map streetNumber = (Map) addressComponent.get("streetNumber");
//                address += streetNumber.get("street").toString();//z
//
//                return geocodes.get(0).get("location").toString();
            }

        }
        return "";
    }

}
