package com.elitesland.tw.tw5crm.server.contract.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5crm.api.contract.payload.ContractBillingDetailPayload;
import com.elitesland.tw.tw5crm.api.contract.query.ContractBillingDetailQuery;
import com.elitesland.tw.tw5crm.api.contract.query.ContractInvoiceBillingDetailQuery;
import com.elitesland.tw.tw5crm.api.contract.vo.ContractBillingDetailVO;
import com.elitesland.tw.tw5crm.server.contract.entity.*;
import com.elitesland.tw.tw5crm.server.contract.repo.ContractBillingDetailRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

/**
 * 合同管理-开票明细
 *
 * @author wangly
 * @date 2023-04-25
 */
@Repository
@RequiredArgsConstructor
public class ContractBillingDetailDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ContractBillingDetailRepo repo;
    private final QContractBillingDetailDO qdo = QContractBillingDetailDO.contractBillingDetailDO;
    private final QContractBillingApplyDO ado = QContractBillingApplyDO.contractBillingApplyDO;
    private final QContractDO cdo = QContractDO.contractDO;
    private final QContractCollectionPlanDO pdo = QContractCollectionPlanDO.contractCollectionPlanDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ContractBillingDetailVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ContractBillingDetailVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 开票申请表id
                qdo.billingId,
                // 开票流水号
                qdo.billingCode,
                // 合同ID
                qdo.contractId,
                // 收款计划ID
                qdo.collectionPlanId,
                // 合同编号
                qdo.contractCode,
                // 合同名称
                qdo.contractName,
                // 收款阶段
                qdo.collectionStage,
                // 客户名称
                qdo.customerName,
                // 开票金额
                qdo.billingAmount,
                // 附件
                qdo.fileCodes,
                // 排序号
                qdo.sortNo,
                // 拓展字段1
                qdo.ext1,
                // 拓展字段2
                qdo.ext2,
                // 拓展字段3
                qdo.ext3,
                // 拓展字段4
                qdo.ext4,
                // 拓展字段5
                qdo.ext5
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ContractBillingDetailVO> getJpaQueryWhere(ContractBillingDetailQuery query) {
        JPAQuery<ContractBillingDetailVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ContractBillingDetailQuery query) {
        long total = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .where(where(query))
                .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ContractBillingDetailQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 开票申请表id 精确 */
        if (!ObjectUtils.isEmpty(query.getBillingId())) {
            list.add(qdo.billingId.eq(query.getBillingId()));
        }
        /** 开票流水号 精确 */
        if (!ObjectUtils.isEmpty(query.getBillingCode())) {
            list.add(qdo.billingCode.eq(query.getBillingCode()));
        }
        /** 合同ID 精确 */
        if (!ObjectUtils.isEmpty(query.getContractId())) {
            list.add(qdo.contractId.eq(query.getContractId()));
        }
        /** 收款计划ID 精确 */
        if (!ObjectUtils.isEmpty(query.getCollectionPlanId())) {
            list.add(qdo.collectionPlanId.eq(query.getCollectionPlanId()));
        }
        /** 合同编号 精确 */
        if (!ObjectUtils.isEmpty(query.getContractCode())) {
            list.add(qdo.contractCode.eq(query.getContractCode()));
        }
        /** 合同名称 精确 */
        if (!ObjectUtils.isEmpty(query.getContractName())) {
            list.add(qdo.contractName.eq(query.getContractName()));
        }
        /** 收款阶段 精确 */
        if (!ObjectUtils.isEmpty(query.getCollectionStage())) {
            list.add(qdo.collectionStage.eq(query.getCollectionStage()));
        }
        /** 客户名称 精确 */
        if (!ObjectUtils.isEmpty(query.getCustomerName())) {
            list.add(qdo.customerName.eq(query.getCustomerName()));
        }
        /** 开票金额 精确 */
        if (!ObjectUtils.isEmpty(query.getBillingAmount())) {
            list.add(qdo.billingAmount.eq(query.getBillingAmount()));
        }
        /** 附件 精确 */
        if (!ObjectUtils.isEmpty(query.getFileCodes())) {
            list.add(qdo.fileCodes.eq(query.getFileCodes()));
        }
        /** 排序号 精确 */
        if (!ObjectUtils.isEmpty(query.getSortNo())) {
            list.add(qdo.sortNo.eq(query.getSortNo()));
        }
        /** 拓展字段1 精确 */
        if (!ObjectUtils.isEmpty(query.getExt1())) {
            list.add(qdo.ext1.eq(query.getExt1()));
        }
        /** 拓展字段2 精确 */
        if (!ObjectUtils.isEmpty(query.getExt2())) {
            list.add(qdo.ext2.eq(query.getExt2()));
        }
        /** 拓展字段3 精确 */
        if (!ObjectUtils.isEmpty(query.getExt3())) {
            list.add(qdo.ext3.eq(query.getExt3()));
        }
        /** 拓展字段4 精确 */
        if (!ObjectUtils.isEmpty(query.getExt4())) {
            list.add(qdo.ext4.eq(query.getExt4()));
        }
        /** 拓展字段5 精确 */
        if (!ObjectUtils.isEmpty(query.getExt5())) {
            list.add(qdo.ext5.eq(query.getExt5()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ContractBillingDetailVO queryByKey(Long id) {
        JPAQuery<ContractBillingDetailVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ContractBillingDetailVO> queryListDynamic(ContractBillingDetailQuery query) {
        JPAQuery<ContractBillingDetailVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 根据开票申请ID查询开票明细
     *
     * @param billingId 开票申请ID
     * @return 结果集合
     */
    public List<ContractBillingDetailVO> queryListByBillingId(Long billingId) {
        List<ContractBillingDetailVO> contractBillingDetailVOList = jpaQueryFactory.select(Projections.bean(ContractBillingDetailVO.class,
                qdo.id,
                //开票流水号
                qdo.billingCode,
                //合同id
                qdo.contractId,
                //收款计划id
                qdo.collectionPlanId,
                // 合同名称
                cdo.contractName,
                // 收款阶段
                pdo.phase ,
                // 客户名称
                qdo.customerName,
                // 开票金额
                qdo.billingAmount,
                // 附件
                qdo.fileCodes)).from(qdo).
                leftJoin(cdo).on(qdo.contractId.eq(cdo.id)).
                leftJoin(pdo).on(qdo.collectionPlanId.eq(pdo.id)).
        where(qdo.billingId.eq(billingId)).fetch();
        return contractBillingDetailVOList;
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ContractBillingDetailVO> queryPaging(ContractBillingDetailQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<ContractBillingDetailVO> jpaQuery = getJpaQueryWhere(query);
        List<ContractBillingDetailVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ContractBillingDetailVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ContractBillingDetailDO save(ContractBillingDetailDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ContractBillingDetailDO> saveAll(List<ContractBillingDetailDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ContractBillingDetailPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 开票申请表id
        if (payload.getBillingId() != null) {
            update.set(qdo.billingId, payload.getBillingId());
        }
        // 开票流水号
        if (payload.getBillingCode() != null) {
            update.set(qdo.billingCode, payload.getBillingCode());
        }
        // 合同ID
        if (payload.getContractId() != null) {
            update.set(qdo.contractId, payload.getContractId());
        }
        // 收款计划ID
        if (payload.getCollectionPlanId() != null) {
            update.set(qdo.collectionPlanId, payload.getCollectionPlanId());
        }
        // 合同编号
        if (payload.getContractCode() != null) {
            update.set(qdo.contractCode, payload.getContractCode());
        }
        // 合同名称
        if (payload.getContractName() != null) {
            update.set(qdo.contractName, payload.getContractName());
        }
        // 收款阶段
        if (payload.getCollectionStage() != null) {
            update.set(qdo.collectionStage, payload.getCollectionStage());
        }
        // 客户名称
        if (payload.getCustomerName() != null) {
            update.set(qdo.customerName, payload.getCustomerName());
        }
        // 开票金额
        if (payload.getBillingAmount() != null) {
            update.set(qdo.billingAmount, payload.getBillingAmount());
        }
        // 附件
        if (payload.getFileCodes() != null) {
            update.set(qdo.fileCodes, payload.getFileCodes());
        }
        // 排序号
        if (payload.getSortNo() != null) {
            update.set(qdo.sortNo, payload.getSortNo());
        }
        // 拓展字段1
        if (payload.getExt1() != null) {
            update.set(qdo.ext1, payload.getExt1());
        }
        // 拓展字段2
        if (payload.getExt2() != null) {
            update.set(qdo.ext2, payload.getExt2());
        }
        // 拓展字段3
        if (payload.getExt3() != null) {
            update.set(qdo.ext3, payload.getExt3());
        }
        // 拓展字段4
        if (payload.getExt4() != null) {
            update.set(qdo.ext4, payload.getExt4());
        }
        // 拓展字段5
        if (payload.getExt5() != null) {
            update.set(qdo.ext5, payload.getExt5());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 开票申请表id
            if (nullFields.contains("billingId")) {
                update.setNull(qdo.billingId);
            }
            // 开票流水号
            if (nullFields.contains("billingCode")) {
                update.setNull(qdo.billingCode);
            }
            // 合同ID
            if (nullFields.contains("contractId")) {
                update.setNull(qdo.contractId);
            }
            // 收款计划ID
            if (nullFields.contains("collectionPlanId")) {
                update.setNull(qdo.collectionPlanId);
            }
            // 合同编号
            if (nullFields.contains("contractCode")) {
                update.setNull(qdo.contractCode);
            }
            // 合同名称
            if (nullFields.contains("contractName")) {
                update.setNull(qdo.contractName);
            }
            // 收款阶段
            if (nullFields.contains("collectionStage")) {
                update.setNull(qdo.collectionStage);
            }
            // 客户名称
            if (nullFields.contains("customerName")) {
                update.setNull(qdo.customerName);
            }
            // 开票金额
            if (nullFields.contains("billingAmount")) {
                update.setNull(qdo.billingAmount);
            }
            // 附件
            if (nullFields.contains("fileCodes")) {
                update.setNull(qdo.fileCodes);
            }
            // 排序号
            if (nullFields.contains("sortNo")) {
                update.setNull(qdo.sortNo);
            }
            // 拓展字段1
            if (nullFields.contains("ext1")) {
                update.setNull(qdo.ext1);
            }
            // 拓展字段2
            if (nullFields.contains("ext2")) {
                update.setNull(qdo.ext2);
            }
            // 拓展字段3
            if (nullFields.contains("ext3")) {
                update.setNull(qdo.ext3);
            }
            // 拓展字段4
            if (nullFields.contains("ext4")) {
                update.setNull(qdo.ext4);
            }
            // 拓展字段5
            if (nullFields.contains("ext5")) {
                update.setNull(qdo.ext5);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    public PagingVO<ContractBillingDetailVO> queryBillingDetailList(ContractInvoiceBillingDetailQuery query) {
        JPAQuery<ContractBillingDetailVO> jpaQuery  = jpaQueryFactory.select(Projections.bean(ContractBillingDetailVO.class,
                qdo.id,
                // 开票申请流水号
                qdo.billingCode,
                // 开票申请id
                qdo.billingId,
                // 合同名称
                qdo.contractName,
                // 收款阶段
                qdo.collectionStage,
                // 客户名称
                qdo.customerName,
                // 开票金额
                qdo.billingAmount,
                // 附件
                qdo.fileCodes,
                //客户名称
                qdo.customerName,
                //开票状态
                ado.billingStatus,
                //发票号码
                ado.invoiceTaxNum,
                //开票主体
                ado.billingMainbody,
                //发票抬头
                ado.invoiceTitle
        )).from(qdo).leftJoin(ado).on(qdo.billingId.eq(ado.id));
        List<Predicate> list = new ArrayList<>();
        /** 创建人id 精确  如果当前登录人id没有就直接返回空*/
        if (ObjectUtils.isEmpty(query.getCreateUserId())) {
            return PagingVO.empty();
        }
        list.add(qdo.createUserId.eq(query.getCreateUserId()));
        /** 合同编号 精确 */
        if (!ObjectUtils.isEmpty(query.getContractCode())) {
            list.add(qdo.contractCode.eq(query.getContractCode()));
        }
        /** 合同名称 精确 */
        if (!ObjectUtils.isEmpty(query.getContractName())) {
            list.add(qdo.contractName.eq(query.getContractName()));
        }
        /** 客户名称 精确 */
        if (!ObjectUtils.isEmpty(query.getCustomerName())) {
            list.add(qdo.customerName.eq(query.getCustomerName()));
        }
        /** 开票主体 精确 */
        if (!ObjectUtils.isEmpty(query.getBillingMainbody())) {
            list.add(ado.billingMainbody.eq(query.getBillingMainbody()));
        }
        /** 开票日期 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceDate())) {
            list.add(ado.invoiceDate.eq(query.getInvoiceDate()));
        }
       long total = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .where(ExpressionUtils.allOf(list))
                .fetchOne();
        if (total == 0) {
            return PagingVO.empty();
        }
        jpaQuery.where(ExpressionUtils.allOf(list));
        // 根据创建时间排序
        List<OrderItem> orderse = new ArrayList<>();
        orderse.add(OrderItem.desc("create_time"));
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, orderse));
        List<ContractBillingDetailVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ContractBillingDetailVO>builder().records(result).total(total).build();
    }

    /**
     * 通过收款计划id获取开票中/已开票金额
     *
     * @param planId 收款计划id
     */
    public BigDecimal getAmoutByPlanIdAndBillingStatus(Long planId,String billingStatus){
        BigDecimal amount = jpaQueryFactory
                .select(qdo.billingAmount.sum())
                .from(qdo).leftJoin(ado).on(qdo.billingId.eq(ado.id))
                .where(qdo.collectionPlanId.eq(planId))
                .where(ado.billingStatus.eq(billingStatus))
                .fetchOne();
        if(amount==null){
            amount = BigDecimal.ZERO;
        }
        return  amount;
    }
}

