package com.elitesland.tw.tw5crm.server.contract.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5crm.api.contract.payload.ContractCollectionPlanPayload;
import com.elitesland.tw.tw5crm.api.contract.query.ContractCollectionPlanQuery;
import com.elitesland.tw.tw5crm.api.contract.vo.ContractCollectionPlanSimpleVO;
import com.elitesland.tw.tw5crm.api.contract.vo.ContractCollectionPlanVO;
import com.elitesland.tw.tw5crm.server.contract.entity.ContractCollectionPlanDO;
import com.elitesland.tw.tw5crm.server.contract.entity.QContractCollectionPlanDO;
import com.elitesland.tw.tw5crm.server.contract.entity.QContractDO;
import com.elitesland.tw.tw5crm.server.contract.repo.ContractCollectionPlanRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.dsl.BooleanExpression;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 合同收款计划
 *
 * @author duwh
 * @date 2023-04-24
 */
@Repository
@RequiredArgsConstructor
public class ContractCollectionPlanDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ContractCollectionPlanRepo repo;
    private final QContractCollectionPlanDO qdo = QContractCollectionPlanDO.contractCollectionPlanDO;
    private final QContractDO qContractDO = QContractDO.contractDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ContractCollectionPlanVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ContractCollectionPlanVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 收款阶段
                qdo.phase,
                // 收款编号
                qdo.planCode,
                // 收款状态 udc[CRM:CONTRACT:COLLECTION_STATUS]未收款、部分收款、已完成
                qdo.status,
                // 收款计划状态  udc[CRM:CONTRACT:COLLECTION_PLAN_STATUS] 新建、激活
                qdo.planStatus,
                // 当期收款金额
                qdo.collectionAmount,
                // 当期收款比例
                qdo.collectionRatio,
                // 预计收款日期
                qdo.expectRecvDate,
                // 合同主键
                qdo.contractId,
                // 附件
                qdo.fileCodes,
                // 流程实例ID
                qdo.procInstId,
                // 流程审批状态
                qdo.procInstStatus,
                // 提交时间
                qdo.submitTime,
                // 审批时间
                qdo.approvedTime,
                // 版本
                qdo.version,
                // 排序号
                qdo.sortNo,
                // 拓展字段1
                qdo.ext1,
                // 拓展字段2
                qdo.ext2,
                // 拓展字段3
                qdo.ext3,
                // 拓展字段4
                qdo.ext4,
                // 拓展字段5
                qdo.ext5
        )).from(qdo);
    }

    private JPAQuery<ContractCollectionPlanVO> getJpaQuerySelectJoin() {
        return jpaQueryFactory.select(Projections.bean(ContractCollectionPlanVO.class,
            qdo.id,
            //qdo.remark,
            qdo.createUserId,
            qdo.creator,

            //qdo.createTime,
            //qdo.modifyUserId,
            //qdo.updater,
            //qdo.modifyTime,

            qContractDO.contractName,
            qContractDO.contractCode,
            qContractDO.customerId,
            qContractDO.customerName,
            // 收款阶段
            qdo.phase,
            // 收款编号
            qdo.planCode,
            // 收款状态 udc[CRM:CONTRACT:COLLECTION_PLAN_STATUS] 新建、激活
            qdo.status,
            // 收款计划状态 udc[CRM:CONTRACT:COLLECTION_STATUS]未收款、部分收款、已完成
            qdo.planStatus,
            // 当期收款金额
            qdo.collectionAmount,
            // 当期收款比例
            qdo.collectionRatio,
            // 预计收款日期
            qdo.expectRecvDate,
            // 合同主键
            qdo.contractId,
            // 附件
            qdo.fileCodes,
            // 流程实例ID
            qdo.procInstId,
            // 流程审批状态
            qdo.procInstStatus,
            // 提交时间
            qdo.submitTime,
            // 审批时间
            qdo.approvedTime,
            // 版本
            qdo.version,
            // 排序号
            qdo.sortNo,
            // 拓展字段1
            qdo.ext1,
            // 拓展字段2
            qdo.ext2,
            // 拓展字段3
            qdo.ext3,
            // 拓展字段4
            qdo.ext4,
            // 拓展字段5
            qdo.ext5
        )).from(qdo)
            .leftJoin(qContractDO).on(qContractDO.id.eq(qdo.contractId));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ContractCollectionPlanVO> getJpaQueryWhereJoin(ContractCollectionPlanQuery query) {
        JPAQuery<ContractCollectionPlanVO> jpaQuery = getJpaQuerySelectJoin();
        // 条件封装
        jpaQuery.where(whereJoin(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ContractCollectionPlanVO> getJpaQueryWhere(ContractCollectionPlanQuery query) {
        JPAQuery<ContractCollectionPlanVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ContractCollectionPlanQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
     * 关联统计
     *
     * @param query 查询
     * @return long
     */
    public long countJoin(ContractCollectionPlanQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .leftJoin(qContractDO)
            .on(qdo.contractId.eq(qContractDO.id))
            .where(whereJoin(query))
            .fetchOne();
        return total;
    }


    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ContractCollectionPlanQuery query){
        List<Predicate> list = new ArrayList<>();
        whereCommon(query, list);
        return ExpressionUtils.allOf(list);
    }


    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate whereJoin(ContractCollectionPlanQuery query){
        List<Predicate> list = new ArrayList<>();
        whereCommon(query, list);

        list.add(qContractDO.deleteFlag.eq(0));

        // 数据权限
        if (query.getDataPermissionFlag()) {
            final BooleanExpression booleanExpression = qContractDO.createUserId.eq(query.getLoginUserId())
                .or(qContractDO.saleDutyId.eq(query.getLoginUserId()));
            if (!CollectionUtils.isEmpty(query.getSignOrgIdList())) {
                booleanExpression.or(qContractDO.signOrgId.in(query.getSignOrgIdList()));
            }
            list.add(
                booleanExpression
            );
        }
        // 合同名称或编号
        if (!ObjectUtils.isEmpty(query.getContractNameOrCode())) {
            list.add(
                qContractDO.contractCode.like(SqlUtil.toSqlLikeString(query.getContractNameOrCode()))
                    .or(qContractDO.contractName.like(SqlUtil.toSqlLikeString(query.getContractNameOrCode())))
            );
        }
        // 客户主键
        if (!ObjectUtils.isEmpty(query.getCustomerId())) {
            list.add(qContractDO.customerId.eq(query.getCustomerId()));
        }
        // 销售负责人
        if (!ObjectUtils.isEmpty(query.getSaleDutyId())) {
            list.add(qContractDO.saleDutyId.eq(query.getSaleDutyId()));
        }

        return ExpressionUtils.allOf(list);
    }

    private void whereCommon(ContractCollectionPlanQuery query, List<Predicate> list) {
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 收款阶段 精确 */
        if (!ObjectUtils.isEmpty(query.getPhase())) {
            list.add(qdo.phase.eq(query.getPhase()));
        }
        /** 收款阶段 精确 */
        if (!ObjectUtils.isEmpty(query.getPhaseLike())) {
            list.add(qdo.phase.eq(query.getPhase()));
        }
        /** 收款编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getPlanCode())) {
            list.add(qdo.planCode.like(SqlUtil.toSqlLikeString(query.getPlanCode())));
        }
        /** 收款状态 udc[CRM:CONTRACT:COLLECTION_PLAN_STATUS] 新建、激活 精确 */
        if (!ObjectUtils.isEmpty(query.getStatus())) {
            list.add(qdo.status.eq(query.getStatus()));
        }
        /** 收款状态 udc[CRM:CONTRACT:COLLECTION_PLAN_STATUS] 新建、激活 精确 */
        if (!ObjectUtils.isEmpty(query.getStatusArrStr())) {
            String[] statusArr = query.getStatusArrStr().split(",");
            list.add(qdo.status.in(statusArr));
        }
        /** 收款计划状态 udc[CRM:CONTRACT:COLLECTION_STATUS]未收款、部分收款、已完成 精确 */
        if (!ObjectUtils.isEmpty(query.getPlanStatus())) {
            list.add(qdo.planStatus.eq(query.getPlanStatus()));
        }
        /** 当期收款金额 精确 */
        if (!ObjectUtils.isEmpty(query.getCollectionAmount())) {
            list.add(qdo.collectionAmount.eq(query.getCollectionAmount()));
        }
        /** 当期收款比例 精确 */
        if (!ObjectUtils.isEmpty(query.getCollectionRatio())) {
            list.add(qdo.collectionRatio.eq(query.getCollectionRatio()));
        }
        /** 预计收款日期 精确 */
        if (!ObjectUtils.isEmpty(query.getExpectRecvDate())) {
            list.add(qdo.expectRecvDate.eq(query.getExpectRecvDate()));
        }
        /** 预计收款日期 开始 */
        if (!ObjectUtils.isEmpty(query.getExpectRecvDateStart())) {
            list.add(qdo.expectRecvDate.goe(query.getExpectRecvDateStart()));
        }
        /** 预计收款日期 结束 */
        if (!ObjectUtils.isEmpty(query.getExpectRecvDateEnd())) {
            list.add(qdo.expectRecvDate.loe(query.getExpectRecvDateEnd()));
        }
        /** 合同主键 精确 */
        if (!ObjectUtils.isEmpty(query.getContractId())) {
            list.add(qdo.contractId.eq(query.getContractId()));
        }
        /** 附件 精确 */
        if (!ObjectUtils.isEmpty(query.getFileCodes())) {
            list.add(qdo.fileCodes.eq(query.getFileCodes()));
        }
        /** 流程实例ID 精确 */
        if (!ObjectUtils.isEmpty(query.getProcInstId())) {
            list.add(qdo.procInstId.eq(query.getProcInstId()));
        }
        /** 流程审批状态 精确 */
        if (!ObjectUtils.isEmpty(query.getProcInstStatus())) {
            list.add(qdo.procInstStatus.eq(query.getProcInstStatus()));
        }
        /** 提交时间 精确 */
        if (!ObjectUtils.isEmpty(query.getSubmitTime())) {
            list.add(qdo.submitTime.eq(query.getSubmitTime()));
        }
        /** 审批时间 精确 */
        if (!ObjectUtils.isEmpty(query.getApprovedTime())) {
            list.add(qdo.approvedTime.eq(query.getApprovedTime()));
        }
        /** 版本 精确 */
        if (!ObjectUtils.isEmpty(query.getVersion())) {
            list.add(qdo.version.eq(query.getVersion()));
        }
        /** 排序号 精确 */
        if (!ObjectUtils.isEmpty(query.getSortNo())) {
            list.add(qdo.sortNo.eq(query.getSortNo()));
        }
        /** 拓展字段1 精确 */
        if (!ObjectUtils.isEmpty(query.getExt1())) {
            list.add(qdo.ext1.eq(query.getExt1()));
        }
        /** 拓展字段2 精确 */
        if (!ObjectUtils.isEmpty(query.getExt2())) {
            list.add(qdo.ext2.eq(query.getExt2()));
        }
        /** 拓展字段3 精确 */
        if (!ObjectUtils.isEmpty(query.getExt3())) {
            list.add(qdo.ext3.eq(query.getExt3()));
        }
        /** 拓展字段4 精确 */
        if (!ObjectUtils.isEmpty(query.getExt4())) {
            list.add(qdo.ext4.eq(query.getExt4()));
        }
        /** 拓展字段5 精确 */
        if (!ObjectUtils.isEmpty(query.getExt5())) {
            list.add(qdo.ext5.eq(query.getExt5()));
        }
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ContractCollectionPlanVO queryByKey(Long id) {
        JPAQuery<ContractCollectionPlanVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ContractCollectionPlanVO> queryListDynamic(ContractCollectionPlanQuery query) {
        JPAQuery<ContractCollectionPlanVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ContractCollectionPlanVO> queryPaging(ContractCollectionPlanQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<ContractCollectionPlanVO> jpaQuery = getJpaQueryWhere(query);
        List<ContractCollectionPlanVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ContractCollectionPlanVO>builder().records(result).total(total).build();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ContractCollectionPlanVO> queryPagingJoin(ContractCollectionPlanQuery query) {
        long total = countJoin(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<ContractCollectionPlanVO> jpaQuery = getJpaQueryWhereJoin(query);
        List<ContractCollectionPlanVO> result = jpaQuery
            .offset(query.getPageRequest().getOffset())
            .limit(query.getPageRequest().getPageSize())
            .fetch();
        return PagingVO.<ContractCollectionPlanVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ContractCollectionPlanDO save(ContractCollectionPlanDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ContractCollectionPlanDO> saveAll(List<ContractCollectionPlanDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ContractCollectionPlanPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 收款阶段
        if (payload.getPhase() != null) {
            update.set(qdo.phase, payload.getPhase());
        }
        // 收款编号
        if (payload.getPlanCode() != null) {
            update.set(qdo.planCode, payload.getPlanCode());
        }
        // 收款状态 udc[CRM:CONTRACT:COLLECTION_PLAN_STATUS] 新建、激活
        if (payload.getStatus() != null) {
            update.set(qdo.status, payload.getStatus());
        }
        // 收款计划状态 udc[CRM:CONTRACT:COLLECTION_STATUS]未收款、部分收款、已完成
        if (payload.getPlanStatus() != null) {
            update.set(qdo.planStatus, payload.getPlanStatus());
        }
        // 当期收款金额
        if (payload.getCollectionAmount() != null) {
            update.set(qdo.collectionAmount, payload.getCollectionAmount());
        }
        // 当期收款比例
        if (payload.getCollectionRatio() != null) {
            update.set(qdo.collectionRatio, payload.getCollectionRatio());
        }
        // 预计收款日期
        if (payload.getExpectRecvDate() != null) {
            update.set(qdo.expectRecvDate, payload.getExpectRecvDate());
        }
        // 合同主键
        if (payload.getContractId() != null) {
            update.set(qdo.contractId, payload.getContractId());
        }
        // 附件
        if (payload.getFileCodes() != null) {
            update.set(qdo.fileCodes, payload.getFileCodes());
        }
        // 流程实例ID
        if (payload.getProcInstId() != null) {
            update.set(qdo.procInstId, payload.getProcInstId());
        }
        // 流程审批状态
        if (payload.getProcInstStatus() != null) {
            update.set(qdo.procInstStatus, payload.getProcInstStatus());
        }
        // 提交时间
        if (payload.getSubmitTime() != null) {
            update.set(qdo.submitTime, payload.getSubmitTime());
        }
        // 审批时间
        if (payload.getApprovedTime() != null) {
            update.set(qdo.approvedTime, payload.getApprovedTime());
        }
        // 版本
        if (payload.getVersion() != null) {
            update.set(qdo.version, payload.getVersion());
        }
        // 排序号
        if (payload.getSortNo() != null) {
            update.set(qdo.sortNo, payload.getSortNo());
        }
        // 拓展字段1
        if (payload.getExt1() != null) {
            update.set(qdo.ext1, payload.getExt1());
        }
        // 拓展字段2
        if (payload.getExt2() != null) {
            update.set(qdo.ext2, payload.getExt2());
        }
        // 拓展字段3
        if (payload.getExt3() != null) {
            update.set(qdo.ext3, payload.getExt3());
        }
        // 拓展字段4
        if (payload.getExt4() != null) {
            update.set(qdo.ext4, payload.getExt4());
        }
        // 拓展字段5
        if (payload.getExt5() != null) {
            update.set(qdo.ext5, payload.getExt5());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 收款阶段
            if (nullFields.contains("phase")) {
                update.setNull(qdo.phase);
            }
            // 收款编号
            if (nullFields.contains("planCode")) {
                update.setNull(qdo.planCode);
            }
            // 收款状态 udc[CRM:CONTRACT:COLLECTION_PLAN_STATUS] 新建、激活
            if (nullFields.contains("status")) {
                update.setNull(qdo.status);
            }
            // 收款计划状态 udc[CRM:CONTRACT:COLLECTION_STATUS]未收款、部分收款、已完成
            if (nullFields.contains("planStatus")) {
                update.setNull(qdo.planStatus);
            }
            // 当期收款金额
            if (nullFields.contains("collectionAmount")) {
                update.setNull(qdo.collectionAmount);
            }
            // 当期收款比例
            if (nullFields.contains("collectionRatio")) {
                update.setNull(qdo.collectionRatio);
            }
            // 预计收款日期
            if (nullFields.contains("expectRecvDate")) {
                update.setNull(qdo.expectRecvDate);
            }
            // 合同主键
            if (nullFields.contains("contractId")) {
                update.setNull(qdo.contractId);
            }
            // 附件
            if (nullFields.contains("fileCodes")) {
                update.setNull(qdo.fileCodes);
            }
            // 流程实例ID
            if (nullFields.contains("procInstId")) {
                update.setNull(qdo.procInstId);
            }
            // 流程审批状态
            if (nullFields.contains("procInstStatus")) {
                update.setNull(qdo.procInstStatus);
            }
            // 提交时间
            if (nullFields.contains("submitTime")) {
                update.setNull(qdo.submitTime);
            }
            // 审批时间
            if (nullFields.contains("approvedTime")) {
                update.setNull(qdo.approvedTime);
            }
            // 版本
            if (nullFields.contains("version")) {
                update.setNull(qdo.version);
            }
            // 排序号
            if (nullFields.contains("sortNo")) {
                update.setNull(qdo.sortNo);
            }
            // 拓展字段1
            if (nullFields.contains("ext1")) {
                update.setNull(qdo.ext1);
            }
            // 拓展字段2
            if (nullFields.contains("ext2")) {
                update.setNull(qdo.ext2);
            }
            // 拓展字段3
            if (nullFields.contains("ext3")) {
                update.setNull(qdo.ext3);
            }
            // 拓展字段4
            if (nullFields.contains("ext4")) {
                update.setNull(qdo.ext4);
            }
            // 拓展字段5
            if (nullFields.contains("ext5")) {
                update.setNull(qdo.ext5);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据合同主键修改计划状态
     *
     * @param status     状态
     * @param contractId 合同标识
     * @return long
     */
    public long updateStatusByContractId(String status, Long contractId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.planStatus, status)
            .where(qdo.contractId.eq(contractId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据收款计划id查询收款计划和合同信息
     *
     * @param planIdList     收款计划id
     * @return List<ContractCollectionPlanSimpleVO>
     */
    public List<ContractCollectionPlanSimpleVO> queryListByContractId(List<Long> planIdList) {
        return jpaQueryFactory.select(Projections.bean(ContractCollectionPlanSimpleVO.class,
                        qdo.id,
                        // 收款阶段
                        qdo.phase,
                        // 收款编号
                        qdo.planCode,
                        // 当期收款金额
                        qdo.collectionAmount,
                        // 合同主键
                        qdo.contractId,
                        // 合同名称
                        qContractDO.contractCode,
                        // 合同名称
                        qContractDO.contractName,
                        // 客户id
                        qContractDO.customerId,
                        // 客户名称
                        qContractDO.customerName,
                        // 销售负责人id
                        qContractDO.saleDutyId
                )).from(qdo).leftJoin(qContractDO).on(qdo.contractId.eq(qContractDO.id)).
                where(qdo.id.in(planIdList)).fetch();
    }
}

