package com.elitesland.tw.tw5crm.server.contract.dao;

import com.elitesland.tw.tw5crm.api.contract.payload.ContractCountersidePayload;
import com.elitesland.tw.tw5crm.api.contract.query.ContractCountersideQuery;
import com.elitesland.tw.tw5crm.api.contract.vo.ContractCountersideVO;
import com.elitesland.tw.tw5crm.server.contract.entity.ContractCountersideDO;
import com.elitesland.tw.tw5crm.server.contract.entity.QContractCountersideDO;
import com.elitesland.tw.tw5crm.server.contract.repo.ContractCountersideRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 相对方信息
 *
 * @author zach.zhang
 * @date 2023-04-23
 */
@Repository
@RequiredArgsConstructor
public class ContractCountersideDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ContractCountersideRepo repo;
    private final QContractCountersideDO qdo = QContractCountersideDO.contractCountersideDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ContractCountersideVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ContractCountersideVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 相对方名称
                qdo.countersideName,
                // 相对方主体类型
                qdo.countersideType,
                // 联系人
                qdo.contactPerson,
                // 联系人电话
                qdo.phone,
                // 合同id
                qdo.contractId,
                // 拓展字段1
                qdo.ext1,
                // 拓展字段2
                qdo.ext2,
                // 拓展字段3
                qdo.ext3,
                // 拓展字段4
                qdo.ext4,
                // 拓展字段5
                qdo.ext5
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ContractCountersideVO> getJpaQueryWhere(ContractCountersideQuery query) {
        JPAQuery<ContractCountersideVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
            }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ContractCountersideQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ContractCountersideQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 相对方id 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        if (!ObjectUtils.isEmpty(query.getContractId())) {
            list.add(qdo.contractId.eq(query.getContractId()));
        }
        /** 相对方名称 精确 */
        if (!ObjectUtils.isEmpty(query.getCountersideName())) {
            list.add(qdo.countersideName.eq(query.getCountersideName()));
        }
        /** 相对方主体类型 精确 */
        if (!ObjectUtils.isEmpty(query.getCountersideType())) {
            list.add(qdo.countersideType.eq(query.getCountersideType()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ContractCountersideVO queryByKey(Long id) {
        JPAQuery<ContractCountersideVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ContractCountersideVO> queryListDynamic(ContractCountersideQuery query) {
        JPAQuery<ContractCountersideVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ContractCountersideVO> queryPaging(ContractCountersideQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<ContractCountersideVO> jpaQuery = getJpaQueryWhere(query);
        List<ContractCountersideVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ContractCountersideVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ContractCountersideDO save(ContractCountersideDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ContractCountersideDO> saveAll(List<ContractCountersideDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ContractCountersidePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 相对方id
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 相对方名称
        if (payload.getCountersideName() != null) {
            update.set(qdo.countersideName, payload.getCountersideName());
        }
        // 相对方主体类型
        if (payload.getCountersideType() != null) {
            update.set(qdo.countersideType, payload.getCountersideType());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 相对方id
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 相对方名称
            if (nullFields.contains("countersideName")) {
                update.setNull(qdo.countersideName);
            }
            // 相对方主体类型
            if (nullFields.contains("countersideType")) {
                update.setNull(qdo.countersideType);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

