package com.elitesland.tw.tw5crm.server.contract.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5crm.api.contract.payload.ContractProductRefPayload;
import com.elitesland.tw.tw5crm.api.contract.query.ContractProductRefQuery;
import com.elitesland.tw.tw5crm.api.contract.vo.ContractProductRefVO;
import com.elitesland.tw.tw5crm.server.contract.entity.ContractProductRefDO;
import com.elitesland.tw.tw5crm.server.contract.entity.QContractProductRefDO;
import com.elitesland.tw.tw5crm.server.contract.repo.ContractProductRefRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 合同产品信息关系表
 *
 * @author duwh
 * @date 2023-04-25
 */
@Repository
@RequiredArgsConstructor
public class ContractProductRefDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ContractProductRefRepo repo;
    private final QContractProductRefDO qdo = QContractProductRefDO.contractProductRefDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ContractProductRefVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ContractProductRefVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 合同主键
                qdo.contractId,
                // crm_product_price.id 价目表主键
                qdo.priceId,
                // 商品sku主键
                qdo.skuId,
                // 产品SKU名称
                qdo.skuName,
                // 产品SKU编码
                qdo.skuCode,
                // 产品类型 udc[crm:product:spu_type]
                qdo.spuType,
                // 销售单位 udc[crm:product:sku_sale_unit]
                qdo.saleUnit,
                // 税率 产品类型udc拓展字段1中维护
                qdo.taxRate,
                // 数量
                qdo.quantity,
                // 价目表价格
                qdo.pricePrice,
                // 规格属性
                qdo.attributeStr,
                // 排序号
                qdo.sortNo,
                // 拓展字段1
                qdo.ext1,
                // 拓展字段2
                qdo.ext2,
                // 拓展字段3
                qdo.ext3,
                // 拓展字段4
                qdo.ext4,
                // 拓展字段5
                qdo.ext5
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ContractProductRefVO> getJpaQueryWhere(ContractProductRefQuery query) {
        JPAQuery<ContractProductRefVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
            }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ContractProductRefQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ContractProductRefQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 合同主键 精确 */
        if (!ObjectUtils.isEmpty(query.getContractId())) {
            list.add(qdo.contractId.eq(query.getContractId()));
        }
        /** crm_product_price.id 价目表主键 精确 */
        if (!ObjectUtils.isEmpty(query.getPriceId())) {
            list.add(qdo.priceId.eq(query.getPriceId()));
        }
        /** 商品sku主键 精确 */
        if (!ObjectUtils.isEmpty(query.getSkuId())) {
            list.add(qdo.skuId.eq(query.getSkuId()));
        }
        /** 产品SKU名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getSkuName())) {
             list.add(qdo.skuName.like(SqlUtil.toSqlLikeString(query.getSkuName())));
        }
        /** 产品SKU编码 模糊 */
        if (!ObjectUtils.isEmpty(query.getSkuCode())) {
             list.add(qdo.skuCode.like(SqlUtil.toSqlLikeString(query.getSkuCode())));
        }
        /** 产品类型 udc[crm:product:spu_type] 精确 */
        if (!ObjectUtils.isEmpty(query.getSpuType())) {
            list.add(qdo.spuType.eq(query.getSpuType()));
        }
        /** 销售单位 udc[crm:product:sku_sale_unit] 精确 */
        if (!ObjectUtils.isEmpty(query.getSaleUnit())) {
            list.add(qdo.saleUnit.eq(query.getSaleUnit()));
        }
        /** 税率 产品类型udc拓展字段1中维护 精确 */
        if (!ObjectUtils.isEmpty(query.getTaxRate())) {
            list.add(qdo.taxRate.eq(query.getTaxRate()));
        }
        /** 数量 精确 */
        if (!ObjectUtils.isEmpty(query.getQuantity())) {
            list.add(qdo.quantity.eq(query.getQuantity()));
        }
        /** 价目表价格 精确 */
        if (!ObjectUtils.isEmpty(query.getPricePrice())) {
            list.add(qdo.pricePrice.eq(query.getPricePrice()));
        }
        /** 规格属性 模糊 */
        if (!ObjectUtils.isEmpty(query.getAttributeStr())) {
             list.add(qdo.attributeStr.like(SqlUtil.toSqlLikeString(query.getAttributeStr())));
        }
        /** 排序号 精确 */
        if (!ObjectUtils.isEmpty(query.getSortNo())) {
            list.add(qdo.sortNo.eq(query.getSortNo()));
        }
        /** 拓展字段1 精确 */
        if (!ObjectUtils.isEmpty(query.getExt1())) {
            list.add(qdo.ext1.eq(query.getExt1()));
        }
        /** 拓展字段2 精确 */
        if (!ObjectUtils.isEmpty(query.getExt2())) {
            list.add(qdo.ext2.eq(query.getExt2()));
        }
        /** 拓展字段3 精确 */
        if (!ObjectUtils.isEmpty(query.getExt3())) {
            list.add(qdo.ext3.eq(query.getExt3()));
        }
        /** 拓展字段4 精确 */
        if (!ObjectUtils.isEmpty(query.getExt4())) {
            list.add(qdo.ext4.eq(query.getExt4()));
        }
        /** 拓展字段5 精确 */
        if (!ObjectUtils.isEmpty(query.getExt5())) {
            list.add(qdo.ext5.eq(query.getExt5()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ContractProductRefVO queryByKey(Long id) {
        JPAQuery<ContractProductRefVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ContractProductRefVO> queryListDynamic(ContractProductRefQuery query) {
        JPAQuery<ContractProductRefVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ContractProductRefVO> queryPaging(ContractProductRefQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<ContractProductRefVO> jpaQuery = getJpaQueryWhere(query);
        List<ContractProductRefVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ContractProductRefVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ContractProductRefDO save(ContractProductRefDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ContractProductRefDO> saveAll(List<ContractProductRefDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ContractProductRefPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 合同主键
        if (payload.getContractId() != null) {
            update.set(qdo.contractId, payload.getContractId());
        }
        // crm_product_price.id 价目表主键
        if (payload.getPriceId() != null) {
            update.set(qdo.priceId, payload.getPriceId());
        }
        // 商品sku主键
        if (payload.getSkuId() != null) {
            update.set(qdo.skuId, payload.getSkuId());
        }
        // 产品SKU名称
        if (payload.getSkuName() != null) {
            update.set(qdo.skuName, payload.getSkuName());
        }
        // 产品SKU编码
        if (payload.getSkuCode() != null) {
            update.set(qdo.skuCode, payload.getSkuCode());
        }
        // 产品类型 udc[crm:product:spu_type]
        if (payload.getSpuType() != null) {
            update.set(qdo.spuType, payload.getSpuType());
        }
        // 销售单位 udc[crm:product:sku_sale_unit]
        if (payload.getSaleUnit() != null) {
            update.set(qdo.saleUnit, payload.getSaleUnit());
        }
        // 税率 产品类型udc拓展字段1中维护
        if (payload.getTaxRate() != null) {
            update.set(qdo.taxRate, payload.getTaxRate());
        }
        // 数量
        if (payload.getQuantity() != null) {
            update.set(qdo.quantity, payload.getQuantity());
        }
        // 价目表价格
        if (payload.getPricePrice() != null) {
            update.set(qdo.pricePrice, payload.getPricePrice());
        }
        // 规格属性
        if (payload.getAttributeStr() != null) {
            update.set(qdo.attributeStr, payload.getAttributeStr());
        }
        // 排序号
        if (payload.getSortNo() != null) {
            update.set(qdo.sortNo, payload.getSortNo());
        }
        // 拓展字段1
        if (payload.getExt1() != null) {
            update.set(qdo.ext1, payload.getExt1());
        }
        // 拓展字段2
        if (payload.getExt2() != null) {
            update.set(qdo.ext2, payload.getExt2());
        }
        // 拓展字段3
        if (payload.getExt3() != null) {
            update.set(qdo.ext3, payload.getExt3());
        }
        // 拓展字段4
        if (payload.getExt4() != null) {
            update.set(qdo.ext4, payload.getExt4());
        }
        // 拓展字段5
        if (payload.getExt5() != null) {
            update.set(qdo.ext5, payload.getExt5());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 合同主键
            if (nullFields.contains("contractId")) {
                update.setNull(qdo.contractId);
            }
            // crm_product_price.id 价目表主键
            if (nullFields.contains("priceId")) {
                update.setNull(qdo.priceId);
            }
            // 商品sku主键
            if (nullFields.contains("skuId")) {
                update.setNull(qdo.skuId);
            }
            // 产品SKU名称
            if (nullFields.contains("skuName")) {
                update.setNull(qdo.skuName);
            }
            // 产品SKU编码
            if (nullFields.contains("skuCode")) {
                update.setNull(qdo.skuCode);
            }
            // 产品类型 udc[crm:product:spu_type]
            if (nullFields.contains("spuType")) {
                update.setNull(qdo.spuType);
            }
            // 销售单位 udc[crm:product:sku_sale_unit]
            if (nullFields.contains("saleUnit")) {
                update.setNull(qdo.saleUnit);
            }
            // 税率 产品类型udc拓展字段1中维护
            if (nullFields.contains("taxRate")) {
                update.setNull(qdo.taxRate);
            }
            // 数量
            if (nullFields.contains("quantity")) {
                update.setNull(qdo.quantity);
            }
            // 价目表价格
            if (nullFields.contains("pricePrice")) {
                update.setNull(qdo.pricePrice);
            }
            // 规格属性
            if (nullFields.contains("attributeStr")) {
                update.setNull(qdo.attributeStr);
            }
            // 排序号
            if (nullFields.contains("sortNo")) {
                update.setNull(qdo.sortNo);
            }
            // 拓展字段1
            if (nullFields.contains("ext1")) {
                update.setNull(qdo.ext1);
            }
            // 拓展字段2
            if (nullFields.contains("ext2")) {
                update.setNull(qdo.ext2);
            }
            // 拓展字段3
            if (nullFields.contains("ext3")) {
                update.setNull(qdo.ext3);
            }
            // 拓展字段4
            if (nullFields.contains("ext4")) {
                update.setNull(qdo.ext4);
            }
            // 拓展字段5
            if (nullFields.contains("ext5")) {
                update.setNull(qdo.ext5);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

