package com.elitesland.tw.tw5crm.server.contract.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5crm.api.contract.payload.ContractReceiveDetailsPayload;
import com.elitesland.tw.tw5crm.api.contract.query.ContractReceiveDetailsQuery;
import com.elitesland.tw.tw5crm.api.contract.vo.ContractReceiveDetailsVO;
import com.elitesland.tw.tw5crm.server.contract.entity.ContractReceiveDetailsDO;
import com.elitesland.tw.tw5crm.server.contract.entity.QContractReceiveDetailsDO;
import com.elitesland.tw.tw5crm.server.contract.repo.ContractReceiveDetailsRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

/**
 * 到款明细
 *
 * @author zach.zhang
 * @date 2023-04-25
 */
@Repository
@RequiredArgsConstructor
public class ContractReceiveDetailsDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ContractReceiveDetailsRepo repo;
    private final QContractReceiveDetailsDO qdo = QContractReceiveDetailsDO.contractReceiveDetailsDO;



    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ContractReceiveDetailsVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ContractReceiveDetailsVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 合同id
                qdo.contractId,
                // 合同名称
                qdo.contractName,
                // 收款计划id
                qdo.planId,
                // 收款阶段
                qdo.phase,
                // 客户id
                qdo.customerId,
                // 客户名称
                qdo.customerName,
                // 销售负责人id
                qdo.saleDutyId,
                // 到款金额
                qdo.receiveMoney,
                // 销售负责人名称
                qdo.saleDutyName
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ContractReceiveDetailsVO> getJpaQueryWhere(ContractReceiveDetailsQuery query) {
        JPAQuery<ContractReceiveDetailsVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ContractReceiveDetailsQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ContractReceiveDetailsQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 主表主键 crm_contract_receive_account.id 精确 */
        if (!ObjectUtils.isEmpty(query.getReceiveAccountId())) {
            list.add(qdo.receiveAccountId.eq(query.getReceiveAccountId()));
        }
        /** 到款明细id 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 合同id 精确 */
        if (!ObjectUtils.isEmpty(query.getContractId())) {
            list.add(qdo.contractId.eq(query.getContractId()));
        }
        /** 合同名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getContractName())) {
            list.add(qdo.contractName.like(SqlUtil.toSqlLikeString(query.getContractName())));
        }
        /** 收款计划id 精确 */
        if (!ObjectUtils.isEmpty(query.getPlanId())) {
            list.add(qdo.planId.eq(query.getPlanId()));
        }
        /** 收款阶段 模糊 */
        if (!ObjectUtils.isEmpty(query.getPhase())) {
            list.add(qdo.phase.like(SqlUtil.toSqlLikeString(query.getPhase())));
        }
        /** 客户id 精确 */
        if (!ObjectUtils.isEmpty(query.getCustomerId())) {
            list.add(qdo.customerId.eq(query.getCustomerId()));
        }
        /** 客户名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getCustomerName())) {
            list.add(qdo.customerName.like(SqlUtil.toSqlLikeString(query.getCustomerName())));
        }
        /** 销售负责人id 精确 */
        if (!ObjectUtils.isEmpty(query.getSaleDutyId())) {
            list.add(qdo.saleDutyId.eq(query.getSaleDutyId()));
        }
        /** 到款金额 精确 */
        if (!ObjectUtils.isEmpty(query.getReceiveMoney())) {
            list.add(qdo.receiveMoney.eq(query.getReceiveMoney()));
        }
        /** 销售负责人名称 精确 */
        if (!ObjectUtils.isEmpty(query.getSaleDutyName())) {
            list.add(qdo.saleDutyName.eq(query.getSaleDutyName()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ContractReceiveDetailsVO queryByKey(Long id) {
        JPAQuery<ContractReceiveDetailsVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ContractReceiveDetailsVO> queryListDynamic(ContractReceiveDetailsQuery query) {
        JPAQuery<ContractReceiveDetailsVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ContractReceiveDetailsVO> queryPaging(ContractReceiveDetailsQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<ContractReceiveDetailsVO> jpaQuery = getJpaQueryWhere(query);
        List<ContractReceiveDetailsVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ContractReceiveDetailsVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ContractReceiveDetailsDO save(ContractReceiveDetailsDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ContractReceiveDetailsDO> saveAll(List<ContractReceiveDetailsDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ContractReceiveDetailsPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
    /**
     * 根据收款计划id查询总的到款金额
     *
     * @param planId 收款计划id
     * @return totalReceiveMoney 到款录入总金额
     */
    public BigDecimal queryTotalReceiveMoneyByPlanId(Long planId){

        BigDecimal totalReceiveMoney = jpaQueryFactory.select(qdo.receiveMoney.sum()).from(qdo).where(qdo.planId.eq(planId)).fetchOne();
        if(totalReceiveMoney==null){
            totalReceiveMoney = BigDecimal.ZERO;
        }
        return totalReceiveMoney;
    }

}

