package com.elitesland.tw.tw5crm.server.contract.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5crm.api.contract.payload.ContractSignPayload;
import com.elitesland.tw.tw5crm.api.contract.query.ContractSignQuery;
import com.elitesland.tw.tw5crm.api.contract.vo.ContractSignVO;
import com.elitesland.tw.tw5crm.server.contract.entity.ContractSignDO;
import com.elitesland.tw.tw5crm.server.contract.entity.QContractSignDO;
import com.elitesland.tw.tw5crm.server.contract.repo.ContractSignRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;

/**
 * 合同签订
 *
 * @author zach.zhang
 * @date 2023-04-26
 */
@Repository
@RequiredArgsConstructor
public class ContractSignDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ContractSignRepo repo;
    private final QContractSignDO qdo = QContractSignDO.contractSignDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ContractSignVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ContractSignVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 合同id
                qdo.contractId,
                // 合同名称
                qdo.contractName,
                // 客户id
                qdo.customerId,
                // 客户名称
                qdo.customerName,
                // 我方签约主体id
                qdo.signcompanyId,
                // 我方签约主体名称
                qdo.signcompanyName,
                // 币种
                qdo.currency,
                // 合同金额
                qdo.contractAmount,
                // 签订方式udc[CRM:SIGN:TYPE](面签、物流签、第三方电子签)
                qdo.
signType,
                // 签订时间
                qdo.signDate,
                // 物流公司
                qdo.logisticsCompany,
                // 物流单号
                qdo.logisticsNumber,
                // 单据号
                qdo.filecodes
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ContractSignVO> getJpaQueryWhere(ContractSignQuery query) {
        JPAQuery<ContractSignVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
            }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ContractSignQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ContractSignQuery query){
        List<Predicate> list = new ArrayList<>();
        if(null != query.getContractId()){
            list.add(qdo.contractId.eq(query.getContractId()));
        }
        if(null != query.getCustomerId()){
            list.add(qdo.customerId.eq(query.getCustomerId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ContractSignVO queryByKey(Long id) {
        JPAQuery<ContractSignVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ContractSignVO> queryListDynamic(ContractSignQuery query) {
        JPAQuery<ContractSignVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ContractSignVO> queryPaging(ContractSignQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<ContractSignVO> jpaQuery = getJpaQueryWhere(query);
        List<ContractSignVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ContractSignVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ContractSignDO save(ContractSignDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ContractSignDO> saveAll(List<ContractSignDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ContractSignPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

