package com.elitesland.tw.tw5crm.server.contract.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5crm.api.contract.payload.ContractTemplatePayload;
import com.elitesland.tw.tw5crm.api.contract.query.ContractTemplateQuery;
import com.elitesland.tw.tw5crm.api.contract.vo.ContractTemplateVO;
import com.elitesland.tw.tw5crm.server.contract.entity.ContractTemplateDO;
import com.elitesland.tw.tw5crm.server.contract.entity.QContractTemplateDO;
import com.elitesland.tw.tw5crm.server.contract.entity.QContractTemplateOrgRefDO;
import com.elitesland.tw.tw5crm.server.contract.repo.ContractTemplateRepo;
import com.elitesland.tw.tw5crm.server.contract.service.ContractTemplateServiceImpl;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.dsl.BooleanExpression;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 合同模版
 *
 * @author duwh
 * @date 2023-04-20
 */
@Repository
@RequiredArgsConstructor
public class ContractTemplateDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ContractTemplateRepo repo;
    private final QContractTemplateDO qdo = QContractTemplateDO.contractTemplateDO;
    private final QContractTemplateOrgRefDO qContractTemplateOrgRefDO = QContractTemplateOrgRefDO.contractTemplateOrgRefDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ContractTemplateVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ContractTemplateVO.class,
                qdo.id,
                //qdo.remark,
                qdo.createUserId,
                qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 模版名称
                qdo.name,
                // 模版编号
                qdo.code,
                // 模版类型 udc[CRM:CONTRACT:TEMPLATE:TYPE]
                qdo.type,
                // 状态 udc[CRM:CONTRACT:TEMPLATE:STATUS]
                qdo.status,
                // 分类主键（预留后期用）
                qdo.categoryId,
                // 版本
                qdo.version,
                // 试用部门主键集合，英文逗号隔开；全部 ALL
                qdo.orgIdList,
                // 试用部门名称（冗余）
                qdo.orgNameList,
                // 流程实例ID
                qdo.procInstId,
                // 流程审批状态
                qdo.procInstStatus,
                // 提交时间
                qdo.submitTime,
                // 审批时间
                qdo.approvedTime,
                // 排序号
                qdo.sortNo,
                // 拓展字段1
                qdo.ext1,
                // 拓展字段2
                qdo.ext2,
                // 拓展字段3
                qdo.ext3,
                // 拓展字段4
                qdo.ext4,
                // 拓展字段5
                qdo.ext5,
                // 模版分类 udc[CRM:CONTRACT:TEMPLATE:CATEGORY]
                qdo.category,
                qdo.categoryValue,
                qdo.categoryDesc,
                qdo.fileCodes,
                qdo.yeedocFileCodes
        )).from(qdo).leftJoin(qContractTemplateOrgRefDO).on(qdo.id.eq(qContractTemplateOrgRefDO.tempId))
        .distinct();
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ContractTemplateVO> getJpaQueryWhere(ContractTemplateQuery query) {
        JPAQuery<ContractTemplateVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ContractTemplateQuery query) {
        long total = jpaQueryFactory
            .select(qdo.id.countDistinct())
            .from(qdo)
            .leftJoin(qContractTemplateOrgRefDO).on(qdo.id.eq(qContractTemplateOrgRefDO.tempId))
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ContractTemplateQuery query){
        List<Predicate> list = new ArrayList<>();
        list.add(qContractTemplateOrgRefDO.deleteFlag.eq(0));
        // 需要权限
        if (query.getDataPermissionFlag()) {
            BooleanExpression jpaQueryOr = qdo.createUserId.eq(query.getLoginUserId())
                .or(qContractTemplateOrgRefDO.orgId.eq(query.getLoginOrgId()))
                .or(qdo.orgIdList.eq(ContractTemplateServiceImpl.ALL));
            list.add(jpaQueryOr);
        }
        /** 关键字 模糊 */
        if (!ObjectUtils.isEmpty(query.getKeyword())) {
            list.add(qdo.name.like(SqlUtil.toSqlLikeString(query.getKeyword())).or(qdo.code.like(SqlUtil.toSqlLikeString(query.getKeyword()))));
        }
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 模版名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getName())) {
             list.add(qdo.name.like(SqlUtil.toSqlLikeString(query.getName())));
        }
        /** 模版编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getCode())) {
             list.add(qdo.code.like(SqlUtil.toSqlLikeString(query.getCode())));
        }
        /** 模版类型 udc[CRM:CONTRACT:TEMPLATE:TYPE] 精确 */
        if (!ObjectUtils.isEmpty(query.getType())) {
            list.add(qdo.type.eq(query.getType()));
        }
        /** 状态 udc[CRM:CONTRACT:TEMPLATE:STATUS] 精确 */
        if (!ObjectUtils.isEmpty(query.getStatus())) {
            list.add(qdo.status.eq(query.getStatus()));
        }
        if (!ObjectUtils.isEmpty(query.getStatusArrStr())){
            String[] statusArr = query.getStatusArrStr().split(",");
            list.add(qdo.status.in(statusArr));
        }
        /** 分类主键（预留后期用） 精确 */
        if (!ObjectUtils.isEmpty(query.getCategoryId())) {
            list.add(qdo.categoryId.eq(query.getCategoryId()));
        }
        /** 版本 精确 */
        if (!ObjectUtils.isEmpty(query.getVersion())) {
            list.add(qdo.version.eq(query.getVersion()));
        }
        final String orgIdList = query.getOrgIdList();
        /** 试用部门主键集合，英文逗号隔开；全部 ALL 精确 */
        if (!ObjectUtils.isEmpty(orgIdList)) {
            if (!orgIdList.equals(ContractTemplateServiceImpl.ALL)) {
                final String[] split = orgIdList.split(",");
                final List<String> strings = Arrays.asList(split);
                final List<Long> orgIdLongList = strings.stream().map(s -> Long.valueOf(s)).collect(Collectors.toList());
                list.add(qContractTemplateOrgRefDO.orgId.in(orgIdLongList));
            }
        }
        /** 试用部门名称（冗余） 模糊 */
        if (!ObjectUtils.isEmpty(query.getOrgNameList())) {
             list.add(qdo.orgNameList.like(SqlUtil.toSqlLikeString(query.getOrgNameList())));
        }
        /** 流程实例ID 精确 */
        if (!ObjectUtils.isEmpty(query.getProcInstId())) {
            list.add(qdo.procInstId.eq(query.getProcInstId()));
        }
        /** 流程审批状态 精确 */
        if (!ObjectUtils.isEmpty(query.getProcInstStatus())) {
            list.add(qdo.procInstStatus.eq(query.getProcInstStatus()));
        }
        /** 提交时间 精确 */
        if (!ObjectUtils.isEmpty(query.getSubmitTime())) {
            list.add(qdo.submitTime.eq(query.getSubmitTime()));
        }
        /** 审批时间 精确 */
        if (!ObjectUtils.isEmpty(query.getApprovedTime())) {
            list.add(qdo.approvedTime.eq(query.getApprovedTime()));
        }
        /** 排序号 精确 */
        if (!ObjectUtils.isEmpty(query.getSortNo())) {
            list.add(qdo.sortNo.eq(query.getSortNo()));
        }
        /** 拓展字段1 精确 */
        if (!ObjectUtils.isEmpty(query.getExt1())) {
            list.add(qdo.ext1.eq(query.getExt1()));
        }
        /** 拓展字段2 精确 */
        if (!ObjectUtils.isEmpty(query.getExt2())) {
            list.add(qdo.ext2.eq(query.getExt2()));
        }
        /** 拓展字段3 精确 */
        if (!ObjectUtils.isEmpty(query.getExt3())) {
            list.add(qdo.ext3.eq(query.getExt3()));
        }
        /** 拓展字段4 精确 */
        if (!ObjectUtils.isEmpty(query.getExt4())) {
            list.add(qdo.ext4.eq(query.getExt4()));
        }
        /** 拓展字段5 精确 */
        if (!ObjectUtils.isEmpty(query.getExt5())) {
            list.add(qdo.ext5.eq(query.getExt5()));
        }
        /** 模版分类 udc[CRM:CONTRACT:TEMPLATE:CATEGORY] 精确 */
        if (!ObjectUtils.isEmpty(query.getCategory())) {
            list.add(qdo.category.eq(query.getCategory()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ContractTemplateVO queryByKey(Long id) {
        JPAQuery<ContractTemplateVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ContractTemplateVO> queryListDynamic(ContractTemplateQuery query) {
        JPAQuery<ContractTemplateVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ContractTemplateVO> queryPaging(ContractTemplateQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<ContractTemplateVO> jpaQuery = getJpaQueryWhere(query);
        List<ContractTemplateVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ContractTemplateVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ContractTemplateDO save(ContractTemplateDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ContractTemplateDO> saveAll(List<ContractTemplateDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ContractTemplatePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 模版名称
        if (payload.getName() != null) {
            update.set(qdo.name, payload.getName());
        }
        // 模版编号
        if (payload.getCode() != null) {
            update.set(qdo.code, payload.getCode());
        }
        // 模版类型 udc[CRM:CONTRACT:TEMPLATE:TYPE]
        if (payload.getType() != null) {
            update.set(qdo.type, payload.getType());
        }
        // 状态 udc[CRM:CONTRACT:TEMPLATE:STATUS]
        if (payload.getStatus() != null) {
            update.set(qdo.status, payload.getStatus());
        }
        // 分类主键（预留后期用）
        if (payload.getCategoryId() != null) {
            update.set(qdo.categoryId, payload.getCategoryId());
        }
        // 版本
        if (payload.getVersion() != null) {
            update.set(qdo.version, payload.getVersion());
        }
        // 试用部门主键集合，英文逗号隔开；全部 ALL
        if (payload.getOrgIdList() != null) {
            update.set(qdo.orgIdList, payload.getOrgIdList());
        }
        // 试用部门名称（冗余）
        if (payload.getOrgNameList() != null) {
            update.set(qdo.orgNameList, payload.getOrgNameList());
        }
        // 流程实例ID
        if (payload.getProcInstId() != null) {
            update.set(qdo.procInstId, payload.getProcInstId());
        }
        // 流程审批状态
        if (payload.getProcInstStatus() != null) {
            update.set(qdo.procInstStatus, payload.getProcInstStatus());
        }
        // 提交时间
        if (payload.getSubmitTime() != null) {
            update.set(qdo.submitTime, payload.getSubmitTime());
        }
        // 审批时间
        if (payload.getApprovedTime() != null) {
            update.set(qdo.approvedTime, payload.getApprovedTime());
        }
        // 排序号
        if (payload.getSortNo() != null) {
            update.set(qdo.sortNo, payload.getSortNo());
        }
        // 拓展字段1
        if (payload.getExt1() != null) {
            update.set(qdo.ext1, payload.getExt1());
        }
        // 拓展字段2
        if (payload.getExt2() != null) {
            update.set(qdo.ext2, payload.getExt2());
        }
        // 拓展字段3
        if (payload.getExt3() != null) {
            update.set(qdo.ext3, payload.getExt3());
        }
        // 拓展字段4
        if (payload.getExt4() != null) {
            update.set(qdo.ext4, payload.getExt4());
        }
        // 拓展字段5
        if (payload.getExt5() != null) {
            update.set(qdo.ext5, payload.getExt5());
        }
        // 模版分类 udc[CRM:CONTRACT:TEMPLATE:CATEGORY]
        if (payload.getCategory() != null) {
            update.set(qdo.category, payload.getCategory());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 模版名称
            if (nullFields.contains("name")) {
                update.setNull(qdo.name);
            }
            // 模版编号
            if (nullFields.contains("code")) {
                update.setNull(qdo.code);
            }
            // 模版类型 udc[CRM:CONTRACT:TEMPLATE:TYPE]
            if (nullFields.contains("type")) {
                update.setNull(qdo.type);
            }
            // 状态 udc[CRM:CONTRACT:TEMPLATE:STATUS]
            if (nullFields.contains("status")) {
                update.setNull(qdo.status);
            }
            // 分类主键（预留后期用）
            if (nullFields.contains("categoryId")) {
                update.setNull(qdo.categoryId);
            }
            // 版本
            if (nullFields.contains("version")) {
                update.setNull(qdo.version);
            }
            // 试用部门主键集合，英文逗号隔开；全部 ALL
            if (nullFields.contains("orgIdList")) {
                update.setNull(qdo.orgIdList);
            }
            // 试用部门名称（冗余）
            if (nullFields.contains("orgNameList")) {
                update.setNull(qdo.orgNameList);
            }
            // 流程实例ID
            if (nullFields.contains("procInstId")) {
                update.setNull(qdo.procInstId);
            }
            // 流程审批状态
            if (nullFields.contains("procInstStatus")) {
                update.setNull(qdo.procInstStatus);
            }
            // 提交时间
            if (nullFields.contains("submitTime")) {
                update.setNull(qdo.submitTime);
            }
            // 审批时间
            if (nullFields.contains("approvedTime")) {
                update.setNull(qdo.approvedTime);
            }
            // 排序号
            if (nullFields.contains("sortNo")) {
                update.setNull(qdo.sortNo);
            }
            // 拓展字段1
            if (nullFields.contains("ext1")) {
                update.setNull(qdo.ext1);
            }
            // 拓展字段2
            if (nullFields.contains("ext2")) {
                update.setNull(qdo.ext2);
            }
            // 拓展字段3
            if (nullFields.contains("ext3")) {
                update.setNull(qdo.ext3);
            }
            // 拓展字段4
            if (nullFields.contains("ext4")) {
                update.setNull(qdo.ext4);
            }
            // 拓展字段5
            if (nullFields.contains("ext5")) {
                update.setNull(qdo.ext5);
            }
            // 模版分类 udc[CRM:CONTRACT:TEMPLATE:CATEGORY]
            if (nullFields.contains("category")) {
                update.setNull(qdo.category);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 更新工作流程
     *
     * @param payload 有效载荷
     * @return long
     */
    public long updateWorkFlow(ContractTemplatePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .where(qdo.id.eq(payload.getId()));

        // 流程实例id
        if (!ObjectUtils.isEmpty(payload.getProcInstId())) {
            update.set(qdo.procInstId, payload.getProcInstId());
        } else {
            update.setNull(qdo.procInstId);
        }
        // 审批状态
        if (!ObjectUtils.isEmpty(payload.getProcInstStatus())) {
            update.set(qdo.procInstStatus, payload.getProcInstStatus());
        } else {
            update.setNull(qdo.procInstStatus);
        }
        // 状态
        if (!ObjectUtils.isEmpty(payload.getStatus())) {
            update.set(qdo.status, payload.getStatus());
        } else {
            update.setNull(qdo.status);
        }
        // 提审时间
        if (!ObjectUtils.isEmpty(payload.getSubmitTime())) {
            update.set(qdo.submitTime, payload.getSubmitTime());
        }
        // 审批结束时间
        if (!ObjectUtils.isEmpty(payload.getApprovedTime())) {
            update.set(qdo.approvedTime, payload.getApprovedTime());
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 更新变更工作流程
     *
     * @param payload 有效载荷
     * @return long
     */
    public long updateChangeWorkFlow(ContractTemplatePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .where(qdo.id.eq(payload.getId()));

        // 变更主键
        if (!ObjectUtils.isEmpty(payload.getChangeId())) {
            update.set(qdo.changeId, payload.getChangeId());
        } else {
            update.setNull(qdo.changeId);
        }

        // 流程实例id
        if (!ObjectUtils.isEmpty(payload.getChangeProcInstId())) {
            update.set(qdo.changeProcInstId, payload.getChangeProcInstId());
        } else {
            update.setNull(qdo.changeProcInstId);
        }
        // 审批状态
        if (!ObjectUtils.isEmpty(payload.getChangeProcInstStatus())) {
            update.set(qdo.changeProcInstStatus, payload.getChangeProcInstStatus());
        } else {
            update.setNull(qdo.changeProcInstStatus);
        }
        // 状态
        if (!ObjectUtils.isEmpty(payload.getStatus())) {
            update.set(qdo.status, payload.getStatus());
        } else {
            update.setNull(qdo.status);
        }
        // 提审时间
        if (!ObjectUtils.isEmpty(payload.getChangeSubmitTime())) {
            update.set(qdo.changeSubmitTime, payload.getChangeSubmitTime());
        }
        // 审批结束时间
        if (!ObjectUtils.isEmpty(payload.getChangeApprovedTime())) {
            update.set(qdo.changeApprovedTime, payload.getChangeApprovedTime());
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

}

