package com.elitesland.tw.tw5crm.server.contract.dao;

import com.elitesland.tw.tw5crm.api.contract.payload.ContractTemplateOrgRefPayload;
import com.elitesland.tw.tw5crm.api.contract.query.ContractTemplateOrgRefQuery;
import com.elitesland.tw.tw5crm.api.contract.vo.ContractTemplateOrgRefVO;
import com.elitesland.tw.tw5crm.server.contract.entity.ContractTemplateOrgRefDO;
import com.elitesland.tw.tw5crm.server.contract.entity.QContractTemplateOrgRefDO;
import com.elitesland.tw.tw5crm.server.contract.repo.ContractTemplateOrgRefRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 合同模版试用部门关系表
 *
 * @author duwh
 * @date 2023-04-20
 */
@Repository
@RequiredArgsConstructor
public class ContractTemplateOrgRefDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ContractTemplateOrgRefRepo repo;
    private final QContractTemplateOrgRefDO qdo = QContractTemplateOrgRefDO.contractTemplateOrgRefDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ContractTemplateOrgRefVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ContractTemplateOrgRefVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 部门名称（冗余）
                qdo.name,
                // 模版ID crm_contract_template.id
                qdo.tempId,
                // 组织BuID prd_org_organization.id
                qdo.orgId
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ContractTemplateOrgRefVO> getJpaQueryWhere(ContractTemplateOrgRefQuery query) {
        JPAQuery<ContractTemplateOrgRefVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
            }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ContractTemplateOrgRefQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ContractTemplateOrgRefQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 部门名称（冗余） 模糊 */
        if (!ObjectUtils.isEmpty(query.getName())) {
             list.add(qdo.name.like(SqlUtil.toSqlLikeString(query.getName())));
        }
        /** 模版ID crm_contract_template.id 精确 */
        if (!ObjectUtils.isEmpty(query.getTempId())) {
            list.add(qdo.tempId.eq(query.getTempId()));
        }
        /** 组织BuID prd_org_organization.id 精确 */
        if (!ObjectUtils.isEmpty(query.getOrgId())) {
            list.add(qdo.orgId.eq(query.getOrgId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ContractTemplateOrgRefVO queryByKey(Long id) {
        JPAQuery<ContractTemplateOrgRefVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ContractTemplateOrgRefVO> queryListDynamic(ContractTemplateOrgRefQuery query) {
        JPAQuery<ContractTemplateOrgRefVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ContractTemplateOrgRefVO> queryPaging(ContractTemplateOrgRefQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<ContractTemplateOrgRefVO> jpaQuery = getJpaQueryWhere(query);
        List<ContractTemplateOrgRefVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ContractTemplateOrgRefVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ContractTemplateOrgRefDO save(ContractTemplateOrgRefDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ContractTemplateOrgRefDO> saveAll(List<ContractTemplateOrgRefDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ContractTemplateOrgRefPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 部门名称（冗余）
        if (payload.getName() != null) {
            update.set(qdo.name, payload.getName());
        }
        // 模版ID crm_contract_template.id
        if (payload.getTempId() != null) {
            update.set(qdo.tempId, payload.getTempId());
        }
        // 组织BuID prd_org_organization.id
        if (payload.getOrgId() != null) {
            update.set(qdo.orgId, payload.getOrgId());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 部门名称（冗余）
            if (nullFields.contains("name")) {
                update.setNull(qdo.name);
            }
            // 模版ID crm_contract_template.id
            if (nullFields.contains("tempId")) {
                update.setNull(qdo.tempId);
            }
            // 组织BuID prd_org_organization.id
            if (nullFields.contains("orgId")) {
                update.setNull(qdo.orgId);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

