package com.elitesland.tw.tw5crm.server.contract.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmCustomerService;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmCustomerSimpleVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.FileUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.RoleEnum;
import com.elitesland.tw.tw5crm.api.contract.payload.ContractBillingApplyPayload;
import com.elitesland.tw.tw5crm.api.contract.payload.ContractBillingDetailPayload;
import com.elitesland.tw.tw5crm.api.contract.payload.ContractInvoicePayload;
import com.elitesland.tw.tw5crm.api.contract.query.ContractBillingApplyQuery;
import com.elitesland.tw.tw5crm.api.contract.service.ContractBillingApplyService;
import com.elitesland.tw.tw5crm.api.contract.service.ContractBillingDetailService;
import com.elitesland.tw.tw5crm.api.contract.service.ContractInvoiceService;
import com.elitesland.tw.tw5crm.api.contract.vo.ContractBillingApplyVO;
import com.elitesland.tw.tw5crm.api.contract.vo.ContractBillingDetailVO;
import com.elitesland.tw.tw5crm.api.contract.vo.ContractInvoiceVO;
import com.elitesland.tw.tw5crm.server.common.constants.ContractBillingApplyStatusEnum;
import com.elitesland.tw.tw5crm.server.common.constants.GenerateSeqNumConstants;
import com.elitesland.tw.tw5crm.server.contract.convert.ContractBillingApplyConvert;
import com.elitesland.tw.tw5crm.server.contract.dao.ContractBillingApplyDAO;
import com.elitesland.tw.tw5crm.server.contract.entity.ContractBillingApplyDO;
import com.elitesland.tw.tw5crm.server.contract.repo.ContractBillingApplyRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.time.LocalDate;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;

/**
 * 合同管理-开票申请
 *
 * @author wangly
 * @date 2023-04-25
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class ContractBillingApplyServiceImpl extends BaseServiceImpl implements ContractBillingApplyService {

    private final ContractBillingApplyRepo contractBillingApplyRepo;
    private final ContractBillingApplyDAO contractBillingApplyDAO;
    private final ContractBillingDetailService contractBillingDetailService;
    private final ContractInvoiceService contractInvoiceService;
    private final CacheUtil cacheUtil;
    private final FileUtil fileUtil;
    private final CrmCustomerService crmCustomerService;

    @Override
    public PagingVO<ContractBillingApplyVO> queryPaging(ContractBillingApplyQuery query){
        // 数据权限
        dataPermissionFlag(query);
        return contractBillingApplyDAO.queryPaging(query);
    }

    @Override
    public List<ContractBillingApplyVO> queryListDynamic(ContractBillingApplyQuery query){
        // 数据权限
        dataPermissionFlag(query);
        return contractBillingApplyDAO.queryListDynamic(query);
    }

    @Override
    public ContractBillingApplyVO queryByKey(Long key) {
        ContractBillingApplyDO entity = contractBillingApplyRepo.findById(key).orElseGet(ContractBillingApplyDO::new);
        Assert.notNull(entity.getId(), "发票申请不存在");
        ContractBillingApplyVO vo = ContractBillingApplyConvert.INSTANCE.toVo(entity);
        vo.setFileDatas(fileUtil.getFileDatas(vo.getFileCodes()));
        return vo;
    }

    @Override
    public ContractBillingApplyVO queryBillingInfoById(Long key) {
        ContractBillingApplyVO contractBillingApplyVO = queryByKey(key);
        //查询开票明细
        contractBillingApplyVO.setContractBillingDetailVOList(contractBillingDetailService.queryListByBillingId(contractBillingApplyVO.getId()));
        //查询发票信息
        contractBillingApplyVO.setContractInvoiceVO(contractInvoiceService.queryInvoicetByBillingId(contractBillingApplyVO.getId()));
        return contractBillingApplyVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ContractBillingApplyVO insert(ContractBillingApplyPayload payload) {
        if(payload.getContractBillingDetailPayloadList()==null||payload.getContractBillingDetailPayloadList().size()==0){
            throw TwException.error("","开票明细不能为空");
        }
        payload.setBillingCode(getBillingCode());
        //保存开票信息
        ContractInvoicePayload contractInvoicePayload = payload.getContractInvoicePayload();
        //申请人
        payload.setApplyUserId(GlobalUtil.getLoginUserId());
        payload.setApplyUserName(GlobalUtil.getLoginUserName());
        if (payload.getCustomerId() != null) {
            //保存客户名称
            CrmCustomerSimpleVO customerVO = crmCustomerService.querySimpleByKey(payload.getCustomerId());
            Assert.notNull(customerVO, "客户不存在");
            payload.setCustomerName(customerVO.getCustomerName());
        }
        // 将发票信息放到申请表中
        payload.setBillingMainbody(contractInvoicePayload.getBillingMainbody());
        payload.setBillingStatus(ContractBillingApplyStatusEnum.BILLING.getCode());
        payload.setInvoiceTitle(contractInvoicePayload.getInvoiceTitle());
        ContractBillingApplyDO entityDo = ContractBillingApplyConvert.INSTANCE.toDo(payload);
        ContractBillingApplyVO contractBillingApplyVO = ContractBillingApplyConvert.INSTANCE.toVo(contractBillingApplyRepo.save(entityDo));
        List<ContractBillingDetailPayload> contractBillingDetailPayloadList = payload.getContractBillingDetailPayloadList();
        //保存开票明细
        List<ContractBillingDetailVO> contractBillingDetailVOList = contractBillingDetailService.saveAll(contractBillingDetailPayloadList,contractBillingApplyVO.getId(),contractBillingApplyVO.getBillingCode());
        //保存发票信息
        contractInvoicePayload.setBillingId(contractBillingApplyVO.getId());
        contractInvoicePayload.setBillingCode(contractBillingApplyVO.getBillingCode());
        contractInvoicePayload.setCurrCode(contractBillingApplyVO.getCurrCode());
        ContractInvoiceVO contractInvoiceVO = contractInvoiceService.insert(contractInvoicePayload);
        //将保存的信息返回
        contractBillingApplyVO.setContractBillingDetailVOList(contractBillingDetailVOList);
        contractBillingApplyVO.setContractInvoiceVO(contractInvoiceVO);
        return contractBillingApplyVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ContractBillingApplyVO update(ContractBillingApplyPayload payload) {
        ContractBillingApplyDO entity = contractBillingApplyRepo.findById(payload.getId()).orElseGet(ContractBillingApplyDO::new);
        Assert.notNull(entity.getId(), "发票申请不存在");
        ContractBillingApplyDO entityDo = ContractBillingApplyConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return ContractBillingApplyConvert.INSTANCE.toVo(contractBillingApplyRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            keys.stream().forEach(id -> {
                Optional<ContractBillingApplyDO> optional = contractBillingApplyRepo.findById(id);
                if (!optional.isEmpty()) {
                    ContractBillingApplyDO entity = optional.get();
                    entity.setDeleteFlag(1);
                    contractBillingApplyRepo.save(entity);
                }
             });
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void invoiceEntry(ContractBillingApplyPayload payload) {
        ContractBillingApplyDO entity = contractBillingApplyRepo.findById(payload.getId()).orElseGet(ContractBillingApplyDO::new);
        Assert.notNull(entity.getId(), "不存在");
        if(!ContractBillingApplyStatusEnum.BILLING.getCode().equals(entity.getBillingStatus())){
            throw TwException.error("","只有开票中的申请才可以录入发票,请重新选择");
        }
        entity.setInvoiceTaxNum(payload.getInvoiceTaxNum());
        //设置开票状态为已开票
        entity.setBillingStatus(ContractBillingApplyStatusEnum.INVOICED.getCode());
        entity.setInvoiceDate(LocalDate.now());
        contractBillingApplyRepo.save(entity);
        contractInvoiceService.updateInvoiceEntry(payload);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void cancleInvoice( List<String> billingList){
        //todo 如果id不存在 是否要做校验
        if (!billingList.isEmpty()) {
            billingList.stream().forEach(id -> {
                Optional<ContractBillingApplyDO> optional = contractBillingApplyRepo.findById(Long.valueOf(id));
                if (!optional.isEmpty()) {
                    ContractBillingApplyDO entity = optional.get();
                    //设置开票状态为已作废
                    entity.setBillingStatus(ContractBillingApplyStatusEnum.VOIDED.getCode());
                    contractBillingApplyRepo.save(entity);
                }
            });
        }
    }

    public String getBillingCode() {
        return generateSeqNum(GenerateSeqNumConstants.CRM_CONTRACT_BILLING_APPLY);
    }
    /**
     * 数据权限
     * 权限处理
     *
     * @param query
     */
    void dataPermissionFlag(ContractBillingApplyQuery query) {
        Boolean rolePermission = cacheUtil.hasSystemRolePermission(Arrays.asList(RoleEnum.SYS.getCode(), RoleEnum.CONTRACT_ADMIN.getCode()));
        query.setDataPermissionFlag(!rolePermission);
        if (!rolePermission) {
            //需要处理权限
            final Long loginUserId = GlobalUtil.getLoginUserId();
            query.setLoginUserId(loginUserId);
        }
    }

}
