package com.elitesland.tw.tw5crm.server.contract.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.FileUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.RoleEnum;
import com.elitesland.tw.tw5crm.api.contract.payload.ContractCollectionPlanPayload;
import com.elitesland.tw.tw5crm.api.contract.query.ContractCollectionPlanQuery;
import com.elitesland.tw.tw5crm.api.contract.service.ContractBillingDetailService;
import com.elitesland.tw.tw5crm.api.contract.service.ContractCollectionPlanService;
import com.elitesland.tw.tw5crm.api.contract.vo.ContractCollectionPlanSimpleVO;
import com.elitesland.tw.tw5crm.api.contract.vo.ContractCollectionPlanVO;
import com.elitesland.tw.tw5crm.server.common.constants.ContractBillingApplyStatusEnum;
import com.elitesland.tw.tw5crm.server.contract.convert.ContractCollectionPlanConvert;
import com.elitesland.tw.tw5crm.server.contract.dao.ContractCollectionPlanDAO;
import com.elitesland.tw.tw5crm.server.contract.entity.ContractCollectionPlanDO;
import com.elitesland.tw.tw5crm.server.contract.repo.ContractCollectionPlanRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * 合同收款计划
 *
 * @author duwh
 * @date 2023-04-24
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class ContractCollectionPlanServiceImpl extends BaseServiceImpl implements ContractCollectionPlanService {

    private final ContractCollectionPlanRepo contractCollectionPlanRepo;
    private final ContractCollectionPlanDAO contractCollectionPlanDAO;
    private final CacheUtil cacheUtil;
    private final FileUtil fileUtil;
    private final ContractBillingDetailService contractBillingDetailService;

    @Override
    public PagingVO<ContractCollectionPlanVO> queryPaging(ContractCollectionPlanQuery query) {
        // 数据权限
        dataPermissionFlag(query);

        final PagingVO<ContractCollectionPlanVO> pagingVO = contractCollectionPlanDAO.queryPagingJoin(query);
        final List<ContractCollectionPlanVO> records = pagingVO.getRecords();
        if (!CollectionUtils.isEmpty(records)) {
            records.forEach(vo -> {
                // 翻译
                translation(vo);
            });
        }

        return pagingVO;
    }

    /**
     * 检查参数
     *
     * @param query 查询
     */
    private void checkParam(ContractCollectionPlanQuery query) {
        if (ObjectUtils.isEmpty(query.getContractId())) {
            throw TwException.error("", "contractId不能为空");
        }
    }

    private void dataPermissionFlag(ContractCollectionPlanQuery query) {
        Boolean rolePermission = cacheUtil.hasSystemRolePermission(Arrays.asList(RoleEnum.SYS.getCode(), RoleEnum.CONTRACT_ADMIN.getCode()));
        query.setDataPermissionFlag(!rolePermission);
        if (!rolePermission) {
            //需要处理权限
            final Long loginUserId = GlobalUtil.getLoginUserId();
            query.setLoginUserId(loginUserId);
            final List<Long> managerOrgIdsByUserId = cacheUtil.getManagerOrgIdsByUserId(loginUserId);
            query.setSignOrgIdList(managerOrgIdsByUserId);
        }
    }

    /**
     * 翻译
     *
     * @param vo 签证官
     */
    private void translation(ContractCollectionPlanVO vo) {
        vo.setFileDatas(fileUtil.getFileDatas(vo.getFileCodes()));
        //开票中金额
        BigDecimal billingTotalAmount = contractBillingDetailService.getAmoutByPlanIdAndBillingStatus(vo.getId(), ContractBillingApplyStatusEnum.BILLING.getCode());
        //已开票金额
        BigDecimal invoicedTotalAmount = contractBillingDetailService.getAmoutByPlanIdAndBillingStatus(vo.getId(), ContractBillingApplyStatusEnum.INVOICED.getCode());
        vo.setInvoicingAmount(billingTotalAmount);
        vo.setInvoicedAmount(invoicedTotalAmount);
        //已收款金额 已开票+开票中
        vo.setReceivedAmount(billingTotalAmount.add(invoicedTotalAmount));
        //未收款金额 当前收款金额- 已收款金额
        vo.setUnCollectionAmount((vo.getCollectionAmount()==null?BigDecimal.ZERO:vo.getCollectionAmount()).subtract(vo.getReceivedAmount()));
        //收款比例
        if(vo.getCollectionAmount()!=null&&vo.getUnCollectionAmount()!=null&&!vo.getCollectionAmount().equals(BigDecimal.ZERO)&&!vo.getUnCollectionAmount().equals(BigDecimal.ZERO)){
            vo.setCollectionRatio(vo.getUnCollectionAmount().divide(vo.getCollectionAmount()).setScale(2, RoundingMode.HALF_UP));
        }

    }

    @Override
    public List<ContractCollectionPlanVO> queryListDynamic(ContractCollectionPlanQuery query) {
        // 参数校验
        checkParam(query);
        return contractCollectionPlanDAO.queryListDynamic(query);
    }

    @Override
    public List<ContractCollectionPlanSimpleVO> queryListDynamicSimple(ContractCollectionPlanQuery query) {
        // 参数校验
        checkParam(query);
        List<ContractCollectionPlanVO> contractCollectionPlanVOS = contractCollectionPlanDAO.queryListDynamic(query);
        List<ContractCollectionPlanSimpleVO> resultList = contractCollectionPlanVOS.stream().map(contractCollectionPlanVO -> {
            ContractCollectionPlanSimpleVO contractCollectionPlanSimpleVO = ContractCollectionPlanConvert.INSTANCE.toSimpleVo(contractCollectionPlanVO);
            return contractCollectionPlanSimpleVO;
        }).collect(Collectors.toList());
        return resultList;
    }

    @Override
    public ContractCollectionPlanVO queryByKey(Long key) {
        ContractCollectionPlanDO entity = contractCollectionPlanRepo.findById(key).orElseGet(ContractCollectionPlanDO::new);
        Assert.notNull(entity.getId(), "收款计划不存在");
        ContractCollectionPlanVO vo = ContractCollectionPlanConvert.INSTANCE.toVo(entity);
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ContractCollectionPlanVO insert(ContractCollectionPlanPayload payload) {
        // 参数校验
        checkPayload(payload);
        ContractCollectionPlanDO entityDo = ContractCollectionPlanConvert.INSTANCE.toDo(payload);
        return ContractCollectionPlanConvert.INSTANCE.toVo(contractCollectionPlanRepo.save(entityDo));
    }

    private void checkPayload(ContractCollectionPlanPayload payload) {
        if (null == payload.getContractId()) {
            throw TwException.error("", "contractId不能为空");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ContractCollectionPlanVO update(ContractCollectionPlanPayload payload) {
        ContractCollectionPlanDO entity = contractCollectionPlanRepo.findById(payload.getId()).orElseGet(ContractCollectionPlanDO::new);
        Assert.notNull(entity.getId(), "收款计划不存在");
        ContractCollectionPlanDO entityDo = ContractCollectionPlanConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return ContractCollectionPlanConvert.INSTANCE.toVo(contractCollectionPlanRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            keys.stream().forEach(id -> {
                Optional<ContractCollectionPlanDO> optional = contractCollectionPlanRepo.findById(id);
                if (!optional.isEmpty()) {
                    ContractCollectionPlanDO entity = optional.get();
                    entity.setDeleteFlag(1);
                    contractCollectionPlanRepo.save(entity);
                }
            });
        }
    }

    @Override
    public List<ContractCollectionPlanSimpleVO> queryListByContractId(List<Long> planIdList) {
        return contractCollectionPlanDAO.queryListByContractId(planIdList);
    }

}
