package com.elitesland.tw.tw5crm.server.contract.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitesland.tw.tw5.api.common.TwWorkFlowCommonVO;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmCustomerService;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmOpportunityService;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmCustomerSimpleVO;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmOpportunityVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.service.TransactionUtilService;
import com.elitesland.tw.tw5.server.common.workFlow.ProcDefKey;
import com.elitesland.tw.tw5.server.common.workFlow.WorkflowUtil;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.FileUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.RoleEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.WorkFlowStatusEnum;
import com.elitesland.tw.tw5crm.api.contract.payload.ContractCollectionPlanPayload;
import com.elitesland.tw.tw5crm.api.contract.payload.ContractCountersidePayload;
import com.elitesland.tw.tw5crm.api.contract.payload.ContractPayload;
import com.elitesland.tw.tw5crm.api.contract.payload.ContractProductRefPayload;
import com.elitesland.tw.tw5crm.api.contract.query.ContractCollectionPlanQuery;
import com.elitesland.tw.tw5crm.api.contract.query.ContractCountersideQuery;
import com.elitesland.tw.tw5crm.api.contract.query.ContractProductRefQuery;
import com.elitesland.tw.tw5crm.api.contract.query.ContractQuery;
import com.elitesland.tw.tw5crm.api.contract.service.ContractCollectionPlanService;
import com.elitesland.tw.tw5crm.api.contract.service.ContractCountersideService;
import com.elitesland.tw.tw5crm.api.contract.service.ContractProductRefService;
import com.elitesland.tw.tw5crm.api.contract.service.ContractService;
import com.elitesland.tw.tw5crm.api.contract.vo.*;
import com.elitesland.tw.tw5crm.server.common.constants.ContractCollectionPlanPlanStatus;
import com.elitesland.tw.tw5crm.server.common.constants.ContractCollectionPlanStatus;
import com.elitesland.tw.tw5crm.server.common.constants.ContractStatus;
import com.elitesland.tw.tw5crm.server.common.constants.GenerateSeqNumConstants;
import com.elitesland.tw.tw5crm.server.contract.convert.ContractCollectionPlanConvert;
import com.elitesland.tw.tw5crm.server.contract.convert.ContractConvert;
import com.elitesland.tw.tw5crm.server.contract.convert.ContractCountersideConvert;
import com.elitesland.tw.tw5crm.server.contract.convert.ContractProductRefConvert;
import com.elitesland.tw.tw5crm.server.contract.dao.ContractCollectionPlanDAO;
import com.elitesland.tw.tw5crm.server.contract.dao.ContractDAO;
import com.elitesland.tw.tw5crm.server.contract.entity.ContractCollectionPlanDO;
import com.elitesland.tw.tw5crm.server.contract.entity.ContractCountersideDO;
import com.elitesland.tw.tw5crm.server.contract.entity.ContractDO;
import com.elitesland.tw.tw5crm.server.contract.entity.ContractProductRefDO;
import com.elitesland.tw.tw5crm.server.contract.repo.ContractCollectionPlanRepo;
import com.elitesland.tw.tw5crm.server.contract.repo.ContractCountersideRepo;
import com.elitesland.tw.tw5crm.server.contract.repo.ContractProductRefRepo;
import com.elitesland.tw.tw5crm.server.contract.repo.ContractRepo;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.TaskInfo;
import com.elitesland.workflow.WorkflowResult;
import com.elitesland.workflow.WorkflowService;
import com.elitesland.workflow.payload.CurrentTaskInfosPayload;
import com.elitesland.workflow.payload.StartProcessPayload;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.time.LocalDateTime;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 合同管理
 *
 * @author kola
 * @date 2023-04-24
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class ContractServiceImpl extends BaseServiceImpl implements ContractService {

    private final ContractRepo contractRepo;
    private final ContractDAO contractDAO;
    private final CacheUtil cacheUtil;
    private final CrmCustomerService crmCustomerService;
    private final CrmOpportunityService crmOpportunityService;
    private final ContractCountersideRepo contractCountersideRepo;
    private final ContractCollectionPlanRepo contractCollectionPlanRepo;
    private final ContractProductRefRepo contractProductRefRepo;
    private final ContractProductRefService contractProductRefService;
    private final ContractCollectionPlanService contractCollectionPlanService;
    private final ContractCountersideService contractCountersideService;
    @Value("${tw5.workflow.enabled:false}")
    private Boolean workflow_enabled;
    private final WorkflowUtil workflowUtil;
    private final WorkflowService workflowService;
    private final TransactionUtilService transactionUtilService;
    private final ContractCollectionPlanDAO contractCollectionPlanDAO;
    private final FileUtil fileUtil;


    @Override
    public PagingVO<ContractVO> queryPaging(ContractQuery query){
        // 数据权限
        dataPermissionFlag(query);
//        query.setParentIdIsNull(11L);
        // 先查第一层
        PagingVO<ContractVO> pagingVO = contractDAO.queryPagingJoin(query);
        List<ContractVO> records = pagingVO.getRecords();
        // 如果有上级，就不展示孩子
//        List<ContractVO> recordsNew = new ArrayList<>();
        // 工作流 审批信息回显
        getTaskInfo(records);
        // 遍历查第二层
        records.forEach(record->{

//            // 如果查询到有父类则过滤掉
//            long count = records.stream().filter(contractVO -> contractVO.getId().equals(record.getParentId())).count();
//            if(count>0){
//                return;
//            }

            ContractQuery queryChild = new ContractQuery();
            BeanUtils.copyProperties(query,queryChild);
            queryChild.setParentId(record.getId());
            List<ContractVO> childrenList = this.queryListDynamic(queryChild);
            // 工作流 审批信息回显
            getTaskInfo(childrenList);
            record.setChildrenList(childrenList);
//            recordsNew.add(record);
        });
        pagingVO.setRecords(records);
//        pagingVO.setRecords(recordsNew);

        return pagingVO;
    }

    /**
     * 获取工作流审批信息
     *
     * @param voList 签证官列表
     */
    private void getTaskInfo(List<ContractVO> voList) {
        // 查询当前处理的任务名称
        if (!CollectionUtils.isEmpty(voList)) {
            // 1、组装成map，key为流程实例Id,value为对象
            Map<String, TwWorkFlowCommonVO> map = voList
                .stream()
                .filter(vo -> org.apache.commons.lang3.StringUtils.isNoneBlank(vo.getProcInstId()))
                .collect(Collectors.toMap(TwWorkFlowCommonVO::getProcInstId, Function.identity(),(entity1, entity2) -> entity1));
            if (null != map && map.isEmpty() == false) {
                // 2、查询当前处理的任务名称
                final Set<String> procInstIds = map.keySet();
                HashSet hashSet = new HashSet(procInstIds);
                // 3、设置任务名称到业务对象上
                CurrentTaskInfosPayload currentTaskInfosPayload = new CurrentTaskInfosPayload();
                currentTaskInfosPayload.setProcInstIds(hashSet);
                // 调用工作流程方法查询任务信息
                final WorkflowResult<HashMap<String, TaskInfo>> workflowResult = workflowService.currentTaskInfos(currentTaskInfosPayload);
                if (workflowResult.isSuccess()) {
                    HashMap<String, TaskInfo> currentTaskNames = workflowResult.getData();
                    // 3、设置任务名称到业务对象上
                    currentTaskNames.forEach((key, value) -> {
                        TwWorkFlowCommonVO vo = map.get(key);
                        vo.setTaskInfo(value);
                    });
                }
            }
        }
    }

    private void dataPermissionFlag(ContractQuery query) {
        Boolean rolePermission = cacheUtil.hasSystemRolePermission(Arrays.asList(RoleEnum.SYS.getCode(), RoleEnum.CONTRACT_ADMIN.getCode()));
        query.setDataPermissionFlag(!rolePermission);
        if (!rolePermission) {
            //需要处理权限
            final Long loginUserId = GlobalUtil.getLoginUserId();
            query.setLoginUserId(loginUserId);
            final List<Long> managerOrgIdsByUserId = cacheUtil.getManagerOrgIdsByUserId(loginUserId);
            query.setSignOrgIdList(managerOrgIdsByUserId);
        }
    }

    @Override
    public List<ContractVO> queryListDynamic(ContractQuery query){
        // 数据权限
        dataPermissionFlag(query);
        return contractDAO.queryListDynamic(query);
    }

    /**
     * 简单列表
     *
     * @param query 查询
     * @return {@link List}<{@link ContractSimpleVO}>
     */
    @Override
    public List<ContractSimpleVO> listSimple(ContractQuery query) {
        // 数据权限
        dataPermissionFlag(query);
        List<ContractSimpleVO> listResult = contractDAO.queryListSimple(query);
        listResult.forEach(contractReceiveVO -> {
            ContractQuery queryChild = new ContractQuery();
            BeanUtils.copyProperties(query,queryChild);
            queryChild.setParentId(contractReceiveVO.getId());
            List<ContractSimpleVO> children = this.listSimpleParentId(queryChild);
            contractReceiveVO.setChildren(children);
        });
        return listResult;
    }

    @Override
    public ContractVO queryByKey(Long key) {
        ContractDO contractDO = contractRepo.findById(key).orElseGet(ContractDO::new);
        Assert.notNull(contractDO.getId(), "合同不存在");
        ContractVO contractVO = ContractConvert.INSTANCE.toVo(contractDO);

        translation(contractVO);

        //获取相对方信息
        getContractCountersideVOList(contractVO);
        //获取收款计划信息
        getContractCollectionPlanVOList(contractVO);
        //获取产品信息
        getContractProductRefVOList(contractVO);
        // 流程信息回显
        final WorkflowResult<TaskInfo> workflowResult = workflowService.currentTaskInfo(contractVO.getProcInstId());
        if (workflowResult.isSuccess()) {
            TaskInfo taskInfo = workflowResult.getData();
            contractVO.setTaskInfo(taskInfo);
        }
        return contractVO;
    }

    private void translation(ContractVO contractVO) {
        contractVO.setContractFile1Data(fileUtil.getFileDatas(contractVO.getContractFile1()));
        contractVO.setContractFile2Data(fileUtil.getFileDatas(contractVO.getContractFile2()));
        contractVO.setContractFile4Data(fileUtil.getFileDatas(contractVO.getContractFile4()));
        contractVO.setContractFile5Data(fileUtil.getFileDatas(contractVO.getContractFile5()));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ContractVO insert(ContractPayload payload) {
        //处理主表数据
        dataProcess(payload);
        ContractDO contractDO = ContractConvert.INSTANCE.toDo(payload);
        ContractDO save = contractRepo.save(contractDO);
        //保存 相对方信息 产品信息 收款计划
        saveOtherData(payload,save);

        //数据提交,执行工作流
        if (payload.getSubmit()) {
            payload.setId(save.getId());
            submitProc(payload);
        }
        return ContractConvert.INSTANCE.toVo(save);
    }

    /**
     * 提交流程
     *
     * @param payload 有效载荷
     */
    void submitProc(ContractPayload payload) {
        ProcessInfo processInfo = new ProcessInfo();
        String status = WorkFlowStatusEnum.APPROVED_WORK.getCode();
        if (workflow_enabled) {
            status = WorkFlowStatusEnum.APPROVING_WORK.getCode();
            //Long orgManageUserId = cacheUtil.getOrgManageUserId(payload.getOrgId());
            HashMap<String, Object> batchMap = new HashMap<>();
            //batchMap.put("Activity_1be86d6", Lists.newArrayList(orgManageUserId.toString()));
            //发起流程审批
            processInfo = workflowUtil.startProcess(StartProcessPayload.of(
                ProcDefKey.CONTRACT.name(),
                "" + payload.getContractName() + "-激活审批流程",
                payload.getId() + "",
                batchMap)
            );
        }
        ContractPayload payload0 = new ContractPayload();
        payload0.setId(payload.getId());
        payload0.setProcInstId(processInfo.getProcInstId());
        payload0.setProcInstStatus(processInfo.getProcInstStatus());
        payload0.setSubmitTime(LocalDateTime.now());
        payload0.setContractStatus(status);
        //开启事务执行修改，主要是修改审批状态
        transactionUtilService.executeWithRunnable(() -> {
            contractDAO.updateWorkFlow(payload0);
        });

        // 未开启流程审批的情况
        if (status.equals(WorkFlowStatusEnum.APPROVED_WORK.getCode())) {
            // 收款计划 计划状态变更为激活
            contractCollectionPlanDAO.updateStatusByContractId(ContractCollectionPlanStatus.ACTIVE.getCode(), payload.getId());
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ContractVO update(ContractPayload payload) {
        if (payload.getId()==null){
            throw TwException.error("1", "合同id不存在");
        }
        ContractDO entity = contractRepo.findById(payload.getId()).orElseGet(ContractDO::new);
        dataProcess(payload);
        ContractDO entityDo = ContractConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        ContractDO contractDO = contractRepo.save(entity);
        //保存 相对方信息 产品信息 收款计划
        saveOtherData(payload,contractDO);

        //数据提交,执行工作流
        if (payload.getSubmit()) {
            payload.setId(contractDO.getId());
            submitProc(payload);
        }
        return ContractConvert.INSTANCE.toVo(contractDO);
    }

    @Override
    public ContractVO updateFiled(ContractPayload payload) {
        if (payload.getId()==null){
            throw TwException.error("1", "合同id不存在");
        }
        ContractDO entity = contractRepo.findById(payload.getId()).orElseGet(ContractDO::new);
        ContractPayload update = new ContractPayload();
        update.setId(payload.getId());
        update.setContractFile1(payload.getContractFile1());
        ContractDO entityDo = ContractConvert.INSTANCE.toDo(payload);
        entityDo.setContractStatus(ContractStatus.FILED.getCode());
        entityDo.setContractFile1(entityDo.getContractFile1());
        entity.copy(entityDo);
        ContractDO contractDO = contractRepo.save(entity);
        return ContractConvert.INSTANCE.toVo(contractDO);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            keys.stream().forEach(id -> {
                Optional<ContractDO> optional = contractRepo.findById(id);
                if (!optional.isEmpty()) {
                    ContractDO entity = optional.get();
                    entity.setDeleteFlag(1);
                    contractRepo.save(entity);
                }
             });
        }
    }

    @Override
    public List<ContractVO> queryByParentKey(Long id) {
        return contractDAO.queryByParentKey(id);
    }

    @Override
    public List<ContractSimpleVO> listSimpleParentId(ContractQuery query) {
        return contractDAO.listSimpleParentId(query);
    }


    /**
     * @param payload
     * 处理合同主表数据
     */
    private void dataProcess(ContractPayload payload) {
        // 新建
        if (null == payload.getId()) {
            if (ObjectUtils.isEmpty(payload.getContractStatus())) {
                payload.setContractStatus(ContractStatus.CREATE.getCode());
            }
            // 发号器
            String code = generateSeqNum(GenerateSeqNumConstants.CRM_CONTRACT);
            payload.setContractCode(code);
        }
        if (payload.getCustomerId()!=null){
            //保存客户名称
            CrmCustomerSimpleVO customerVO = crmCustomerService.querySimpleByKey(payload.getCustomerId());
            payload.setCustomerName(customerVO.getCustomerName());
        }
        if (payload.getSuppliersId()!=null){
            //保存供应商名称
            CrmCustomerSimpleVO suppliersVO = crmCustomerService.querySimpleByKey(payload.getSuppliersId());
            payload.setSuppliersName(suppliersVO.getCustomerName());
        }
       if (payload.getRelateBusinessId()!=null){
           //保存商机名称
           CrmOpportunityVO crmOpportunityVO = crmOpportunityService.queryByKeySimple(payload.getRelateBusinessId());
           payload.setRelateBusinessName(crmOpportunityVO.getProjectName());
       }
       if (payload.getRelateContractId()!=null){
           //保存关联合同名称
        ContractVO contractVO = contractDAO.queryByKey(payload.getRelateContractId());
        payload.setRelateContractName(contractVO.getContractName());
       }
        //保存签单部门名称
        String signOrgName = cacheUtil.getOrgName(payload.getSignOrgId());
        payload.setSignOrgName(signOrgName);
        //保存采购部门名称
        String purchaseOrgName = cacheUtil.getOrgName(payload.getPurchaseOrgId());
        payload.setPurchaseOrgName(purchaseOrgName);
        //保存交付部门主键
        String payOrgName = cacheUtil.getOrgName(payload.getPayOrgId());
        payload.setPayOrgName(payOrgName);
    }

    /**
     * @param contractPayload
     * @param contractDO
     * 保存其他信息
     */
    void saveOtherData(ContractPayload contractPayload, ContractDO contractDO){
        Long contractId = contractDO.getId();
        //保存相对方信息
        List<Long> deleteCountersideIdList = contractPayload.getDeleteCountersideIdList();
        if (!CollectionUtils.isEmpty(deleteCountersideIdList)){
            contractCountersideService.deleteSoft(deleteCountersideIdList);
        }
        List<ContractCountersidePayload> countersideLists = contractPayload.getContractCountersideList();
        if (!ObjectUtils.isEmpty(countersideLists)){
            List<ContractCountersideDO> contractCountersideDOS=new ArrayList<>();
            countersideLists.forEach(countersideList->{
                ContractCountersideDO contractCountersideDO = ContractCountersideConvert.INSTANCE.toDo(countersideList);
                contractCountersideDO.setContractId(contractId);
                contractCountersideDOS.add(contractCountersideDO);
            });
            contractCountersideRepo.saveAll(contractCountersideDOS);
        }
        //保存收款计划信息
        List<Long> deleteCollectionIdList = contractPayload.getDeleteCollectionIdList();
        if (!CollectionUtils.isEmpty(deleteCollectionIdList)) {
            contractCollectionPlanService.deleteSoft(deleteCollectionIdList);
        }
        List<ContractCollectionPlanPayload> collectionPlanLists = contractPayload.getCollectionPlanList();
        if (!ObjectUtils.isEmpty(collectionPlanLists)) {
            List<ContractCollectionPlanDO> contractCollectionPlanDOS = new ArrayList<>();
            collectionPlanLists.forEach(collectionPlanList -> {
                ContractCollectionPlanDO contractCollectionPlanDO = ContractCollectionPlanConvert.INSTANCE.toDo(collectionPlanList);
                contractCollectionPlanDO.setContractId(contractId);
                // 1.计划状态根据合同的审批状态更新，标产中默认合同为新建状态时，合同下对应的收款计划为新建；
                //
                // 激活状态时，合同下对应的收款计划为激活；合同后续状态不影响收款计划状态，例如暂挂也不影响
                //收款状态
                contractCollectionPlanDO.setStatus(ContractCollectionPlanPlanStatus.NO.getCode());
                //收款计划状态
                contractCollectionPlanDO.setPlanStatus(ContractCollectionPlanStatus.CREATE.getCode());
                contractCollectionPlanDOS.add(contractCollectionPlanDO);
            });
            contractCollectionPlanRepo.saveAll(contractCollectionPlanDOS);
        }
        //保存产品信息
        List<Long> deleteProductIdList = contractPayload.getDeleteProductIdList();
        if (!CollectionUtils.isEmpty(deleteProductIdList)){
            contractProductRefService.deleteSoft(deleteProductIdList);
        }
        List<ContractProductRefPayload> productRefPayloadLists = contractPayload.getProductRefPayloadList();
        if (!CollectionUtils.isEmpty(productRefPayloadLists)) {
            List<ContractProductRefDO> contractProductRefDOS = new ArrayList<>();
            productRefPayloadLists.forEach(productRefPayloadList -> {
                ContractProductRefDO contractProductRefDO = ContractProductRefConvert.INSTANCE.toDo(productRefPayloadList);
                contractProductRefDO.setContractId(contractId);
                contractProductRefDOS.add(contractProductRefDO);
            });
            contractProductRefRepo.saveAll(contractProductRefDOS);
        }
    }

    /**
     * 相对方信息
     * @param contractVO
     */
    void getContractCountersideVOList(ContractVO contractVO){
        ContractCountersideQuery query=new ContractCountersideQuery();
        query.setContractId(contractVO.getId());
        List<ContractCountersideVO> contractCountersideVOS = contractCountersideService.queryListDynamic(query);
        contractVO.setContractCountersideVOList(contractCountersideVOS);
    }

    /**
     * 收款计划
     * @param contractVO
     */
    void getContractCollectionPlanVOList(ContractVO contractVO){
        ContractCollectionPlanQuery query=new ContractCollectionPlanQuery();
        query.setContractId(contractVO.getId());
        List<ContractCollectionPlanVO> contractCollectionPlanVOS = contractCollectionPlanService.queryListDynamic(query);
        contractVO.setContractCollectionPlanVOList(contractCollectionPlanVOS);
    }

    /**
     * 产品信息
     * @param contractVO
     */
    void getContractProductRefVOList(ContractVO contractVO){
        ContractProductRefQuery query = new ContractProductRefQuery();
        query.setContractId(contractVO.getId());
        List<ContractProductRefVO> contractProductRefVOS = contractProductRefService.queryListDynamic(query);
        contractVO.setContractProductRefVOList(contractProductRefVOS);
    }
}
