package com.elitesland.tw.tw5crm.server.contract.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5crm.api.contract.payload.ContractSignPayload;
import com.elitesland.tw.tw5crm.api.contract.query.ContractSignQuery;
import com.elitesland.tw.tw5crm.api.contract.service.ContractSignService;
import com.elitesland.tw.tw5crm.api.contract.vo.ContractSignVO;
import com.elitesland.tw.tw5crm.server.common.constants.ContractStatus;
import com.elitesland.tw.tw5crm.server.contract.convert.ContractSignConvert;
import com.elitesland.tw.tw5crm.server.contract.dao.ContractSignDAO;
import com.elitesland.tw.tw5crm.server.contract.entity.ContractDO;
import com.elitesland.tw.tw5crm.server.contract.entity.ContractSignDO;
import com.elitesland.tw.tw5crm.server.contract.repo.ContractRepo;
import com.elitesland.tw.tw5crm.server.contract.repo.ContractSignRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.List;
import java.util.Optional;

/**
 * 合同签订
 *
 * @author zach.zhang
 * @date 2023-04-26
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class ContractSignServiceImpl extends BaseServiceImpl implements ContractSignService {

    private final ContractSignRepo contractSignRepo;
    private final ContractSignDAO contractSignDAO;

    private final ContractRepo contractRepo;

    @Override
    public PagingVO<ContractSignVO> queryPaging(ContractSignQuery query){
        // 参数校验
        checkQuery(query);
        return contractSignDAO.queryPaging(query);
    }

    @Override
    public List<ContractSignVO> queryListDynamic(ContractSignQuery query){
        // 参数校验
        checkQuery(query);
        return contractSignDAO.queryListDynamic(query);
    }

    private void checkQuery(ContractSignQuery query) {
        if (null == query.getContractId()) {
            throw TwException.error("", "contractId不能为空");
        }
    }

    @Override
    public ContractSignVO queryByKey(Long key) {
        ContractSignDO entity = contractSignRepo.findById(key).orElseGet(ContractSignDO::new);
        Assert.notNull(entity.getId(), "不存在");
        ContractSignVO vo = ContractSignConvert.INSTANCE.toVo(entity);
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ContractSignVO insert(ContractSignPayload payload) {
        if (null == payload.getContractId()) {
            throw TwException.error("", "contractId不能为空");
        }
        ContractDO contractDO = contractRepo.findById(payload.getContractId()).orElseGet(ContractDO::new);
        if(!contractDO.getContractStatus().equals(ContractStatus.ACTIVE.getCode())){
            throw TwException.error("","合同状态是激活的支持签订");
        }
        payload.setContractName(contractDO.getContractName());
        payload.setCustomerId(contractDO.getCustomerId());
        payload.setCustomerName(contractDO.getCustomerName());
        payload.setSigncompanyId(contractDO.getSignCompanyId());
        payload.setSigncompanyName(contractDO.getSignCompanyName());
        payload.setCurrency(contractDO.getCurrency());
        payload.setContractAmount(contractDO.getContractAmount());
        ContractSignDO entityDo = ContractSignConvert.INSTANCE.toDo(payload);
        ContractSignDO save = contractSignRepo.save(entityDo);

        // 修改合同状态
        contractDO.setContractStatus(ContractStatus.TOFILED.getCode());
        contractRepo.save(contractDO);
        return ContractSignConvert.INSTANCE.toVo(save);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ContractSignVO update(ContractSignPayload payload) {
        ContractSignDO entity = contractSignRepo.findById(payload.getId()).orElseGet(ContractSignDO::new);
        Assert.notNull(entity.getId(), "不存在");
        ContractSignDO entityDo = ContractSignConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return ContractSignConvert.INSTANCE.toVo(contractSignRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            keys.stream().forEach(id -> {
                Optional<ContractSignDO> optional = contractSignRepo.findById(id);
                if (!optional.isEmpty()) {
                    ContractSignDO entity = optional.get();
                    entity.setDeleteFlag(1);
                    contractSignRepo.save(entity);
                }
             });
        }
    }

}
