package com.elitesland.tw.tw5crm.server.contract.service;

import com.alibaba.fastjson.JSON;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitesland.tw.tw5.api.common.TwWorkFlowCommonVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.service.TransactionUtilService;
import com.elitesland.tw.tw5.server.common.workFlow.ProcDefKey;
import com.elitesland.tw.tw5.server.common.workFlow.WorkflowUtil;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.FileUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.RoleEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.WorkFlowStatusEnum;
import com.elitesland.tw.tw5.server.udc.UdcUtil;
import com.elitesland.tw.tw5crm.api.common.change.payload.ComBusinessChangePayload;
import com.elitesland.tw.tw5crm.api.common.change.service.ComBusinessChangeService;
import com.elitesland.tw.tw5crm.api.common.change.vo.ComBusinessChangeVO;
import com.elitesland.tw.tw5crm.api.contract.payload.ContractTemplateOrgRefPayload;
import com.elitesland.tw.tw5crm.api.contract.payload.ContractTemplatePayload;
import com.elitesland.tw.tw5crm.api.contract.query.ContractTemplateQuery;
import com.elitesland.tw.tw5crm.api.contract.service.ContractTemplateOrgRefService;
import com.elitesland.tw.tw5crm.api.contract.service.ContractTemplateService;
import com.elitesland.tw.tw5crm.api.contract.vo.ContractTemplateVO;
import com.elitesland.tw.tw5crm.server.common.change.changeTypeEnum.ChangeTypeEnum;
import com.elitesland.tw.tw5crm.server.common.change.dao.ComBusinessChangeDAO;
import com.elitesland.tw.tw5crm.server.common.constants.ContractTemplateStatusEnum;
import com.elitesland.tw.tw5crm.server.common.constants.GenerateSeqNumConstants;
import com.elitesland.tw.tw5crm.server.contract.convert.ContractTemplateConvert;
import com.elitesland.tw.tw5crm.server.contract.dao.ContractTemplateDAO;
import com.elitesland.tw.tw5crm.server.contract.entity.ContractTemplateDO;
import com.elitesland.tw.tw5crm.server.contract.repo.ContractTemplateRepo;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.TaskInfo;
import com.elitesland.workflow.WorkflowResult;
import com.elitesland.workflow.WorkflowService;
import com.elitesland.workflow.payload.CurrentTaskInfosPayload;
import com.elitesland.workflow.payload.StartProcessPayload;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 合同模版
 *
 * @author duwh
 * @date 2023-04-20
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class ContractTemplateServiceImpl extends BaseServiceImpl implements ContractTemplateService {

    private final ContractTemplateRepo contractTemplateRepo;
    private final ContractTemplateDAO contractTemplateDAO;
    private final CacheUtil cacheUtil;
    public final static String ALL = "ALL";
    private final ContractTemplateOrgRefService contractTemplateOrgRefService;
    private final FileUtil fileUtil;
    @Value("${tw5.workflow.enabled:false}")
    private Boolean workflow_enabled;
    private final WorkflowUtil workflowUtil;
    private final UdcUtil udcUtil;
    private final TransactionUtilService transactionUtilService;
    private final ComBusinessChangeService businessChangeService;
    private final ComBusinessChangeDAO businessChangeDao;
    private final WorkflowService workflowService;

    @Override
    public PagingVO<ContractTemplateVO> queryPaging(ContractTemplateQuery query) {
        // 权限处理
        dataPermissionFlag(query);
        final PagingVO<ContractTemplateVO> contractTemplateVOPagingVO = contractTemplateDAO.queryPaging(query);
        List<ContractTemplateVO> records = contractTemplateVOPagingVO.getRecords();
        // 工作流 审批信息回显
        getTaskInfo(records);
        return contractTemplateVOPagingVO;
    }

    @Override
    public List<ContractTemplateVO> queryListDynamic(ContractTemplateQuery query) {
        // 权限处理
        dataPermissionFlag(query);
        final List<ContractTemplateVO> records = contractTemplateDAO.queryListDynamic(query);
        // 工作流 审批信息回显
//        getTaskInfo(records);
        return records;
    }

    /**
     * 获取工作流审批信息
     *
     * @param voList 签证官列表
     */
    private void getTaskInfo(List<ContractTemplateVO> voList) {
        // 查询当前处理的任务名称
        if (!CollectionUtils.isEmpty(voList)) {
            // 1、组装成map，key为流程实例Id,value为对象
            Map<String, TwWorkFlowCommonVO> map = voList
                .stream()
                .filter(vo -> org.apache.commons.lang3.StringUtils.isNoneBlank(vo.getProcInstId()))
                .collect(Collectors.toMap(TwWorkFlowCommonVO::getProcInstId, Function.identity()));
            if (null != map && map.isEmpty() == false) {
                // 2、查询当前处理的任务名称
                final Set<String> procInstIds = map.keySet();
                HashSet hashSet = new HashSet(procInstIds);
                // 3、设置任务名称到业务对象上
                CurrentTaskInfosPayload currentTaskInfosPayload = new CurrentTaskInfosPayload();
                currentTaskInfosPayload.setProcInstIds(hashSet);
                // 调用工作流程方法查询任务信息
                final WorkflowResult<HashMap<String, TaskInfo>> workflowResult = workflowService.currentTaskInfos(currentTaskInfosPayload);
                if (workflowResult.isSuccess()) {
                    HashMap<String, TaskInfo> currentTaskNames = workflowResult.getData();
                    // 3、设置任务名称到业务对象上
                    currentTaskNames.forEach((key, value) -> {
                        TwWorkFlowCommonVO vo = map.get(key);
                        vo.setTaskInfo(value);
                    });
                }
            }
        }
    }

    /**
     * 数据权限
     * 权限处理
     *
     * @param query
     */
    void dataPermissionFlag(ContractTemplateQuery query) {
        Boolean rolePermission = cacheUtil.hasSystemRolePermission(Arrays.asList(RoleEnum.SYS.getCode(), RoleEnum.CONTRACT_ADMIN.getCode()));
        query.setDataPermissionFlag(!rolePermission);
        if (!rolePermission) {
            //需要处理权限
            final Long loginUserId = GlobalUtil.getLoginUserId();
            query.setLoginUserId(loginUserId);
            query.setLoginOrgId(cacheUtil.getDefaultOrgIdByUserId(loginUserId));
        }
    }

    @Override
    public ContractTemplateVO queryByKey(Long key) {
        final ContractTemplateVO vo = contractTemplateDAO.queryByKey(key);
        Assert.notNull(vo, "合同模版不存在");
        // 翻译
        translation(vo);
        // 流程信息回显
        final WorkflowResult<TaskInfo> workflowResult = workflowService.currentTaskInfo(vo.getProcInstId());
        if (workflowResult.isSuccess()) {
            TaskInfo taskInfo = workflowResult.getData();
            vo.setTaskInfo(taskInfo);
        }
        return vo;
    }

    private void translation(ContractTemplateVO vo) {
        vo.setFileDatas(fileUtil.getFileDatas(vo.getFileCodes()));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ContractTemplateVO insert(ContractTemplatePayload payload) {
        // 数据校验
        check(payload);
        // 数据处理
        dataProcess(payload);

        ContractTemplateDO entityDo = ContractTemplateConvert.INSTANCE.toDo(payload);
        final ContractTemplateDO save = contractTemplateRepo.save(entityDo);
        // 试用组织关系处理
        saveOrgRef(payload, save);

        //数据提交,执行工作流
        if (payload.getSubmit()) {
            payload.setId(save.getId());
            submitProc(payload);
        }

        // 提交流程
        return ContractTemplateConvert.INSTANCE.toVo(save);
    }

    /**
     * 试用部门 模版关系
     *
     * @param payload 有效载荷
     * @param save    保存
     */
    private void saveOrgRef(ContractTemplatePayload payload, ContractTemplateDO save) {
        final Long tempId = save.getId();
        // 先清空关系表数据
        contractTemplateOrgRefService.deleteByTempId(tempId);

        final String orgIdList = payload.getOrgIdList();
        if (StringUtils.hasText(orgIdList) && !ALL.equals(orgIdList)) {
            final String[] split = orgIdList.split(",");
            for (String orgId : split) {
                Long orgIdLong = Long.valueOf(orgId);
                final String orgName = cacheUtil.getOrgName(orgIdLong);
                final ContractTemplateOrgRefPayload contractTemplateOrgRefPayload = new ContractTemplateOrgRefPayload(orgName, tempId, orgIdLong);
                contractTemplateOrgRefService.insert(contractTemplateOrgRefPayload);
            }
        }
    }

    /**
     * 检查
     *
     * @param payload 有效载荷
     */
    private void check(ContractTemplatePayload payload) {
        if (ObjectUtils.isEmpty(payload.getName())) {
            throw TwException.error("", "请输入模版名称");
        }
    }

    /**
     * 数据处理
     *
     * @param payload 有效载荷
     */
    private void dataProcess(ContractTemplatePayload payload) {
        // 新建情况
        if (null == payload.getId()) {
            if (ObjectUtils.isEmpty(payload.getStatus())) {
                payload.setStatus(ContractTemplateStatusEnum.CREATE.getCode());
            }
            // 发号器
            String code = generateSeqNum(GenerateSeqNumConstants.CONTRACT_TEMPLATE);
            payload.setCode(code);
        }
        final String orgIdList = payload.getOrgIdList();

        if (StringUtils.hasText(orgIdList) && !ALL.equals(orgIdList)) {
            List<String> orgIdName = new ArrayList<>();
            final String[] split = orgIdList.split(",");
            List<ContractTemplateOrgRefPayload> templateOrgRefPayloads = new ArrayList<>();

            for (String orgId : split) {
                Long orgIdLong = Long.valueOf(orgId);
                final String orgName = cacheUtil.getOrgName(orgIdLong);
                orgIdName.add(orgName);
            }
            final String orgIdNameStr = orgIdName.stream().collect(Collectors.joining(","));
            payload.setOrgNameList(orgIdNameStr);
        }

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ContractTemplateVO update(ContractTemplatePayload payload) {
        ContractTemplateDO entity = contractTemplateRepo.findById(payload.getId()).orElseGet(ContractTemplateDO::new);
        Assert.notNull(entity.getId(), "合同模版不存在");
        // 数据校验
        check(payload);
        // 数据处理
        dataProcess(payload);
        ContractTemplateDO entityDo = ContractTemplateConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        final ContractTemplateDO save = contractTemplateRepo.save(entity);
        // 试用组织关系处理
        saveOrgRef(payload, save);

        //数据提交,执行工作流
        if (payload.getSubmit()) {
            payload.setId(save.getId());
            submitProc(payload);
        }
        return ContractTemplateConvert.INSTANCE.toVo(save);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ContractTemplateVO updateForFlow(ContractTemplatePayload payload) {
        ContractTemplateDO entity = contractTemplateRepo.findById(payload.getId()).orElseGet(ContractTemplateDO::new);
        Assert.notNull(entity.getId(), "合同模版不存在");
        // 数据校验
        check(payload);
        // 数据处理
        dataProcess(payload);
        ContractTemplateDO entityDo = ContractTemplateConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        final ContractTemplateDO save = contractTemplateRepo.save(entity);
        // 试用组织关系处理
        saveOrgRef(payload, save);
        return ContractTemplateConvert.INSTANCE.toVo(save);
    }

    /**
     * 提交流程
     *
     * @param payload 有效载荷
     */
    void submitProc(ContractTemplatePayload payload) {
        ProcessInfo processInfo = new ProcessInfo();
        String status = WorkFlowStatusEnum.APPROVED_WORK.getCode();
        if (workflow_enabled) {
            status = WorkFlowStatusEnum.APPROVING_WORK.getCode();
            //Long orgManageUserId = cacheUtil.getOrgManageUserId(payload.getOrgId());
            HashMap<String, Object> batchMap = new HashMap<>();
            //batchMap.put("Activity_1be86d6", Lists.newArrayList(orgManageUserId.toString()));
            //发起流程审批
            processInfo = workflowUtil.startProcess(StartProcessPayload.of(
                ProcDefKey.CONTRACT_TEMPLATE.name(),
                "合同模版-" + payload.getName() + "-审批流程",
                payload.getId() + "",
                batchMap)
            );
        }
        ContractTemplatePayload payload0 = new ContractTemplatePayload();
        payload0.setId(payload.getId());
        payload0.setProcInstId(processInfo.getProcInstId());
        payload0.setProcInstStatus(processInfo.getProcInstStatus());
        payload0.setSubmitTime(LocalDateTime.now());
        payload0.setStatus(status);
        //开启事务执行修改，主要是修改审批状态
        transactionUtilService.executeWithRunnable(() -> {
            contractTemplateDAO.updateWorkFlow(payload0);
        });
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long change(ContractTemplatePayload payload) {
        if (payload.getId() == null) {
            throw TwException.error("", "获取id失败");
        }
        if (payload.getStatus() == null || !payload.getStatus().equals(WorkFlowStatusEnum.APPROVED_WORK.getCode())) {
            throw TwException.error("", "仅激活的数据可以申请变更");
        }
        // 拼凑原始数据
        final ContractTemplateVO contractTemplateVO = contractTemplateDAO.queryByKey(payload.getId());
        Assert.notNull(contractTemplateVO, "合同模版不存在");
        // 翻译
        translation(contractTemplateVO);
        final ContractTemplatePayload contractTemplatePayload = ContractTemplateConvert.INSTANCE.toPayload(contractTemplateVO);

        payload.setFileDatas(fileUtil.getFileDatas(payload.getFileCodes()));

        // 记录变更
        Long saveId = businessChangeService.save(ChangeTypeEnum.CONTRACT_TEMPLATE.getCode(), udcUtil.translate(contractTemplatePayload), udcUtil.translate(payload), payload.getId() + "");

        ProcessInfo processInfo = new ProcessInfo();
        String apprStatusTemp = WorkFlowStatusEnum.APPROVED_WORK.getCode();
        String changeApprStatusTemp = WorkFlowStatusEnum.APPROVED_WORK.getCode();
        String changeStatusTemp = WorkFlowStatusEnum.APPROVED_WORK.getCode();
        if (workflow_enabled) {
            apprStatusTemp = WorkFlowStatusEnum.APPROVING_WORK.getCode();
            changeApprStatusTemp = WorkFlowStatusEnum.APPROVING_WORK.getCode();
            changeStatusTemp = WorkFlowStatusEnum.CHANGING_WORK.getCode();
            //Long orgManageUserId = cacheUtil.getOrgManageUserId(opportunityQuoteVO.getOrgId());
            HashMap<String, Object> batchMap = new HashMap<>();
            //batchMap.put("Activity_0bih1cr", Lists.newArrayList(orgManageUserId.toString()));
            //发起流程审批
            processInfo = workflowUtil.startProcess(StartProcessPayload.of(
                ProcDefKey.TEMPLATE_CHANGE.name(),
                "合同模版-" + payload.getName() + "-变更审批流程",
                saveId + "",
                batchMap)
            );
        }

        ContractTemplatePayload payload0 = new ContractTemplatePayload();
        payload0.setId(payload.getId());
        payload0.setChangeId(saveId);
        payload0.setChangeSubmitTime(LocalDateTime.now());
        payload0.setChangeProcInstId(processInfo.getProcInstId());
        payload0.setChangeProcInstStatus(processInfo.getProcInstStatus());
        payload0.setStatus(apprStatusTemp);

        ComBusinessChangePayload businessChangePayload = new ComBusinessChangePayload();
        businessChangePayload.setId(saveId);
        businessChangePayload.setApprProcInstId(processInfo.getProcInstId());
        businessChangePayload.setApprStatus(changeApprStatusTemp);
        businessChangePayload.setProcInstId(processInfo.getProcInstId());
        businessChangePayload.setProcInstStatus(processInfo.getProcInstStatus());
        businessChangePayload.setChangeStatus(changeStatusTemp);
        //开启事务执行修改，主要是修改审批状态
        transactionUtilService.executeWithRunnable(() -> {
            contractTemplateDAO.updateChangeWorkFlow(payload0);
            businessChangeDao.updateWorkFlow(businessChangePayload);
        });
        // 未走流程的 变更直接生效
        if (!workflow_enabled) {
            //根据业务key查询当前业务对象
            ComBusinessChangeVO comBusinessChangeVO = businessChangeDao.queryByKey(saveId);
            if (comBusinessChangeVO != null) {
                ContractTemplatePayload updatePayload = new ContractTemplatePayload();
                updatePayload.setId(Long.valueOf(comBusinessChangeVO.getChangeDocId()));
                updatePayload.setStatus(WorkFlowStatusEnum.APPROVED_WORK.getCode());
                updatePayload = JSON.parseObject(comBusinessChangeVO.getChangeContent(), ContractTemplatePayload.class);
                updatePayload.setApprovedTime(LocalDateTime.now());
                updatePayload.setVersion("v" + comBusinessChangeVO.getVersionNo().toString());

                ComBusinessChangePayload changePayload = new ComBusinessChangePayload();
                changePayload.setId(comBusinessChangeVO.getId());
                changePayload.setChangeStatus(WorkFlowStatusEnum.APPROVED_WORK.getCode());
                businessChangeDao.updateWorkFlow(changePayload);
                updateForFlow(updatePayload);
            }
        }

        return saveId;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            keys.stream().forEach(id -> {
                Optional<ContractTemplateDO> optional = contractTemplateRepo.findById(id);
                if (!optional.isEmpty()) {
                    ContractTemplateDO entity = optional.get();
                    entity.setDeleteFlag(1);
                    contractTemplateRepo.save(entity);
                }
            });
        }
    }

    /**
     * 变更更新
     *
     * @param payload 有效载荷
     * @return {@link Long}
     */
    @Override
    public Long changeUpdate(ContractTemplatePayload payload) {
        payload.setFileDatas(fileUtil.getFileDatas(payload.getFileCodes()));
        // 记录变更
        Long changeId = businessChangeService.update(ChangeTypeEnum.CONTRACT_TEMPLATE.getCode(), payload.getId() + "", udcUtil.translate(payload));
        return changeId;
    }

}
