package com.elitesland.tw.tw5crm.server.oppo.dao;

import com.elitesland.tw.tw5.server.prd.crm.entity.QCrmCustomerDO;
import com.elitesland.tw.tw5.server.prd.org.entity.QPrdOrgOrganizationDO;
import com.elitesland.tw.tw5crm.api.oppo.payload.OpportunityQuotePayload;
import com.elitesland.tw.tw5crm.api.oppo.query.OpportunityQuoteQuery;
import com.elitesland.tw.tw5crm.api.oppo.vo.OpportunityQuoteVO;
import com.elitesland.tw.tw5crm.api.product.payload.ProductPricePayload;
import com.elitesland.tw.tw5crm.server.oppo.entity.OpportunityQuoteDO;
import com.elitesland.tw.tw5crm.server.oppo.entity.QOpportunityQuoteDO;
import com.elitesland.tw.tw5crm.server.oppo.repo.OpportunityQuoteRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5crm.server.product.entity.QProductPriceOrgDO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 商机报价
 *
 * @author liwenpeng
 * @date 2023-03-23
 */
@Repository
@RequiredArgsConstructor
public class OpportunityQuoteDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final OpportunityQuoteRepo repo;
    private final QOpportunityQuoteDO qdo = QOpportunityQuoteDO.opportunityQuoteDO;

    private final QCrmCustomerDO customerQdo = QCrmCustomerDO.crmCustomerDO;

    private final QPrdOrgOrganizationDO organizationQdo = QPrdOrgOrganizationDO.prdOrgOrganizationDO;



    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<OpportunityQuoteVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(OpportunityQuoteVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 商机主键
                qdo.oppoId,
                // 报价单编码
                qdo.quoteNo,
                qdo.quoteStatus,
                // 客户名称
                qdo.customerName,
                // 整单折扣
                qdo.wholeOrderDiscount,
                // 报价单金额
                qdo.quoteAmt,
                // 报价单金额(含税)
                qdo.quoteTaxAmt,
                // 币种 udc[SYSTEM_BASIC:CURRENCY]
                qdo.currCode,
                // 归属部门
                qdo.orgId,
                organizationQdo.orgName,
                // 报价日期
                qdo.quoteDate,
                // 版本
                qdo.version,
                // 流程实例ID
                qdo.procInstId,
                // 流程审批状态
                qdo.procInstStatus,
                // 提交时间
                qdo.submitTime,
                // 审批时间
                qdo.approvedTime,
                // 排序号
                qdo.sortNo,
                // 拓展字段1
                qdo.extString1,
                // 拓展字段2
                qdo.extString2,
                // 拓展字段3
                qdo.extString3,
                // 拓展字段4
                qdo.extString4,
                // 拓展字段5
                qdo.extString5,
                // 负责人
                qdo.manageUserId
        )).from(qdo).leftJoin(organizationQdo).on(organizationQdo.id.eq(qdo.orgId));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<OpportunityQuoteVO> getJpaQueryWhere(OpportunityQuoteQuery query) {
        JPAQuery<OpportunityQuoteVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
            }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(OpportunityQuoteQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(OpportunityQuoteQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 商机主键 精确 */
        if (!ObjectUtils.isEmpty(query.getOppoId())) {
            list.add(qdo.oppoId.eq(query.getOppoId()));
        }
        /** 报价单编码 模糊 */
        if (!ObjectUtils.isEmpty(query.getQuoteNo())) {
             list.add(qdo.quoteNo.like(SqlUtil.toSqlLikeString(query.getQuoteNo())));
        }
        /** 客户名称 精确 */
        if (!ObjectUtils.isEmpty(query.getCustomerName())) {
            list.add(qdo.customerName.like(SqlUtil.toSqlLikeString(query.getCustomerName())));
        }
        /** 整单折扣 精确 */
        if (!ObjectUtils.isEmpty(query.getWholeOrderDiscount())) {
            list.add(qdo.wholeOrderDiscount.eq(query.getWholeOrderDiscount()));
        }
        /** 报价单金额 精确 */
        if (!ObjectUtils.isEmpty(query.getQuoteAmt())) {
            list.add(qdo.quoteAmt.eq(query.getQuoteAmt()));
        }
        /** 报价单金额(含税) 精确 */
        if (!ObjectUtils.isEmpty(query.getQuoteTaxAmt())) {
            list.add(qdo.quoteTaxAmt.eq(query.getQuoteTaxAmt()));
        }
        /** 币种 udc[SYSTEM_BASIC:CURRENCY] 精确 */
        if (!ObjectUtils.isEmpty(query.getCurrCode())) {
            list.add(qdo.currCode.eq(query.getCurrCode()));
        }
        /** 归属部门 精确 */
        if (!ObjectUtils.isEmpty(query.getOrgId())) {
            list.add(qdo.orgId.eq(query.getOrgId()));
        }
        /** 报价日期 精确 */
        if (!ObjectUtils.isEmpty(query.getQuoteDate())) {
            list.add(qdo.quoteDate.eq(query.getQuoteDate()));
        }
        /** 版本 精确 */
        if (!ObjectUtils.isEmpty(query.getVersion())) {
            list.add(qdo.version.eq(query.getVersion()));
        }
        /** 流程实例ID 精确 */
        if (!ObjectUtils.isEmpty(query.getProcInstId())) {
            list.add(qdo.procInstId.eq(query.getProcInstId()));
        }
        /** 流程审批状态 精确 */
//        if (!ObjectUtils.isEmpty(query.getProcInstStatus())) {
//            list.add(qdo.procInstStatus.eq(query.getProcInstStatus()));
//        }
        /** 排序号 精确 */
        if (!ObjectUtils.isEmpty(query.getSortNo())) {
            list.add(qdo.sortNo.eq(query.getSortNo()));
        }
        /** 拓展字段1 精确 */
        if (!ObjectUtils.isEmpty(query.getExtString1())) {
            list.add(qdo.extString1.eq(query.getExtString1()));
        }
        /** 拓展字段2 精确 */
        if (!ObjectUtils.isEmpty(query.getExtString2())) {
            list.add(qdo.extString2.eq(query.getExtString2()));
        }
        /** 拓展字段3 精确 */
        if (!ObjectUtils.isEmpty(query.getExtString3())) {
            list.add(qdo.extString3.eq(query.getExtString3()));
        }
        /** 拓展字段4 精确 */
        if (!ObjectUtils.isEmpty(query.getExtString4())) {
            list.add(qdo.extString4.eq(query.getExtString4()));
        }
        /** 拓展字段5 精确 */
        if (!ObjectUtils.isEmpty(query.getExtString5())) {
            list.add(qdo.extString5.eq(query.getExtString5()));
        }
        /** 负责人 精确 */
        if (!ObjectUtils.isEmpty(query.getManageUserId())) {
            list.add(qdo.manageUserId.eq(query.getManageUserId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public OpportunityQuoteVO queryByKey(Long id) {
        JPAQuery<OpportunityQuoteVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<OpportunityQuoteVO> queryListDynamic(OpportunityQuoteQuery query) {
        JPAQuery<OpportunityQuoteVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<OpportunityQuoteVO> queryPaging(OpportunityQuoteQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<OpportunityQuoteVO> jpaQuery = getJpaQueryWhere(query);
        List<OpportunityQuoteVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<OpportunityQuoteVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public OpportunityQuoteDO save(OpportunityQuoteDO ado) {
        return repo.save(ado);
    }

    public long updateWorkFlow(OpportunityQuotePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));

        // 流程实例id
        if (!ObjectUtils.isEmpty(payload.getProcInstId())) {
            update.set(qdo.procInstId, payload.getProcInstId());
        } else {
            update.setNull(qdo.procInstId);
        }
        // 审批状态
        if (!ObjectUtils.isEmpty(payload.getProcInstStatus())) {
            update.set(qdo.procInstStatus, payload.getProcInstStatus());
        } else {
            update.setNull(qdo.procInstStatus);
        }
        // 状态
        if (!ObjectUtils.isEmpty(payload.getQuoteStatus())) {
            update.set(qdo.quoteStatus, payload.getQuoteStatus());
        } else {
            update.setNull(qdo.quoteStatus);
        }
        // 提审时间
        if (!ObjectUtils.isEmpty(payload.getSubmitTime())) {
            update.set(qdo.submitTime, payload.getSubmitTime());
        }
        // 审批结束时间
        if (!ObjectUtils.isEmpty(payload.getApprovedTime())) {
            update.set(qdo.approvedTime, payload.getApprovedTime());
        }
        if (!ObjectUtils.isEmpty(payload.getDeleteFlag())) {
            update.set(qdo.deleteFlag, payload.getDeleteFlag());
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<OpportunityQuoteDO> saveAll(List<OpportunityQuoteDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(OpportunityQuotePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 商机主键
        if (payload.getOppoId() != null) {
            update.set(qdo.oppoId, payload.getOppoId());
        }
        // 报价单编码
        if (payload.getQuoteNo() != null) {
            update.set(qdo.quoteNo, payload.getQuoteNo());
        }
        // 客户名称
        if (payload.getCustomerName() != null) {
            update.set(qdo.customerName, payload.getCustomerName());
        }
        // 整单折扣
        if (payload.getWholeOrderDiscount() != null) {
            update.set(qdo.wholeOrderDiscount, payload.getWholeOrderDiscount());
        }
        // 报价单金额
        if (payload.getQuoteAmt() != null) {
            update.set(qdo.quoteAmt, payload.getQuoteAmt());
        }
        // 报价单金额(含税)
        if (payload.getQuoteTaxAmt() != null) {
            update.set(qdo.quoteTaxAmt, payload.getQuoteTaxAmt());
        }
        // 币种 udc[SYSTEM_BASIC:CURRENCY]
        if (payload.getCurrCode() != null) {
            update.set(qdo.currCode, payload.getCurrCode());
        }
        // 归属部门
        if (payload.getOrgId() != null) {
            update.set(qdo.orgId, payload.getOrgId());
        }
        // 报价日期
        if (payload.getQuoteDate() != null) {
            update.set(qdo.quoteDate, payload.getQuoteDate());
        }
        // 版本
        if (payload.getVersion() != null) {
            update.set(qdo.version, payload.getVersion());
        }
        // 流程实例ID
        if (payload.getProcInstId() != null) {
            update.set(qdo.procInstId, payload.getProcInstId());
        }
        // 流程审批状态
        if (payload.getQuoteStatus() != null) {
            update.set(qdo.quoteStatus, payload.getQuoteStatus());
        }
        // 排序号
        if (payload.getSortNo() != null) {
            update.set(qdo.sortNo, payload.getSortNo());
        }
        // 拓展字段1
        if (payload.getExtString1() != null) {
            update.set(qdo.extString1, payload.getExtString1());
        }
        // 拓展字段2
        if (payload.getExtString2() != null) {
            update.set(qdo.extString2, payload.getExtString2());
        }
        // 拓展字段3
        if (payload.getExtString3() != null) {
            update.set(qdo.extString3, payload.getExtString3());
        }
        // 拓展字段4
        if (payload.getExtString4() != null) {
            update.set(qdo.extString4, payload.getExtString4());
        }
        // 拓展字段5
        if (payload.getExtString5() != null) {
            update.set(qdo.extString5, payload.getExtString5());
        }
        // 负责人
        if (payload.getManageUserId() != null) {
            update.set(qdo.manageUserId, payload.getManageUserId());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 商机主键
            if (nullFields.contains("oppoId")) {
                update.setNull(qdo.oppoId);
            }
            // 报价单编码
            if (nullFields.contains("quoteNo")) {
                update.setNull(qdo.quoteNo);
            }
            // 客户名称
            if (nullFields.contains("customerName")) {
                update.setNull(qdo.customerName);
            }
            // 整单折扣
            if (nullFields.contains("wholeOrderDiscount")) {
                update.setNull(qdo.wholeOrderDiscount);
            }
            // 报价单金额
            if (nullFields.contains("quoteAmt")) {
                update.setNull(qdo.quoteAmt);
            }
            // 报价单金额(含税)
            if (nullFields.contains("quoteTaxAmt")) {
                update.setNull(qdo.quoteTaxAmt);
            }
            // 币种 udc[SYSTEM_BASIC:CURRENCY]
            if (nullFields.contains("currCode")) {
                update.setNull(qdo.currCode);
            }
            // 归属部门
            if (nullFields.contains("orgId")) {
                update.setNull(qdo.orgId);
            }
            // 报价日期
            if (nullFields.contains("quoteDate")) {
                update.setNull(qdo.quoteDate);
            }
            // 版本
            if (nullFields.contains("version")) {
                update.setNull(qdo.version);
            }
            // 流程实例ID
            if (nullFields.contains("procInstId")) {
                update.setNull(qdo.procInstId);
            }
            // 流程审批状态
            if (nullFields.contains("procInstStatus")) {
                update.setNull(qdo.procInstStatus);
            }
            // 排序号
            if (nullFields.contains("sortNo")) {
                update.setNull(qdo.sortNo);
            }
            // 拓展字段1
            if (nullFields.contains("extString1")) {
                update.setNull(qdo.extString1);
            }
            // 拓展字段2
            if (nullFields.contains("extString2")) {
                update.setNull(qdo.extString2);
            }
            // 拓展字段3
            if (nullFields.contains("extString3")) {
                update.setNull(qdo.extString3);
            }
            // 拓展字段4
            if (nullFields.contains("extString4")) {
                update.setNull(qdo.extString4);
            }
            // 拓展字段5
            if (nullFields.contains("extString5")) {
                update.setNull(qdo.extString5);
            }
            // 负责人
            if (nullFields.contains("manageUserId")) {
                update.setNull(qdo.manageUserId);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

