package com.elitesland.tw.tw5crm.server.oppo.service;

import com.alibaba.fastjson.JSON;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitesland.tw.tw5.api.common.TwWorkFlowCommonVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.service.TransactionUtilService;
import com.elitesland.tw.tw5.server.common.workFlow.ProcDefKey;
import com.elitesland.tw.tw5.server.common.workFlow.WorkflowUtil;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.WorkFlowStatusEnum;
import com.elitesland.tw.tw5.server.udc.UdcUtil;
import com.elitesland.tw.tw5crm.api.common.change.payload.ComBusinessChangePayload;
import com.elitesland.tw.tw5crm.api.common.change.service.ComBusinessChangeService;
import com.elitesland.tw.tw5crm.api.common.change.vo.ComBusinessChangeVO;
import com.elitesland.tw.tw5crm.api.oppo.payload.OpportunityQuoteDetailsPayload;
import com.elitesland.tw.tw5crm.api.oppo.payload.OpportunityQuotePayload;
import com.elitesland.tw.tw5crm.api.oppo.query.OpportunityQuoteDetailsQuery;
import com.elitesland.tw.tw5crm.api.oppo.query.OpportunityQuoteQuery;
import com.elitesland.tw.tw5crm.api.oppo.service.OpportunityQuoteService;
import com.elitesland.tw.tw5crm.api.oppo.vo.OpportunityQuoteDetailsVO;
import com.elitesland.tw.tw5crm.api.oppo.vo.OpportunityQuoteVO;
import com.elitesland.tw.tw5crm.server.common.change.changeTypeEnum.ChangeTypeEnum;
import com.elitesland.tw.tw5crm.server.common.change.dao.ComBusinessChangeDAO;
import com.elitesland.tw.tw5crm.server.common.constants.GenerateSeqNumConstants;
import com.elitesland.tw.tw5crm.server.oppo.convert.OpportunityQuoteConvert;
import com.elitesland.tw.tw5crm.server.oppo.convert.OpportunityQuoteDetailsConvert;
import com.elitesland.tw.tw5crm.server.oppo.dao.OpportunityQuoteDAO;
import com.elitesland.tw.tw5crm.server.oppo.dao.OpportunityQuoteDetailsDAO;
import com.elitesland.tw.tw5crm.server.oppo.entity.OpportunityQuoteDO;
import com.elitesland.tw.tw5crm.server.oppo.entity.OpportunityQuoteDetailsDO;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.TaskInfo;
import com.elitesland.workflow.WorkflowResult;
import com.elitesland.workflow.WorkflowService;
import com.elitesland.workflow.payload.CurrentTaskInfosPayload;
import com.elitesland.workflow.payload.StartProcessPayload;
import com.google.common.collect.Lists;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 商机报价
 *
 * @author liwenpeng
 * @date 2023-03-23
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class OpportunityQuoteServiceImpl extends BaseServiceImpl implements OpportunityQuoteService {

    private final WorkflowService workflowService;

    private final OpportunityQuoteDAO dao;
    private final OpportunityQuoteDetailsDAO detailsDao;
    private final CacheUtil cacheUtil;
    private final WorkflowUtil workflowUtil;
    private final TransactionUtilService transactionUtilService;
    private final UdcUtil udcUtil;
    private final ComBusinessChangeService businessChangeService;
    private final ComBusinessChangeDAO businessChangeDao;
    @Value("${tw5.workflow.enabled:false}")
    private Boolean workflow_enabled;

    @Override
    public PagingVO<OpportunityQuoteVO> queryPaging(OpportunityQuoteQuery query){
        PagingVO<OpportunityQuoteVO> opportunityQuoteVOPagingVO = dao.queryPaging(query);
        List<OpportunityQuoteVO> records = opportunityQuoteVOPagingVO.getRecords();
        records.forEach(this::translate);
        // 工作流 审批信息回显
        getTaskInfo(records);
        return opportunityQuoteVOPagingVO;
    }

    /**
     * 获取工作流审批信息
     *
     * @param voList 签证官列表
     */
    private void getTaskInfo(List<OpportunityQuoteVO> voList) {
        // 查询当前处理的任务名称
        if (!CollectionUtils.isEmpty(voList)) {
            // 1、组装成map，key为流程实例Id,value为对象
            Map<String, TwWorkFlowCommonVO> map = voList
                .stream()
                .filter(vo -> StringUtils.isNoneBlank(vo.getProcInstId()))
                .collect(Collectors.toMap(TwWorkFlowCommonVO::getProcInstId, Function.identity()));
            if (null != map && map.isEmpty() == false) {
                // 2、查询当前处理的任务名称
                final Set<String> procInstIds = map.keySet();
                HashSet hashSet = new HashSet(procInstIds);
                // 3、设置任务名称到业务对象上
                CurrentTaskInfosPayload currentTaskInfosPayload = new CurrentTaskInfosPayload();
                currentTaskInfosPayload.setProcInstIds(hashSet);
                // 调用工作流程方法查询任务信息
                final WorkflowResult<HashMap<String, TaskInfo>> workflowResult = workflowService.currentTaskInfos(currentTaskInfosPayload);
                if (workflowResult.isSuccess()) {
                    HashMap<String, TaskInfo> currentTaskNames = workflowResult.getData();
                    // 3、设置任务名称到业务对象上
                    currentTaskNames.forEach((key, value) -> {
                        TwWorkFlowCommonVO vo = map.get(key);
                        vo.setTaskInfo(value);
                    });
                }
            }
        }
    }

    @Override
    public OpportunityQuoteVO queryByKey(Long key) {

        OpportunityQuoteVO vo = dao.queryByKey(key);
        translate(vo);
        //查询明细
        OpportunityQuoteDetailsQuery query =new OpportunityQuoteDetailsQuery();
        query.setQuoteId(vo.getId());
        List<OpportunityQuoteDetailsVO> detailsVOS = detailsDao.queryListDynamic(query);
        vo.setDetails(detailsVOS);

        // 流程信息回显
        final WorkflowResult<TaskInfo> workflowResult = workflowService.currentTaskInfo(vo.getProcInstId());
        if (workflowResult.isSuccess()) {
            TaskInfo taskInfo = workflowResult.getData();
            vo.setTaskInfo(taskInfo);
        }

        return vo;
    }

    private void translate(OpportunityQuoteVO vo){
        Long manageUserId = vo.getManageUserId();
        if (manageUserId != null){
            vo.setManageUserName(cacheUtil.getUserName(manageUserId));
        }
        if (vo.getQuoteStatus()!=null){
            WorkFlowStatusEnum byCode = WorkFlowStatusEnum.getByCode(vo.getQuoteStatus());
            vo.setQuoteStatusDesc(byCode.getDesc());
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public OpportunityQuoteVO insert(OpportunityQuotePayload payload) {
        verifyTotalAmt(payload);
        OpportunityQuoteDO entityDo = OpportunityQuoteConvert.INSTANCE.toDo(payload);
        String code = generateSeqNum(GenerateSeqNumConstants.CRM_OPPO_QUOTE);
        entityDo.setQuoteNo(code);
        entityDo.setQuoteStatus(WorkFlowStatusEnum.CREATE_WORK.getCode());
        entityDo.setVersion("v0");
        OpportunityQuoteDO save = dao.save(entityDo);
        //添加明细
        insertDetail(payload.getDetails(),save.getId());

        // 添加流程相关
        //数据提交,执行工作流
        if (payload.getSubmit()) {
            payload.setId(save.getId());
            if(payload.getQuoteNo() == null){
                payload.setQuoteNo(save.getQuoteNo());
            }
            submitProc(payload);

        }

        return OpportunityQuoteConvert.INSTANCE.toVo(save);
    }

    void submitProc(OpportunityQuotePayload payload) {
        ProcessInfo processInfo = new ProcessInfo();
        String status = WorkFlowStatusEnum.APPROVED_WORK.getCode();
        if (workflow_enabled) {
            status = WorkFlowStatusEnum.APPROVING_WORK.getCode();
            Long orgManageUserId = cacheUtil.getOrgManageUserId(payload.getOrgId());
            HashMap<String, Object> batchMap = new HashMap<>();
            batchMap.put("Activity_1be86d6", Lists.newArrayList(orgManageUserId.toString()));
            //发起流程审批
            processInfo = workflowUtil.startProcess(StartProcessPayload.of(
                ProcDefKey.OPPO_QUOTE.name(),
                "商机报价-"+payload.getQuoteNo() + "-审批流程",
                payload.getId() + "",
                batchMap)
            );
        }
        OpportunityQuotePayload payload0 = new OpportunityQuotePayload();
        payload0.setId(payload.getId());
        payload0.setProcInstId(processInfo.getProcInstId());
        payload0.setProcInstStatus(processInfo.getProcInstStatus());
        payload0.setSubmitTime(LocalDateTime.now());
        payload0.setQuoteStatus(status);
        //开启事务执行修改，主要是修改审批状态
        transactionUtilService.executeWithRunnable(() -> {
            dao.updateWorkFlow(payload0);
        });

    }

    private ProcessInfo startChangeWorkFlow(OpportunityQuoteVO opportunityQuoteVO, Long saveId) {
        Long orgManageUserId = cacheUtil.getOrgManageUserId(opportunityQuoteVO.getOrgId());
        HashMap<String, Object> batchMap = new HashMap<>();
        batchMap.put("Activity_0bih1cr", Lists.newArrayList(orgManageUserId.toString()));
        //发起流程审批
        ProcessInfo processInfo = workflowUtil.startProcess(StartProcessPayload.of(
                ProcDefKey.OPPO_QUOTE_CHANGE.name(),
                "商机报价-"+opportunityQuoteVO.getQuoteNo() + "-变更审批流程",
                saveId + "",
                batchMap)
        );
        return processInfo;
    }

    public void verifyTotalAmt(OpportunityQuotePayload payload){
        List<OpportunityQuoteDetailsPayload> details = payload.getDetails();
        if (details != null && details.size() > 0){
            // 汇总产品售价 产品售价 = 数量 * 价目表售价 * 销售折扣
            BigDecimal productPriceTotal = details.stream().map(e -> {
                Integer quantity = e.getQuantity() == null ? 1 : e.getQuantity();
                BigDecimal markedMoney = e.getMarkedMoney() == null ? BigDecimal.ZERO : e.getMarkedMoney();
                BigDecimal salesDiscount = e.getSalesDiscount() == null ? BigDecimal.ZERO : e.getSalesDiscount();
                return markedMoney.multiply(new BigDecimal(quantity)).multiply(salesDiscount);
            }).reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
            // 判断前端的报价单金额 和 后端计算的金额是否一致 报价单金额 = 产品售价汇总 * 整单折扣
            String wholeOrderDiscount = payload.getWholeOrderDiscount();
            BigDecimal quoteAmtByCount = productPriceTotal.multiply(new BigDecimal(wholeOrderDiscount)).setScale(2, RoundingMode.HALF_UP);
            BigDecimal quoteAmt = payload.getQuoteAmt() == null ? BigDecimal.ZERO : payload.getQuoteAmt().setScale(2, RoundingMode.HALF_UP);
            if (quoteAmtByCount.compareTo(quoteAmt) != 0){
                log.error("前端传递金额={},后端计算金额={}",quoteAmt,quoteAmtByCount);
                throw TwException.error("","报价单金额验证失败");
            }

            // 汇总产品含税售价 产品含税售价 = 产品售价 * （1+税率）
            BigDecimal productSalesIncludeTaxTotal = details.stream().map(e -> {
                Integer quantity = e.getQuantity() == null ? 1 : e.getQuantity();
                BigDecimal markedMoney = e.getMarkedMoney() == null ? BigDecimal.ZERO : e.getMarkedMoney();
                BigDecimal salesDiscount = e.getSalesDiscount() == null ? BigDecimal.ZERO : e.getSalesDiscount();
                BigDecimal taxRate = e.getTaxRate() == null ? BigDecimal.ZERO : e.getTaxRate();
                return markedMoney.multiply(new BigDecimal(quantity)).multiply(salesDiscount).multiply(taxRate.add(BigDecimal.ONE));
            }).reduce(BigDecimal.ZERO, BigDecimal::add).setScale(2, RoundingMode.HALF_UP);
            // 判断前端的报价单金额 和 后端计算的金额是否一致 报价单金额 = 产品售价汇总 * 整单折扣
            BigDecimal quoteTaxAmtByCount = productSalesIncludeTaxTotal.multiply(new BigDecimal(wholeOrderDiscount)).setScale(2, RoundingMode.HALF_UP);
            BigDecimal quoteTaxAmt = payload.getQuoteTaxAmt() == null ? BigDecimal.ZERO : payload.getQuoteTaxAmt().setScale(2, RoundingMode.HALF_UP);
            if (quoteTaxAmtByCount.compareTo(quoteTaxAmt) != 0){
                log.error("前端传递含税金额={},后端计算含税金额={}",quoteTaxAmt,quoteTaxAmtByCount);
                throw TwException.error("","报价单含税金额验证失败");
            }
        }

    }

    private void insertDetail(List<OpportunityQuoteDetailsPayload> details,Long quoteId){
        if (details != null && details.size() > 0){
            details.forEach(e->{
                OpportunityQuoteDetailsDO detailsDO = OpportunityQuoteDetailsConvert.INSTANCE.toDo(e);
                detailsDO.setQuoteId(quoteId);
                detailsDao.save(detailsDO);
            });
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public long update(OpportunityQuotePayload payload) {
        if (payload.getId()==null){
            throw TwException.error("","获取id失败");
        }
        verifyTotalAmt(payload);
        long l = dao.updateByKeyDynamic(payload);
        if (payload.getDetails()!=null && payload.getDetails().size()>0){
            detailsDao.deleteByQuoteId(payload.getId());
            //添加明细
            insertDetail(payload.getDetails(),payload.getId());
        }
        // 添加流程相关
        //数据提交,执行工作流
        if (payload.getSubmit()) {
            payload.setId(payload.getId());
            submitProc(payload);

        }
        return l;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public long updateChange(OpportunityQuotePayload payload) {
        if (payload.getId()==null){
            throw TwException.error("","获取id失败");
        }
        verifyTotalAmt(payload);
        long l = dao.updateByKeyDynamic(payload);
        if (payload.getDetails()!=null && payload.getDetails().size()>0){
            List<OpportunityQuoteDetailsPayload> details = payload.getDetails();
            details.forEach(e->detailsDao.updateByKeyDynamic(e));

        }
        return l;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public long change(OpportunityQuotePayload payload) {
        if (payload.getId()==null){
            throw TwException.error("","获取id失败");
        }
        if (payload.getQuoteStatus() == null  ||  !payload.getQuoteStatus().equals(WorkFlowStatusEnum.APPROVED.getCode())){
            throw TwException.error("","仅审批通过的数据可以申请变更");
        }
        verifyTotalAmt(payload);
        // 拼凑原始数据
        OpportunityQuoteVO opportunityQuoteVO = dao.queryByKey(payload.getId());
        if (opportunityQuoteVO.getId()==null){
            throw TwException.error("","原数据不存在");
        }
        OpportunityQuotePayload opportunityQuotePayload = OpportunityQuoteConvert.INSTANCE.toPayload(opportunityQuoteVO);
        // 拼凑原始数据明细
        OpportunityQuoteDetailsQuery query =new OpportunityQuoteDetailsQuery();
        query.setQuoteId(payload.getId());
        List<OpportunityQuoteDetailsVO> detailsVOS = detailsDao.queryListDynamic(query);
        if (detailsVOS!=null && detailsVOS.size()>0){
            List<OpportunityQuoteDetailsPayload> opportunityQuoteDetailsPayloads = OpportunityQuoteDetailsConvert.INSTANCE.toPayloadList(detailsVOS);
            opportunityQuotePayload.setDetails((List<OpportunityQuoteDetailsPayload>) udcUtil.translate(opportunityQuoteDetailsPayloads));
        }

        // 记录变更
        Long saveId = businessChangeService.save(ChangeTypeEnum.OPPO_QUOTE.getCode(), udcUtil.translate(opportunityQuotePayload), udcUtil.translate(payload), payload.getId() + "");

        //ProcessInfo processInfo = startChangeWorkFlow(opportunityQuoteVO, saveId);
        ProcessInfo processInfo = new ProcessInfo();
        String apprStatusTemp = WorkFlowStatusEnum.APPROVED_WORK.getCode();
        String changeApprStatusTemp = WorkFlowStatusEnum.APPROVED_WORK.getCode();
        String changeStatusTemp = WorkFlowStatusEnum.APPROVED_WORK.getCode();
        if(workflow_enabled){
            apprStatusTemp = WorkFlowStatusEnum.CHANGING_WORK.getCode();
            changeApprStatusTemp = WorkFlowStatusEnum.APPROVING_WORK.getCode();
            changeStatusTemp = WorkFlowStatusEnum.CHANGING_WORK.getCode();
            Long orgManageUserId = cacheUtil.getOrgManageUserId(opportunityQuoteVO.getOrgId());
            HashMap<String, Object> batchMap = new HashMap<>();
            batchMap.put("Activity_0bih1cr", Lists.newArrayList(orgManageUserId.toString()));
            //发起流程审批
            processInfo = workflowUtil.startProcess(StartProcessPayload.of(
                ProcDefKey.OPPO_QUOTE_CHANGE.name(),
                "商机报价-"+opportunityQuoteVO.getQuoteNo() + "-变更审批流程",
                saveId + "",
                batchMap)
            );
        }

        OpportunityQuotePayload payload0 = new OpportunityQuotePayload();
        payload0.setId(payload.getId());
        payload0.setSubmitTime(LocalDateTime.now());
        payload0.setQuoteStatus(apprStatusTemp);

        ComBusinessChangePayload businessChangePayload = new ComBusinessChangePayload();
        businessChangePayload.setId(saveId);
        businessChangePayload.setApprProcInstId(processInfo.getProcInstId());
        businessChangePayload.setApprStatus(changeApprStatusTemp);
        businessChangePayload.setChangeStatus(changeStatusTemp);
        //开启事务执行修改，主要是修改审批状态
        transactionUtilService.executeWithRunnable(() -> {
            dao.updateWorkFlow(payload0);
            businessChangeDao.updateWorkFlow(businessChangePayload);
        });
        // 未走流程的 变更直接生效
        if (!workflow_enabled) {
            //根据业务key查询当前业务对象
            ComBusinessChangeVO comBusinessChangeVO = businessChangeDao.queryByKey(saveId);
            if (comBusinessChangeVO != null) {
                OpportunityQuotePayload quotePayload = new OpportunityQuotePayload();
                quotePayload.setId(Long.valueOf(comBusinessChangeVO.getChangeDocId()));
                quotePayload.setQuoteStatus(WorkFlowStatusEnum.APPROVED_WORK.getCode());
                quotePayload = JSON.parseObject(comBusinessChangeVO.getChangeContent(), OpportunityQuotePayload.class);
                quotePayload.setApprovedTime(LocalDateTime.now());
                quotePayload.setVersion("v" + comBusinessChangeVO.getVersionNo().toString());

                ComBusinessChangePayload changePayload = new ComBusinessChangePayload();
                changePayload.setId(comBusinessChangeVO.getId());
                changePayload.setChangeStatus(WorkFlowStatusEnum.APPROVED_WORK.getCode());
                businessChangeDao.updateWorkFlow(changePayload);
                updateChange(quotePayload);
            }
        }

        return saveId;
    }




    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            // todo 查询非新增状态的count
            dao.deleteSoft(keys);
        }
    }

}
