package com.elitesland.tw.tw5crm.server.product.controller;

import com.elitesland.tw.tw5.server.common.TwOutputUtil;
import com.elitesland.tw.tw5.server.common.annotation.ReSubmitCheck;
import com.elitesland.tw.tw5crm.api.product.payload.ProductCategoryPayload;
import com.elitesland.tw.tw5crm.api.product.query.ProductCategoryQuery;
import com.elitesland.tw.tw5crm.api.product.service.ProductCategoryService;
import com.elitesland.tw.tw5crm.api.product.vo.ProductCategoryVO;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import java.util.Arrays;
import java.util.List;

/**
 * 产品管理-产品分类
 *
 * @author duwh
 * @date 2023-03-02
 */
@Api(tags = "产品管理-产品分类")
@RestController
@RequiredArgsConstructor
@RequestMapping("/api/crm/product/category")
@Slf4j
public class ProductCategoryController {

    private final ProductCategoryService productCategoryService;

    /**
     * 新增
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PostMapping
    @ApiOperation("新增")
    @ReSubmitCheck(argExpressions = {"[0].id", "[0].objName"})
    public TwOutputUtil insert(@RequestBody ProductCategoryPayload payload) {
        return TwOutputUtil.ok(productCategoryService.insert(payload));
    }

    /**
     * 校验名称重复
     *
     * @param objName 分类名称
     * @return {@link TwOutputUtil}
     */
    @PostMapping("checkUnique")
    public TwOutputUtil checkUnique(Long id, String objName) {
        return TwOutputUtil.ok(productCategoryService.checkUnique(id, objName));
    }

    /**
     * 更新
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PutMapping
    public TwOutputUtil update(@RequestBody ProductCategoryPayload payload) {
        return TwOutputUtil.ok(productCategoryService.update(payload));
    }

    /**
     * 返回全部的菜单
     *
     * @param pid 父id
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/lazy")
    // @UdcNameClass
    public TwOutputUtil queryAllCategory(Long pid) {
        return TwOutputUtil.ok(productCategoryService.queryAllCategory(pid));
    }

    /**
     * 构建树结构
     *
     * @return {@link TwOutputUtil}
     */
    @GetMapping(value = "/tree")
    public TwOutputUtil tree() {
        List<ProductCategoryVO> categoryVOList = productCategoryService.tree();
        return TwOutputUtil.ok(categoryVOList);
    }

    /**
     * 主键查询
     *
     * @param key 关键
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/{key}")
    // @UdcNameClass
    public TwOutputUtil queryOneByKey(@PathVariable Long key) {
        return TwOutputUtil.ok(productCategoryService.queryByKey(key));
    }

    /**
     * 获取 规格属性JSON
     *
     * @return
     */
    @GetMapping("/getAttrs")
    // @UdcNameClass
    public TwOutputUtil getAttrs() {
        return TwOutputUtil.ok(productCategoryService.getAttrs());
    }

    /**
     * 根据分类 获取属性组（组下属性集合）、规格属性
     *
     * @param key 产品分类主键
     * @return
     */
    @GetMapping("/getAttrsAndAttrGroups/{key}")
    // @UdcNameClass
    public TwOutputUtil getAttrsAndAttrGroups(@PathVariable Long key) {
        return TwOutputUtil.ok(productCategoryService.getAttrsAndAttrGroups(key));
    }

    /**
     * 分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/paging")
    // @UdcNameClass
    public TwOutputUtil paging(ProductCategoryQuery query) {
        return TwOutputUtil.ok(productCategoryService.paging(query));
    }

    /**
     * 查询列表
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/list")
    // @UdcNameClass
    public TwOutputUtil queryList(ProductCategoryQuery query) {
        return TwOutputUtil.ok(productCategoryService.queryList(query));
    }

    /**
     * 删除
     *
     * @param keys 主键id
     * @return result
     */
    @DeleteMapping("/deleteSoft")
    public TwOutputUtil deleteSoft(Long[] keys) {
        productCategoryService.deleteSoft(Arrays.asList(keys));
        return TwOutputUtil.ok();
    }

}
