package com.elitesland.tw.tw5crm.server.product.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5crm.api.product.payload.CrmBusinessAttributeGroupDetailPayload;
import com.elitesland.tw.tw5crm.api.product.query.CrmBusinessAttributeGroupDetailQuery;
import com.elitesland.tw.tw5crm.api.product.vo.CrmBusinessAttributeGroupDetailVO;
import com.elitesland.tw.tw5crm.server.product.entity.CrmBusinessAttributeGroupDetailDO;
import com.elitesland.tw.tw5crm.server.product.entity.QCrmBusinessAttributeGroupDetailDO;
import com.elitesland.tw.tw5crm.server.product.entity.QCrmBusinessTableColumnsDO;
import com.elitesland.tw.tw5crm.server.product.repo.CrmBusinessAttributeGroupDetailRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.dsl.Expressions;
import com.querydsl.core.types.dsl.StringTemplate;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 通用扩展属性明细
 *
 * @author liwenpeng
 * @date 2023-03-02
 */
@Repository
@RequiredArgsConstructor
public class CrmBusinessAttributeGroupDetailDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final CrmBusinessAttributeGroupDetailRepo repo;
    private final QCrmBusinessAttributeGroupDetailDO qdo = QCrmBusinessAttributeGroupDetailDO.crmBusinessAttributeGroupDetailDO;
    private final QCrmBusinessTableColumnsDO columnsQdo = QCrmBusinessTableColumnsDO.crmBusinessTableColumnsDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery
            <CrmBusinessAttributeGroupDetailVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(CrmBusinessAttributeGroupDetailVO.class,
                // 属性组id
                qdo.groupId,
                // 属性id
                qdo.attributeId,
                // 维护级别 SPU、SKU
                qdo.unitClass,
                // 是否必填
                qdo.isRequired,
                qdo.creator,
                qdo.createTime,
                qdo.createUserId,
                qdo.updater,
                qdo.modifyTime,
                qdo.modifyUserId,
                // 排序号
                qdo.sortNo
        )).from(qdo);
    }

    private JPAQuery<CrmBusinessAttributeGroupDetailVO> getJpaQueryJoinSelect() {
        return jpaQueryFactory.select(Projections.bean(CrmBusinessAttributeGroupDetailVO.class,
                // 属性组id
                qdo.id,
                qdo.groupId,
                // 属性id
                qdo.attributeId,
                // 维护级别 SPU、SKU
                qdo.unitClass,
                // 是否必填
                qdo.isRequired,
                qdo.creator,
                qdo.createTime,
                qdo.createUserId,
                qdo.updater,
                qdo.modifyTime,
                qdo.modifyUserId,
                //columnsQdo.attributeDesc.as("attributeDesc"),
                columnsQdo.attributeDesc,
                columnsQdo.columnName,
                columnsQdo.selectionCode,
                columnsQdo.isMultiple,
                columnsQdo.attributeType,
                columnsQdo.componentType,
                columnsQdo.attributeScope,
                columnsQdo.attributePrompt,
                columnsQdo.status,
                // 排序号
                qdo.sortNo
        )).from(qdo).leftJoin(columnsQdo).on(qdo.attributeId.eq(columnsQdo.id));
    }

    private JPAQuery<CrmBusinessAttributeGroupDetailVO> getJpaListQuerySelect() {
        StringTemplate attributeNameStr = Expressions.stringTemplate("CONCAT(GROUP_CONCAT({0}),'')", columnsQdo.attributeDesc);
        StringTemplate attributeIdStr = Expressions.stringTemplate("CONCAT(GROUP_CONCAT({0}),'')", columnsQdo.id);
        return jpaQueryFactory.select(Projections.bean(CrmBusinessAttributeGroupDetailVO.class,
                        qdo.groupId.as("id"),
                        qdo.groupId,
                        attributeIdStr.as("attributeIdStr"),
                        attributeNameStr.as("attributeNameStr")
                )).from(qdo)
                .leftJoin(columnsQdo).on(qdo.attributeId.eq(columnsQdo.id))
                .groupBy(qdo.groupId);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<CrmBusinessAttributeGroupDetailVO> getJpaQueryWhere(CrmBusinessAttributeGroupDetailQuery query) {
        JPAQuery<CrmBusinessAttributeGroupDetailVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(CrmBusinessAttributeGroupDetailQuery query) {
        long total = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .where(where(query))
                .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(CrmBusinessAttributeGroupDetailQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 属性组id 精确 */
        if (!ObjectUtils.isEmpty(query.getGroupId())) {
            list.add(qdo.groupId.eq(query.getGroupId()));
        }
        /** 属性组id 精确 */
        if (!ObjectUtils.isEmpty(query.getGroupIdList())) {
            list.add(qdo.groupId.in(query.getGroupIdList()));
        }
        /** 属性id 精确 */
        if (!ObjectUtils.isEmpty(query.getAttributeId())) {
            list.add(qdo.attributeId.eq(query.getAttributeId()));
        }
        /** 维护级别 SPU、SKU 精确 */
        if (!ObjectUtils.isEmpty(query.getUnitClass())) {
            list.add(qdo.unitClass.eq(query.getUnitClass()));
        }
        /** 是否必填 精确 */
        if (!ObjectUtils.isEmpty(query.getIsRequired())) {
            list.add(qdo.isRequired.eq(query.getIsRequired()));
        }
        /** 排序号 精确 */
        if (!ObjectUtils.isEmpty(query.getSortNo())) {
            list.add(qdo.sortNo.eq(query.getSortNo()));
        }
        return ExpressionUtils.allOf(list);
    }


    public List<CrmBusinessAttributeGroupDetailVO> queryDetailNameStrByGroupIdList(List<Long> groupIdList) {
        JPAQuery<CrmBusinessAttributeGroupDetailVO> jpaQuery = getJpaListQuerySelect();
        jpaQuery.where(qdo.groupId.in(groupIdList));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    public List<CrmBusinessAttributeGroupDetailVO> queryByGroupId(List<Long> groupIdList){
        JPAQuery<CrmBusinessAttributeGroupDetailVO> jpaQuerySelect = getJpaQueryJoinSelect();
        jpaQuerySelect.where(qdo.groupId.in(groupIdList));
        jpaQuerySelect.where(qdo.deleteFlag.eq(0));
        return jpaQuerySelect.fetch();

    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public CrmBusinessAttributeGroupDetailVO queryByKey(Long id) {
        JPAQuery<CrmBusinessAttributeGroupDetailVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    public List<CrmBusinessAttributeGroupDetailVO> queryByGroupId(Long groupId) {
        JPAQuery<CrmBusinessAttributeGroupDetailVO> jpaQuery = getJpaQueryJoinSelect();
        jpaQuery.where(qdo.groupId.eq(groupId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<CrmBusinessAttributeGroupDetailVO> queryListDynamic(CrmBusinessAttributeGroupDetailQuery query) {
        JPAQuery<CrmBusinessAttributeGroupDetailVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 关联查询动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<CrmBusinessAttributeGroupDetailVO> queryListJoinDynamic(CrmBusinessAttributeGroupDetailQuery query) {
        JPAQuery<CrmBusinessAttributeGroupDetailVO> jpaQuery = getJpaQueryJoinSelect();
        // 条件封装
        jpaQuery.where(where(query));
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<CrmBusinessAttributeGroupDetailVO> queryPaging(CrmBusinessAttributeGroupDetailQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<CrmBusinessAttributeGroupDetailVO> jpaQuery = getJpaQueryWhere(query);
        List<CrmBusinessAttributeGroupDetailVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<CrmBusinessAttributeGroupDetailVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public CrmBusinessAttributeGroupDetailDO save(CrmBusinessAttributeGroupDetailDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<CrmBusinessAttributeGroupDetailDO> saveAll(List<CrmBusinessAttributeGroupDetailDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(CrmBusinessAttributeGroupDetailPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 属性组id
        if (payload.getGroupId() != null) {
            update.set(qdo.groupId, payload.getGroupId());
        }
        // 属性id
        if (payload.getAttributeId() != null) {
            update.set(qdo.attributeId, payload.getAttributeId());
        }
        // 维护级别 SPU、SKU
        if (payload.getUnitClass() != null) {
            update.set(qdo.unitClass, payload.getUnitClass());
        }
        // 是否必填
        if (payload.getIsRequired() != null) {
            update.set(qdo.isRequired, payload.getIsRequired());
        }
        // 排序号
        if (payload.getSortNo() != null) {
            update.set(qdo.sortNo, payload.getSortNo());
        }
        // 处理要设置成空的字段
//        List<String> nullFields = payload.getNullFields();
//        if (nullFields != null && nullFields.size() > 0) {
//            // 记录唯一ID
//            if (nullFields.contains("id")) {
//                update.setNull(qdo.id);
//            }
//            // 属性组id
//            if (nullFields.contains("groupId")) {
//                update.setNull(qdo.groupId);
//            }
//            // 属性id
//            if (nullFields.contains("attributeId")) {
//                update.setNull(qdo.attributeId);
//            }
//            // 维护级别 SPU、SKU
//            if (nullFields.contains("unitClass")) {
//                update.setNull(qdo.unitClass);
//            }
//            // 是否必填
//            if (nullFields.contains("isRequired")) {
//                update.setNull(qdo.isRequired);
//            }
//            // 排序号
//            if (nullFields.contains("sortNo")) {
//                update.setNull(qdo.sortNo);
//            }
//        }
//        //拼装更新
//        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    public long deleteByGroupId(Long groupId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.groupId.eq(groupId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

