package com.elitesland.tw.tw5crm.server.product.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5crm.api.product.payload.ProductCategoryAttrGroupRefPayload;
import com.elitesland.tw.tw5crm.api.product.query.ProductCategoryAttrGroupRefQuery;
import com.elitesland.tw.tw5crm.api.product.vo.ProductCategoryAttrGroupRefVO;
import com.elitesland.tw.tw5crm.server.product.entity.ProductCategoryAttrGroupRefDO;
import com.elitesland.tw.tw5crm.server.product.entity.QProductCategoryAttrGroupRefDO;
import com.elitesland.tw.tw5crm.server.product.repo.ProductCategoryAttrGroupRefRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
* 产品分类属性组关系表
*
* @author duwh
* @date 2023-03-03
*/
@Repository
@RequiredArgsConstructor
public class ProductCategoryAttrGroupRefDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ProductCategoryAttrGroupRefRepo repo;
    private final QProductCategoryAttrGroupRefDO qdo = QProductCategoryAttrGroupRefDO.productCategoryAttrGroupRefDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ProductCategoryAttrGroupRefVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ProductCategoryAttrGroupRefVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 产品分类ID:crm_product_category.id
                qdo.categoryId,
                // 分类名称（冗余）crm_product_category.obj_name
                qdo.categoryName,
                // 属性表ID:crm_business_attribute_group.id
                qdo.groupId,
                // 扩展属性组名称（冗余）
                qdo.groupName,
                // 状态，0未启用，1已启用
                qdo.status,
                // 排序号（冗余）
                qdo.sortNo,
                // 自定义排序号（分类）
                qdo.sortNoSelf
        )).from(qdo);
    }

    /**
    * 拼装查询条件
    *
    * @param query 查询参数
    * @return jpaQuery对象
    */
    private JPAQuery<ProductCategoryAttrGroupRefVO> getJpaQueryWhere(ProductCategoryAttrGroupRefQuery query) {
        JPAQuery<ProductCategoryAttrGroupRefVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
            }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ProductCategoryAttrGroupRefQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
    * 查询条件封装
    *
    * @param query 条件
    * @return {@link Predicate}
    */
    private Predicate where(ProductCategoryAttrGroupRefQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 产品分类ID:crm_product_category.id 精确 */
        if (!ObjectUtils.isEmpty(query.getCategoryId())) {
            list.add(qdo.categoryId.eq(query.getCategoryId()));
        }
        /** 分类名称（冗余）crm_product_category.obj_name 模糊 */
        if (!ObjectUtils.isEmpty(query.getCategoryName())) {
             list.add(qdo.categoryName.like(SqlUtil.toSqlLikeString(query.getCategoryName())));
        }
        /** 属性表ID:crm_business_attribute_group.id 精确 */
        if (!ObjectUtils.isEmpty(query.getGroupId())) {
            list.add(qdo.groupId.eq(query.getGroupId()));
        }
        /** 扩展属性组名称（冗余） 模糊 */
        if (!ObjectUtils.isEmpty(query.getGroupName())) {
             list.add(qdo.groupName.like(SqlUtil.toSqlLikeString(query.getGroupName())));
        }
        /** 状态，0未启用，1已启用 精确 */
        if (!ObjectUtils.isEmpty(query.getStatus())) {
            list.add(qdo.status.eq(query.getStatus()));
        }
        /** 排序号（冗余） 精确 */
        if (!ObjectUtils.isEmpty(query.getSortNo())) {
            list.add(qdo.sortNo.eq(query.getSortNo()));
        }
        /** 自定义排序号（分类） 精确 */
        if (!ObjectUtils.isEmpty(query.getSortNoSelf())) {
            list.add(qdo.sortNoSelf.eq(query.getSortNoSelf()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ProductCategoryAttrGroupRefVO queryByKey(Long id) {
        JPAQuery<ProductCategoryAttrGroupRefVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ProductCategoryAttrGroupRefVO> queryListDynamic(ProductCategoryAttrGroupRefQuery query) {
        JPAQuery<ProductCategoryAttrGroupRefVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ProductCategoryAttrGroupRefVO> queryPaging(ProductCategoryAttrGroupRefQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<ProductCategoryAttrGroupRefVO> jpaQuery = getJpaQueryWhere(query);
        List<ProductCategoryAttrGroupRefVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ProductCategoryAttrGroupRefVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ProductCategoryAttrGroupRefDO save(ProductCategoryAttrGroupRefDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ProductCategoryAttrGroupRefDO> saveAll(List<ProductCategoryAttrGroupRefDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ProductCategoryAttrGroupRefPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
                // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
                // 产品分类ID:crm_product_category.id
        if (payload.getCategoryId() != null) {
            update.set(qdo.categoryId, payload.getCategoryId());
        }
                // 分类名称（冗余）crm_product_category.obj_name
        if (payload.getCategoryName() != null) {
            update.set(qdo.categoryName, payload.getCategoryName());
        }
                // 属性表ID:crm_business_attribute_group.id
        if (payload.getGroupId() != null) {
            update.set(qdo.groupId, payload.getGroupId());
        }
                // 扩展属性组名称（冗余）
        if (payload.getGroupName() != null) {
            update.set(qdo.groupName, payload.getGroupName());
        }
                // 状态，0未启用，1已启用
        if (payload.getStatus() != null) {
            update.set(qdo.status, payload.getStatus());
        }
                // 排序号（冗余）
        if (payload.getSortNo() != null) {
            update.set(qdo.sortNo, payload.getSortNo());
        }
                // 自定义排序号（分类）
        if (payload.getSortNoSelf() != null) {
            update.set(qdo.sortNoSelf, payload.getSortNoSelf());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 产品分类ID:crm_product_category.id
            if (nullFields.contains("categoryId")) {
                update.setNull(qdo.categoryId);
            }
            // 分类名称（冗余）crm_product_category.obj_name
            if (nullFields.contains("categoryName")) {
                update.setNull(qdo.categoryName);
            }
            // 属性表ID:crm_business_attribute_group.id
            if (nullFields.contains("groupId")) {
                update.setNull(qdo.groupId);
            }
            // 扩展属性组名称（冗余）
            if (nullFields.contains("groupName")) {
                update.setNull(qdo.groupName);
            }
            // 状态，0未启用，1已启用
            if (nullFields.contains("status")) {
                update.setNull(qdo.status);
            }
            // 排序号（冗余）
            if (nullFields.contains("sortNo")) {
                update.setNull(qdo.sortNo);
            }
            // 自定义排序号（分类）
            if (nullFields.contains("sortNoSelf")) {
                update.setNull(qdo.sortNoSelf);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

