package com.elitesland.tw.tw5crm.server.product.dao;

import com.elitesland.tw.tw5crm.api.product.payload.ProductCategoryColumnRefPayload;
import com.elitesland.tw.tw5crm.api.product.query.ProductCategoryColumnRefQuery;
import com.elitesland.tw.tw5crm.api.product.vo.ProductCategoryColumnRefVO;
import com.elitesland.tw.tw5crm.server.product.entity.ProductCategoryColumnRefDO;
import com.elitesland.tw.tw5crm.server.product.entity.QProductCategoryColumnRefDO;
import com.elitesland.tw.tw5crm.server.product.repo.ProductCategoryColumnRefRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 产品分类属性关系
 *
 * @author duwh
 * @date 2023-03-02
 */
@Repository
@RequiredArgsConstructor
public class ProductCategoryColumnRefDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ProductCategoryColumnRefRepo repo;
    private final QProductCategoryColumnRefDO qdo = QProductCategoryColumnRefDO.productCategoryColumnRefDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ProductCategoryColumnRefVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ProductCategoryColumnRefVO.class,
            qdo.id,
            // 产品分类ID:crm_product_category.id
            qdo.categoryId,
            // 分类名称（冗余）
            qdo.categoryName,
            // 属性表ID:crm_business_table_columns.id
            qdo.columnId,
            // 字段名称（冗余）
            qdo.columnName,
            // 字段描述（冗余）
            qdo.columnDesc,
            // 属性名称（冗余）
            qdo.attributeDesc,
            // 属性类型（冗余）
            qdo.attributeType,
            // 组件类型（冗余）
            qdo.componentType,
            // 值范围（冗余）
            qdo.attributeScope,
            // 提示（冗余）
            qdo.attributePrompt,
            // 系统选择项code（冗余）
            qdo.selectionCode,
            // 是否多选，0单选，1多选（冗余）
            qdo.isMultiple,
            // 排序号（冗余）
            qdo.sortNo,
            // 自定义排序号（分类）
            qdo.sortNoSelf,
            // 状态，0未启用，1已启用
            qdo.status
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ProductCategoryColumnRefVO> getJpaQueryWhere(ProductCategoryColumnRefQuery query) {
        JPAQuery<ProductCategoryColumnRefVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private long count(ProductCategoryColumnRefQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ProductCategoryColumnRefQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 产品分类ID:crm_product_category.id 精确 */
        if (!ObjectUtils.isEmpty(query.getCategoryId())) {
            list.add(qdo.categoryId.eq(query.getCategoryId()));
        }
        /** 分类名称（冗余） 模糊 */
        if (!ObjectUtils.isEmpty(query.getCategoryName())) {
            list.add(qdo.categoryName.like(SqlUtil.toSqlLikeString(query.getCategoryName())));
        }
        /** 属性表ID:crm_business_table_columns.id 精确 */
        if (!ObjectUtils.isEmpty(query.getColumnId())) {
            list.add(qdo.columnId.eq(query.getColumnId()));
        }
        if (!ObjectUtils.isEmpty(query.getColumnIdList())) {
            list.add(qdo.columnId.in(query.getColumnIdList()));
        }
        /** 字段名称（冗余） 模糊 */
        if (!ObjectUtils.isEmpty(query.getColumnName())) {
            list.add(qdo.columnName.like(SqlUtil.toSqlLikeString(query.getColumnName())));
        }
        /** 字段描述（冗余） 模糊 */
        if (!ObjectUtils.isEmpty(query.getColumnDesc())) {
            list.add(qdo.columnDesc.like(SqlUtil.toSqlLikeString(query.getColumnDesc())));
        }
        /** 属性名称（冗余） 模糊 */
        if (!ObjectUtils.isEmpty(query.getAttributeDesc())) {
            list.add(qdo.attributeDesc.like(SqlUtil.toSqlLikeString(query.getAttributeDesc())));
        }
        /** 属性类型（冗余） 精确 */
        if (!ObjectUtils.isEmpty(query.getAttributeType())) {
            list.add(qdo.attributeType.eq(query.getAttributeType()));
        }
        /** 组件类型（冗余） 精确 */
        if (!ObjectUtils.isEmpty(query.getComponentType())) {
            list.add(qdo.componentType.eq(query.getComponentType()));
        }
        /** 值范围（冗余） 模糊 */
        if (!ObjectUtils.isEmpty(query.getAttributeScope())) {
            list.add(qdo.attributeScope.like(SqlUtil.toSqlLikeString(query.getAttributeScope())));
        }
        /** 提示（冗余） 模糊 */
        if (!ObjectUtils.isEmpty(query.getAttributePrompt())) {
            list.add(qdo.attributePrompt.like(SqlUtil.toSqlLikeString(query.getAttributePrompt())));
        }
        /** 系统选择项code（冗余） 精确 */
        if (!ObjectUtils.isEmpty(query.getSelectionCode())) {
            list.add(qdo.selectionCode.eq(query.getSelectionCode()));
        }
        /** 是否多选，0单选，1多选（冗余） 精确 */
        if (!ObjectUtils.isEmpty(query.getIsMultiple())) {
            list.add(qdo.isMultiple.eq(query.getIsMultiple()));
        }
        /** 排序号（冗余） 精确 */
        if (!ObjectUtils.isEmpty(query.getSortNo())) {
            list.add(qdo.sortNo.eq(query.getSortNo()));
        }
        /** 自定义排序号（分类） 精确 */
        if (!ObjectUtils.isEmpty(query.getSortNoSelf())) {
            list.add(qdo.sortNoSelf.eq(query.getSortNoSelf()));
        }
        /** 状态，0未启用，1已启用 精确 */
        if (!ObjectUtils.isEmpty(query.getStatus())) {
            list.add(qdo.status.eq(query.getStatus()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ProductCategoryColumnRefVO queryByKey(Long id) {
        JPAQuery<ProductCategoryColumnRefVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ProductCategoryColumnRefVO> queryListDynamic(ProductCategoryColumnRefQuery query) {
        JPAQuery<ProductCategoryColumnRefVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ProductCategoryColumnRefVO> queryPaging(ProductCategoryColumnRefQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<ProductCategoryColumnRefVO> jpaQuery = getJpaQueryWhere(query);
        List<ProductCategoryColumnRefVO> result = jpaQuery
            .offset(query.getPageRequest().getOffset())
            .limit(query.getPageRequest().getPageSize())
            .fetch();
        return PagingVO.<ProductCategoryColumnRefVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ProductCategoryColumnRefDO save(ProductCategoryColumnRefDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ProductCategoryColumnRefDO> saveAll(List<ProductCategoryColumnRefDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ProductCategoryColumnRefPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 产品分类ID:crm_product_category.id
        if (payload.getCategoryId() != null) {
            update.set(qdo.categoryId, payload.getCategoryId());
        }
        // 分类名称（冗余）
        if (payload.getCategoryName() != null) {
            update.set(qdo.categoryName, payload.getCategoryName());
        }
        // 属性表ID:crm_business_table_columns.id
        if (payload.getColumnId() != null) {
            update.set(qdo.columnId, payload.getColumnId());
        }
        // 字段名称（冗余）
        if (payload.getColumnName() != null) {
            update.set(qdo.columnName, payload.getColumnName());
        }
        // 字段描述（冗余）
        if (payload.getColumnDesc() != null) {
            update.set(qdo.columnDesc, payload.getColumnDesc());
        }
        // 属性名称（冗余）
        if (payload.getAttributeDesc() != null) {
            update.set(qdo.attributeDesc, payload.getAttributeDesc());
        }
        // 属性类型（冗余）
        if (payload.getAttributeType() != null) {
            update.set(qdo.attributeType, payload.getAttributeType());
        }
        // 组件类型（冗余）
        if (payload.getComponentType() != null) {
            update.set(qdo.componentType, payload.getComponentType());
        }
        // 值范围（冗余）
        if (payload.getAttributeScope() != null) {
            update.set(qdo.attributeScope, payload.getAttributeScope());
        }
        // 提示（冗余）
        if (payload.getAttributePrompt() != null) {
            update.set(qdo.attributePrompt, payload.getAttributePrompt());
        }
        // 系统选择项code（冗余）
        if (payload.getSelectionCode() != null) {
            update.set(qdo.selectionCode, payload.getSelectionCode());
        }
        // 是否多选，0单选，1多选（冗余）
        if (payload.getIsMultiple() != null) {
            update.set(qdo.isMultiple, payload.getIsMultiple());
        }
        // 排序号（冗余）
        if (payload.getSortNo() != null) {
            update.set(qdo.sortNo, payload.getSortNo());
        }
        // 自定义排序号（分类）
        if (payload.getSortNoSelf() != null) {
            update.set(qdo.sortNoSelf, payload.getSortNoSelf());
        }
        // 状态，0未启用，1已启用
        if (payload.getStatus() != null) {
            update.set(qdo.status, payload.getStatus());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 产品分类ID:crm_product_category.id
            if (nullFields.contains("categoryId")) {
                update.setNull(qdo.categoryId);
            }
            // 分类名称（冗余）
            if (nullFields.contains("categoryName")) {
                update.setNull(qdo.categoryName);
            }
            // 属性表ID:crm_business_table_columns.id
            if (nullFields.contains("columnId")) {
                update.setNull(qdo.columnId);
            }
            // 字段名称（冗余）
            if (nullFields.contains("columnName")) {
                update.setNull(qdo.columnName);
            }
            // 字段描述（冗余）
            if (nullFields.contains("columnDesc")) {
                update.setNull(qdo.columnDesc);
            }
            // 属性名称（冗余）
            if (nullFields.contains("attributeDesc")) {
                update.setNull(qdo.attributeDesc);
            }
            // 属性类型（冗余）
            if (nullFields.contains("attributeType")) {
                update.setNull(qdo.attributeType);
            }
            // 组件类型（冗余）
            if (nullFields.contains("componentType")) {
                update.setNull(qdo.componentType);
            }
            // 值范围（冗余）
            if (nullFields.contains("attributeScope")) {
                update.setNull(qdo.attributeScope);
            }
            // 提示（冗余）
            if (nullFields.contains("attributePrompt")) {
                update.setNull(qdo.attributePrompt);
            }
            // 系统选择项code（冗余）
            if (nullFields.contains("selectionCode")) {
                update.setNull(qdo.selectionCode);
            }
            // 是否多选，0单选，1多选（冗余）
            if (nullFields.contains("isMultiple")) {
                update.setNull(qdo.isMultiple);
            }
            // 排序号（冗余）
            if (nullFields.contains("sortNo")) {
                update.setNull(qdo.sortNo);
            }
            // 自定义排序号（分类）
            if (nullFields.contains("sortNoSelf")) {
                update.setNull(qdo.sortNoSelf);
            }
            // 状态，0未启用，1已启用
            if (nullFields.contains("status")) {
                update.setNull(qdo.status);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

