package com.elitesland.tw.tw5crm.server.product.dao;

import com.elitesland.tw.tw5crm.api.product.payload.ProductCategoryPayload;
import com.elitesland.tw.tw5crm.api.product.query.ProductCategoryQuery;
import com.elitesland.tw.tw5crm.api.product.vo.ProductCategoryVO;
import com.elitesland.tw.tw5crm.server.product.entity.ProductCategoryDO;
import com.elitesland.tw.tw5crm.server.product.entity.QProductCategoryDO;
import com.elitesland.tw.tw5crm.server.product.repo.ProductCategoryRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 产品分类
 *
 * @author duwh
 * @date 2023-03-02
 */
@Repository
@RequiredArgsConstructor
public class ProductCategoryDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ProductCategoryRepo repo;
    private final QProductCategoryDO qdo = QProductCategoryDO.productCategoryDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ProductCategoryVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ProductCategoryVO.class,
            qdo.id,
            // 分类名称
            qdo.objName,
            // 分类编号
            qdo.objNo,
            // 父ID
            qdo.parentId,
            // 排序号
            qdo.sortNo,
            // 设置前预览
            qdo.previewBefore,
            // 设置后预览
            qdo.previewAfter,
            // 状态，0未启用，1已启用
            qdo.status
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ProductCategoryVO> getJpaQueryWhere(ProductCategoryQuery query) {
        JPAQuery<ProductCategoryVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ProductCategoryQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ProductCategoryQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        // Id 不等于
        if (!ObjectUtils.isEmpty(query.getIdNotEq())) {
            list.add(qdo.id.ne(query.getIdNotEq()));
        }
        /** 分类名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getObjName())) {
            list.add(qdo.objName.like(SqlUtil.toSqlLikeString(query.getObjName())));
        }
        /** 分类名称 精确匹配 */
        if (!ObjectUtils.isEmpty(query.getObjNameEq())) {
            list.add(qdo.objName.eq(query.getObjNameEq()));
        }
        /** 分类编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getObjNo())) {
            list.add(qdo.objNo.like(SqlUtil.toSqlLikeString(query.getObjNo())));
        }
        /** 父ID 精确 */
        if (!ObjectUtils.isEmpty(query.getParentId())) {
            list.add(qdo.parentId.eq(query.getParentId()));
        }
        /** 排序号 精确 */
        if (!ObjectUtils.isEmpty(query.getSortNo())) {
            list.add(qdo.sortNo.eq(query.getSortNo()));
        }
        /** 设置前预览 模糊 */
        if (!ObjectUtils.isEmpty(query.getPreviewBefore())) {
            list.add(qdo.previewBefore.like(SqlUtil.toSqlLikeString(query.getPreviewBefore())));
        }
        /** 设置后预览 模糊 */
        if (!ObjectUtils.isEmpty(query.getPreviewAfter())) {
            list.add(qdo.previewAfter.like(SqlUtil.toSqlLikeString(query.getPreviewAfter())));
        }
        /** 状态，0未启用，1已启用 精确 */
        if (!ObjectUtils.isEmpty(query.getStatus())) {
            list.add(qdo.status.eq(query.getStatus()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ProductCategoryVO queryByKey(Long id) {
        JPAQuery<ProductCategoryVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ProductCategoryVO> queryListDynamic(ProductCategoryQuery query) {
        JPAQuery<ProductCategoryVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ProductCategoryVO> queryPaging(ProductCategoryQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<ProductCategoryVO> jpaQuery = getJpaQueryWhere(query);
        List<ProductCategoryVO> result = jpaQuery
            .offset(query.getPageRequest().getOffset())
            .limit(query.getPageRequest().getPageSize())
            .fetch();
        return PagingVO.<ProductCategoryVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ProductCategoryDO save(ProductCategoryDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ProductCategoryDO> saveAll(List<ProductCategoryDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ProductCategoryPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 分类名称
        if (payload.getObjName() != null) {
            update.set(qdo.objName, payload.getObjName());
        }
        // 分类编号
        if (payload.getObjNo() != null) {
            update.set(qdo.objNo, payload.getObjNo());
        }
        // 父ID
        if (payload.getParentId() != null) {
            update.set(qdo.parentId, payload.getParentId());
        }
        // 排序号
        if (payload.getSortNo() != null) {
            update.set(qdo.sortNo, payload.getSortNo());
        }
        // 设置前预览
        if (payload.getPreviewBefore() != null) {
            update.set(qdo.previewBefore, payload.getPreviewBefore());
        }
        // 设置后预览
        if (payload.getPreviewAfter() != null) {
            update.set(qdo.previewAfter, payload.getPreviewAfter());
        }
        // 状态，0未启用，1已启用
        if (payload.getStatus() != null) {
            update.set(qdo.status, payload.getStatus());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 分类名称
            if (nullFields.contains("objName")) {
                update.setNull(qdo.objName);
            }
            // 分类编号
            if (nullFields.contains("objNo")) {
                update.setNull(qdo.objNo);
            }
            // 父ID
            if (nullFields.contains("parentId")) {
                update.setNull(qdo.parentId);
            }
            // 排序号
            if (nullFields.contains("sortNo")) {
                update.setNull(qdo.sortNo);
            }
            // 设置前预览
            if (nullFields.contains("previewBefore")) {
                update.setNull(qdo.previewBefore);
            }
            // 设置后预览
            if (nullFields.contains("previewAfter")) {
                update.setNull(qdo.previewAfter);
            }
            // 状态，0未启用，1已启用
            if (nullFields.contains("status")) {
                update.setNull(qdo.status);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

