package com.elitesland.tw.tw5crm.server.product.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5crm.api.product.payload.ProductPricePayload;
import com.elitesland.tw.tw5crm.api.product.query.ProductPriceQuery;
import com.elitesland.tw.tw5crm.api.product.vo.ProductPriceVO;
import com.elitesland.tw.tw5crm.server.product.entity.ProductPriceDO;
import com.elitesland.tw.tw5crm.server.product.entity.QProductPriceDO;
import com.elitesland.tw.tw5crm.server.product.entity.QProductPriceOrgDO;
import com.elitesland.tw.tw5crm.server.product.repo.ProductPriceRepo;
import com.querydsl.core.QueryResults;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.dsl.BooleanExpression;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;

/**
 * 商品价目
 *
 * @author carl
 * @date 2023-03-14
 */
@Repository
@RequiredArgsConstructor
public class ProductPriceDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ProductPriceRepo repo;
    private final QProductPriceDO qdo = QProductPriceDO.productPriceDO;

    private final QProductPriceOrgDO qdoOrg = QProductPriceOrgDO.productPriceOrgDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ProductPriceVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ProductPriceVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                qdo.procInstId,
                qdo.procInstStatus,
                qdo.submitTime,
                qdo.approvedTime,
                // 价目表编码
                qdo.priceCode,
                // 价目表名称
                qdo.priceName,
                // 负责人id
                qdo.manageUserId,
                // 归属部门
                qdo.orgId,
                // 有效期开始时间
                qdo.startTime,
                // 结束时间
                qdo.endTime,
                // 币种
                qdo.currCode,
                // 状态
                qdo.priceStatus,
                // 销售组织类型：
                qdo.saleOrgType,
                // 销售组织内容
                qdo.saleOrgIds,
                // 销售客户类型
                qdo.saleCustType,
                // 销售客户
                qdo.saleCustIds
        )).from(qdo).leftJoin(qdoOrg).on(qdo.id.eq(qdoOrg.priceId));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ProductPriceVO> getJpaQueryWhere(ProductPriceQuery query) {
        JPAQuery<ProductPriceVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
//        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ProductPriceQuery query) {
        long total = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .where(where(query))
                .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ProductPriceQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 价目表编码 模糊 */
        if (!ObjectUtils.isEmpty(query.getPriceCode())) {
            list.add(qdo.priceCode.like(SqlUtil.toSqlLikeString(query.getPriceCode())));
        }
        /** 价目表名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getPriceName())) {
            list.add(qdo.priceName.like(SqlUtil.toSqlLikeString(query.getPriceName())));
        }
        /** 价目表名称 or 价目表编码模糊 */
        if (!ObjectUtils.isEmpty(query.getKeyWords())) {
            list.add(qdo.priceName.like(SqlUtil.toSqlLikeString(query.getKeyWords()))
                    .or(qdo.priceCode.like(SqlUtil.toSqlLikeString(query.getKeyWords()))));
        }
        /** 负责人id 精确 */
        if (!ObjectUtils.isEmpty(query.getManageUserId())) {
            list.add(qdo.manageUserId.eq(query.getManageUserId()));
        }
        /** 归属部门 精确 */
        if (!ObjectUtils.isEmpty(query.getOrgId())) {
            list.add(qdo.orgId.eq(query.getOrgId()));
        }
        /** 归属部门 集合*/
        if (!ObjectUtils.isEmpty(query.getOrgIdListStr())) {
            String[] orgIds = query.getOrgIdListStr().split(",");
            List<Long> orgList = new ArrayList<>();
            for(String s :orgIds){
                Long orgId = Long.valueOf(s);
                orgList.add(orgId);
            }
            list.add(qdo.orgId.in(orgList));
        }

        /** 币种 精确 */
        if (!ObjectUtils.isEmpty(query.getCurrCode())) {
            list.add(qdo.currCode.eq(query.getCurrCode()));
        }
        /** 状态 精确 */
        if (!ObjectUtils.isEmpty(query.getPriceStatus())) {
            list.add(qdo.priceStatus.eq(query.getPriceStatus()));
        }
        /** 销售组织类型： 精确 */
        if (!ObjectUtils.isEmpty(query.getSaleOrgType())) {
            list.add(qdo.saleOrgType.eq(query.getSaleOrgType()));
        }
        /** 销售组织内容 精确 */
        if (!ObjectUtils.isEmpty(query.getSaleOrgIds())) {
            list.add(qdo.saleOrgIds.eq(query.getSaleOrgIds()));
        }
        /** 销售客户类型 精确 */
        if (!ObjectUtils.isEmpty(query.getSaleCustType())) {
            list.add(qdo.saleCustType.eq(query.getSaleCustType()));
        }
        /** 销售客户 精确 */
        if (!ObjectUtils.isEmpty(query.getSaleCustIds())) {
            list.add(qdo.saleCustIds.eq(query.getSaleCustIds()));
        }
        //需要权限
        if (query.getPermissionFlag()) {
            BooleanExpression jpaQueryOr = qdo.createUserId.eq(query.getLoginUserId()).or(qdo.manageUserId.eq(query.getLoginUserId()))
                    .or(qdoOrg.userId.eq(query.getLoginUserId())).or(qdo.saleOrgType.eq("all"));
            if (!ObjectUtils.isEmpty(query.getOrgIdList())) {
                jpaQueryOr = jpaQueryOr.or(qdo.orgId.in(query.getOrgIdList())).or(qdoOrg.orgId.in(query.getOrgIdList()));
            }
            list.add(jpaQueryOr);
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ProductPriceVO queryByKey(Long id) {
        JPAQuery<ProductPriceVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键查询
     *
     * @param ids 主键
     * @return 结果
     */
    public List<ProductPriceVO> queryByKeys(List<Long> ids) {
        JPAQuery<ProductPriceVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.in(ids));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 根据主键查询
     *
     * @param now    当前日期
     * @param status 状态
     * @return 结果
     */
    public List<ProductPriceVO> queryByDateAndStatus(LocalDate now, String status) {
        JPAQuery<ProductPriceVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.endTime.loe(now));
        jpaQuery.where(qdo.priceStatus.eq(status));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ProductPriceVO> queryListDynamic(ProductPriceQuery query) {
        JPAQuery<ProductPriceVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ProductPriceVO> queryPaging(ProductPriceQuery query) {
//        long total = count(query);
//        if (total == 0) {
//            return PagingVO.empty();
//        }
        JPAQuery<ProductPriceVO> jpaQuery = getJpaQueryWhere(query);
        QueryResults<ProductPriceVO> results = jpaQuery.groupBy(qdo.id)
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetchResults();
        return PagingVO.<ProductPriceVO>builder().records(results.getResults()).total(results.getTotal()).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ProductPriceDO save(ProductPriceDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ProductPriceDO> saveAll(List<ProductPriceDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ProductPricePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 价目表编码
        if (payload.getPriceCode() != null) {
            update.set(qdo.priceCode, payload.getPriceCode());
        }
        // 价目表名称
        if (payload.getPriceName() != null) {
            update.set(qdo.priceName, payload.getPriceName());
        }
        // 负责人id
        if (payload.getManageUserId() != null) {
            update.set(qdo.manageUserId, payload.getManageUserId());
        }
        // 归属部门
        if (payload.getOrgId() != null) {
            update.set(qdo.orgId, payload.getOrgId());
        }
        // 币种
        if (payload.getCurrCode() != null) {
            update.set(qdo.currCode, payload.getCurrCode());
        }
        // 状态
        if (payload.getPriceStatus() != null) {
            update.set(qdo.priceStatus, payload.getPriceStatus());
        }
        // 销售组织类型：
        if (payload.getSaleOrgType() != null) {
            update.set(qdo.saleOrgType, payload.getSaleOrgType());
        }

        // 销售客户类型
        if (payload.getSaleCustType() != null) {
            update.set(qdo.saleCustType, payload.getSaleCustType());
        }

        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 价目表编码
            if (nullFields.contains("priceCode")) {
                update.setNull(qdo.priceCode);
            }
            // 价目表名称
            if (nullFields.contains("priceName")) {
                update.setNull(qdo.priceName);
            }
            // 负责人id
            if (nullFields.contains("manageUserId")) {
                update.setNull(qdo.manageUserId);
            }
            // 归属部门
            if (nullFields.contains("orgId")) {
                update.setNull(qdo.orgId);
            }
            // 币种
            if (nullFields.contains("currCode")) {
                update.setNull(qdo.currCode);
            }
            // 状态
            if (nullFields.contains("priceStatus")) {
                update.setNull(qdo.priceStatus);
            }
            // 销售组织类型：
            if (nullFields.contains("saleOrgType")) {
                update.setNull(qdo.saleOrgType);
            }
            // 销售组织内容
            if (nullFields.contains("saleOrgIds")) {
                update.setNull(qdo.saleOrgIds);
            }
            // 销售客户类型
            if (nullFields.contains("saleCustType")) {
                update.setNull(qdo.saleCustType);
            }
            // 销售客户
            if (nullFields.contains("saleCustIds")) {
                update.setNull(qdo.saleCustIds);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 变更工作流相关数据
     *
     * @param payload
     * @return
     */

    public long updateWorkFlow(ProductPricePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));

        // 流程实例id
        if (!ObjectUtils.isEmpty(payload.getProcInstId())) {
            update.set(qdo.procInstId, payload.getProcInstId());
        } else {
            update.setNull(qdo.procInstId);
        }
        // 审批状态
        if (!ObjectUtils.isEmpty(payload.getProcInstStatus())) {
            update.set(qdo.procInstStatus, payload.getProcInstStatus());
        } else {
            update.setNull(qdo.procInstStatus);
        }
        // 状态
        if (!ObjectUtils.isEmpty(payload.getPriceStatus())) {
            update.set(qdo.priceStatus, payload.getPriceStatus());
        } else {
            update.setNull(qdo.priceStatus);
        }
        // 提审时间
        if (!ObjectUtils.isEmpty(payload.getSubmitTime())) {
            update.set(qdo.submitTime, payload.getSubmitTime());
        }
        // 审批结束时间
        if (!ObjectUtils.isEmpty(payload.getApprovedTime())) {
            update.set(qdo.approvedTime, payload.getApprovedTime());
        }
        if (!ObjectUtils.isEmpty(payload.getDeleteFlag())) {
            update.set(qdo.deleteFlag, payload.getDeleteFlag());
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 修改状态
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long updateStatus(List<Long> keys, String status) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.priceStatus, status)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    public long updateStatusForXxxJob(List<Long> keys, String status) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.priceStatus, status)
            .where(qdo.id.in(keys));
        //拼装更新
        //SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
}

