package com.elitesland.tw.tw5crm.server.product.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.WorkFlowStatusEnum;
import com.elitesland.tw.tw5crm.api.product.payload.ProductPriceDetailPayload;
import com.elitesland.tw.tw5crm.api.product.query.ProductPriceDetailQuery;
import com.elitesland.tw.tw5crm.api.product.vo.ProductPriceDetailVO;
import com.elitesland.tw.tw5crm.server.product.entity.*;
import com.elitesland.tw.tw5crm.server.product.repo.ProductPriceDetailRepo;
import com.querydsl.core.QueryResults;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.dsl.BooleanExpression;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 商品价目明细
 *
 * @author carl
 * @date 2023-03-14
 */
@Repository
@RequiredArgsConstructor
public class ProductPriceDetailDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ProductPriceDetailRepo repo;
    private final QProductPriceDetailDO qdo = QProductPriceDetailDO.productPriceDetailDO;
    private final QProductPriceDO qdoPrice = QProductPriceDO.productPriceDO;
    private final QProductPriceCustomerDO qdoCustomer = QProductPriceCustomerDO.productPriceCustomerDO;
    private final QProductPriceOrgDO qdoOrg = QProductPriceOrgDO.productPriceOrgDO;
    private final QProductSkuDO qdoSku = QProductSkuDO.productSkuDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ProductPriceDetailVO> getJpaQuerySelect0() {
        return jpaQueryFactory.select(Projections.bean(ProductPriceDetailVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 价目主键
                qdo.priceId,

                qdoSku.spuCategory,
                qdoSku.categoryId,
                qdoSku.spuType,
                qdoSku.attributeStr,

                qdo.skuName,
                qdo.skuCode,
                qdo.standardPrice,
                qdo.saleUnit,
                qdo.isChange,
                // sku主键
                qdo.skuId,
                // 折扣
                qdo.discount,
                // 价目表价格
                qdo.markedMoney,
                // 价格上限
                qdo.topMoney,
                // 价格下限
                qdo.floorMoney
            )).from(qdo)
            .leftJoin(qdoSku)
            .on(qdo.skuId.eq(qdoSku.id));
    }

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ProductPriceDetailVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ProductPriceDetailVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 价目主键
                qdo.priceId,

                qdoSku.skuName,
                qdoSku.spuCategory,
                qdoSku.categoryId,
                qdoSku.spuType,
                qdoSku.attributeStr,
                qdoSku.spuId,
                qdo.skuCode,
                qdo.standardPrice,
                qdo.saleUnit,
                qdo.isChange,
                // sku主键
                qdo.skuId,
                // 折扣
                qdo.discount,
                // 价目表价格（只能参考）
                qdo.markedMoney,
                // 价格上限
                qdo.topMoney,
                // 价格下限
                qdo.floorMoney,
                qdoPrice.priceName,
                qdoPrice.currCode,
                qdoPrice.orgId,
                qdoCustomer.custId,
                qdoCustomer.customerName

        )).from(qdo).leftJoin(qdoPrice).on(qdo.priceId.eq(qdoPrice.id))
                .leftJoin(qdoCustomer).on(qdo.priceId.eq(qdoCustomer.priceId))
                .leftJoin(qdoOrg).on(qdo.priceId.eq(qdoOrg.priceId))
                .leftJoin(qdoSku).on(qdo.skuId.eq(qdoSku.id));
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ProductPriceDetailVO> queryPaging(ProductPriceDetailQuery query) {
//        long total = count(query);
//        if (total == 0) {
//            return PagingVO.empty();
//        }
        JPAQuery<ProductPriceDetailVO> jpaQuery = getJpaQueryWhere(query);
        QueryResults<ProductPriceDetailVO> results = jpaQuery.groupBy(qdo.id)
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetchResults();
        return PagingVO.<ProductPriceDetailVO>builder().records(results.getResults()).total(results.getTotal()).build();
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ProductPriceDetailVO> getJpaQueryWhere(ProductPriceDetailQuery query) {
        JPAQuery<ProductPriceDetailVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));

//        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ProductPriceDetailQuery query) {
        long total = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .where(where(query))
                .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ProductPriceDetailQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 价目主键 精确 */
        if (!ObjectUtils.isEmpty(query.getPriceId())) {
            list.add(qdo.priceId.eq(query.getPriceId()));
        }
        /** sku主键 精确 */
        if (!ObjectUtils.isEmpty(query.getSkuId())) {
            list.add(qdo.skuId.eq(query.getSkuId()));
        }
        /** 折扣 精确 */
        if (!ObjectUtils.isEmpty(query.getDiscount())) {
            list.add(qdo.discount.eq(query.getDiscount()));
        }
        /** 价目表价格（只能参考） 精确 */
        if (!ObjectUtils.isEmpty(query.getMarkedMoney())) {
            list.add(qdo.markedMoney.eq(query.getMarkedMoney()));
        }
        /** 价格上限 精确 */
        if (!ObjectUtils.isEmpty(query.getTopMoney())) {
            list.add(qdo.topMoney.eq(query.getTopMoney()));
        }
        /** 价格下限 精确 */
        if (!ObjectUtils.isEmpty(query.getFloorMoney())) {
            list.add(qdo.floorMoney.eq(query.getFloorMoney()));
        }
        //需要权限
        if (query.getPermissionFlag()) {
            BooleanExpression jpaQueryOr = qdo.createUserId.eq(query.getLoginUserId()).or(qdoPrice.manageUserId.eq(query.getLoginUserId()))
                    .or(qdoOrg.userId.eq(query.getLoginUserId())).or(qdoPrice.saleOrgType.eq("all"));
            if (!ObjectUtils.isEmpty(query.getOrgIdList())) {
                jpaQueryOr = jpaQueryOr.or(qdoPrice.orgId.in(query.getOrgIdList())).or(qdoOrg.orgId.in(query.getOrgIdList()));
            }
            list.add(jpaQueryOr);
        }
        /** 指定客户 */
        if (!ObjectUtils.isEmpty(query.getCustId())) {
            list.add(qdoCustomer.custId.eq(query.getCustId()));
        }
        /** 指定产品 */
        if (!ObjectUtils.isEmpty(query.getSkuIds())) {
            list.add(qdo.skuId.in(query.getSkuIds()));
        }
        /** 指定产品 */
        if (!ObjectUtils.isEmpty(query.getSkuName())) {
            list.add(qdoSku.skuName.like(SqlUtil.toSqlLikeString(query.getSkuName())).or(qdoSku.skuCode.like(SqlUtil.toSqlLikeString(query.getSkuName()))));
        }
        /** 指定产品类型*/
        if (!ObjectUtils.isEmpty(query.getSpuType())) {
            list.add(qdoSku.spuType.eq(query.getSpuType()));
        }
        /** 指定产品类型*/
        if (!ObjectUtils.isEmpty(query.getCategoryId())) {
            list.add(qdoSku.categoryId.eq(query.getCategoryId()));
        }
        //价目表必须是激活状态
        list.add(qdoPrice.priceStatus.eq(WorkFlowStatusEnum.APPROVED_WORK.getCode()));
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ProductPriceDetailVO queryByKey(Long id) {
        JPAQuery<ProductPriceDetailVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据价目表主键查询
     *
     * @param priceIds 价目表主键
     * @return 结果
     */
    public List<ProductPriceDetailVO> queryByPriceIds(List<Long> priceIds) {
        JPAQuery<ProductPriceDetailVO> jpaQuery = getJpaQuerySelect0();
        jpaQuery.where(qdo.priceId.in(priceIds));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 根据sku主键查询
     *
     * @param skuIds 价目表主键
     * @return 结果
     */
    public List<ProductPriceDetailDO> queryBySkuIds(List<Long> skuIds) {
        JPAQuery<ProductPriceDetailDO> jpaQuery = jpaQueryFactory.select(qdo).from(qdo);
        jpaQuery.where(qdo.skuId.in(skuIds));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ProductPriceDetailVO> queryListDynamic(ProductPriceDetailQuery query) {
        JPAQuery<ProductPriceDetailVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }


    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ProductPriceDetailDO save(ProductPriceDetailDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ProductPriceDetailDO> saveAll(List<ProductPriceDetailDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ProductPriceDetailPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 价目主键
        if (payload.getPriceId() != null) {
            update.set(qdo.priceId, payload.getPriceId());
        }
        // sku主键
        if (payload.getSkuId() != null) {
            update.set(qdo.skuId, payload.getSkuId());
        }
        // 折扣
        if (payload.getDiscount() != null) {
            update.set(qdo.discount, payload.getDiscount());
        }
        // 价目表价格（只能参考）
        if (payload.getMarkedMoney() != null) {
            update.set(qdo.markedMoney, payload.getMarkedMoney());
        }
        // 价格上限
        if (payload.getTopMoney() != null) {
            update.set(qdo.topMoney, payload.getTopMoney());
        }
        // 价格下限
        if (payload.getFloorMoney() != null) {
            update.set(qdo.floorMoney, payload.getFloorMoney());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 价目主键
            if (nullFields.contains("priceId")) {
                update.setNull(qdo.priceId);
            }
            // sku主键
            if (nullFields.contains("skuId")) {
                update.setNull(qdo.skuId);
            }
            // 折扣
            if (nullFields.contains("discount")) {
                update.setNull(qdo.discount);
            }
            // 价目表价格（只能参考）
            if (nullFields.contains("markedMoney")) {
                update.setNull(qdo.markedMoney);
            }
            // 价格上限
            if (nullFields.contains("topMoney")) {
                update.setNull(qdo.topMoney);
            }
            // 价格下限
            if (nullFields.contains("floorMoney")) {
                update.setNull(qdo.floorMoney);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param priceIds 主集合
     * @return 删除的行数
     */
    public long deleteSoftByPriceIds(List<Long> priceIds) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.priceId.in(priceIds));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据skuid逻辑删除
     *
     * @param skuIds 主集合
     * @return 删除的行数
     */
    public long deleteSoftBySkuIds(List<Long> skuIds) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.skuId.in(skuIds));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
}

