package com.elitesland.tw.tw5crm.server.product.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5crm.api.product.payload.ProductSkuPayload;
import com.elitesland.tw.tw5crm.api.product.query.ProductSkuQuery;
import com.elitesland.tw.tw5crm.api.product.vo.ProductSkuVO;
import com.elitesland.tw.tw5crm.server.product.entity.ProductSkuDO;
import com.elitesland.tw.tw5crm.server.product.entity.QProductSkuDO;
import com.elitesland.tw.tw5crm.server.product.repo.ProductSkuRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 商品规格组合表SKU
 *
 * @author carl.wang
 * @date 2023-03-03
 */
@Repository
@RequiredArgsConstructor
public class ProductSkuDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ProductSkuRepo repo;
    private final QProductSkuDO qdo = QProductSkuDO.productSkuDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ProductSkuVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ProductSkuVO.class,
                qdo.id,
                // 产品id
                qdo.spuId,
                // 产品分类ID:crm_product_category.id(冗余)
                qdo.categoryId,
                // 产品名称
                qdo.skuName,
                // 产品sku编号
                qdo.skuCode,
                // 标准价格
                qdo.standardPrice,
                // 销售单位udc[crm:product:sku_sale_unit]
                qdo.saleUnit,
                // 图片地址codes（“，”分割字符串）
                qdo.imgCodes,
                //产品类型
                qdo.spuType,
                //产品分类
                qdo.spuCategory,
                // 规格属性Json
                qdo.attribute,
                // 属性组Json
                qdo.attributeGroup,
                // 图文内容
                qdo.skuContent,
                // 文档地址codes（“，”分割字符串）
                qdo.fileCodes,
                // 状态udc[crm:product:sku_status]
                qdo.skuStatus,
                // 排序号
                qdo.sortNo,
                // 规格属性展示
                qdo.attributeStr
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ProductSkuVO> getJpaQueryWhere(ProductSkuQuery query) {
        JPAQuery<ProductSkuVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
//        // 常用基础查询条件拼装
//        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ProductSkuQuery query) {
        long total = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .where(where(query))
                .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ProductSkuQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 产品id 精确 */
        if (!ObjectUtils.isEmpty(query.getSpuId())) {
            list.add(qdo.spuId.eq(query.getSpuId()));
        }
        /** 产品分类ID:crm_product_category.id(冗余) 精确 */
        if (!ObjectUtils.isEmpty(query.getCategoryId())) {
            list.add(qdo.categoryId.eq(query.getCategoryId()));
        }
        /** 产品名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getSkuName())) {
            list.add(qdo.skuName.like(SqlUtil.toSqlLikeString(query.getSkuName())));
        }
        /** 产品sku编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getSkuCode())) {
            list.add(qdo.skuCode.like(SqlUtil.toSqlLikeString(query.getSkuCode())));
        }
        /** 标准价格 精确 */
        if (!ObjectUtils.isEmpty(query.getStandardPrice())) {
            list.add(qdo.standardPrice.eq(query.getStandardPrice()));
        }
        /** 销售单位udc[crm:product:sku_sale_unit] 精确 */
        if (!ObjectUtils.isEmpty(query.getSaleUnit())) {
            list.add(qdo.saleUnit.eq(query.getSaleUnit()));
        }
        /** 图片地址codes（“，”分割字符串） 模糊 */
        if (!ObjectUtils.isEmpty(query.getImgCodes())) {
            list.add(qdo.imgCodes.like(SqlUtil.toSqlLikeString(query.getImgCodes())));
        }
        /** 规格属性Json 模糊 */
        if (!ObjectUtils.isEmpty(query.getAttribute())) {
            list.add(qdo.attribute.like(SqlUtil.toSqlLikeString(query.getAttribute())));
        }
        /** 属性组Json 模糊 */
        if (!ObjectUtils.isEmpty(query.getAttributeGroup())) {
            list.add(qdo.attributeGroup.like(SqlUtil.toSqlLikeString(query.getAttributeGroup())));
        }
        /** 图文内容 模糊 */
        if (!ObjectUtils.isEmpty(query.getSkuContent())) {
            list.add(qdo.skuContent.like(SqlUtil.toSqlLikeString(query.getSkuContent())));
        }
        /** 文档地址codes（“，”分割字符串） 模糊 */
        if (!ObjectUtils.isEmpty(query.getFileCodes())) {
            list.add(qdo.fileCodes.like(SqlUtil.toSqlLikeString(query.getFileCodes())));
        }
        /** 状态udc[crm:product:sku_status] 精确 */
        if (!ObjectUtils.isEmpty(query.getSkuStatus())) {
            list.add(qdo.skuStatus.eq(query.getSkuStatus()));
        }
        /** 排序号 精确 */
        if (!ObjectUtils.isEmpty(query.getSortNo())) {
            list.add(qdo.sortNo.eq(query.getSortNo()));
        }
        /** 规格属性展示 模糊 */
        if (!ObjectUtils.isEmpty(query.getAttributeStr())) {
            list.add(qdo.attributeStr.like(SqlUtil.toSqlLikeString(query.getAttributeStr())));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ProductSkuVO queryByKey(Long id) {
        JPAQuery<ProductSkuVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键查询
     *
     * @param ids 主键s
     * @return 结果
     */
    public List<ProductSkuVO> queryByKeys(List<Long> ids) {
        JPAQuery<ProductSkuVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.in(ids));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ProductSkuVO> queryListDynamic(ProductSkuQuery query) {
        JPAQuery<ProductSkuVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 通过spuIds查询
     *
     * @param spuIds 查询参数
     * @return 结果集合
     */
    public List<ProductSkuVO> queryListBySpuIds(List<Long> spuIds) {
        JPAQuery<ProductSkuVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(qdo.spuId.in(spuIds)).where(qdo.deleteFlag.eq(0));

        // 动态排序
        List<OrderItem> orderse = new ArrayList<>();
        orderse.add(OrderItem.asc("sortNo"));
        orderse.add(OrderItem.desc("createTime"));
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, orderse));
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ProductSkuVO> queryPaging(ProductSkuQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<ProductSkuVO> jpaQuery = getJpaQueryWhere(query);
        List<ProductSkuVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ProductSkuVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ProductSkuDO save(ProductSkuDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ProductSkuDO> saveAll(List<ProductSkuDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ProductSkuPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 产品id
        if (payload.getSpuId() != null) {
            update.set(qdo.spuId, payload.getSpuId());
        }
        // 产品分类ID:crm_product_category.id(冗余)
        if (payload.getCategoryId() != null) {
            update.set(qdo.categoryId, payload.getCategoryId());
        }
        // 产品名称
        if (payload.getSkuName() != null) {
            update.set(qdo.skuName, payload.getSkuName());
        }
        // 产品sku编号
        if (payload.getSkuCode() != null) {
            update.set(qdo.skuCode, payload.getSkuCode());
        }
        // 标准价格
        if (payload.getStandardPrice() != null) {
            update.set(qdo.standardPrice, payload.getStandardPrice());
        }
        // 销售单位udc[crm:product:sku_sale_unit]
        if (payload.getSaleUnit() != null) {
            update.set(qdo.saleUnit, payload.getSaleUnit());
        }
        // 图片地址codes（“，”分割字符串）
        if (payload.getImgCodes() != null) {
            update.set(qdo.imgCodes, payload.getImgCodes());
        }
        // 规格属性Json
//        if (payload.getAttribute() != null) {
//            update.set(qdo.attribute, payload.getAttribute());
//        }
//                // 属性组Json
//        if (payload.getAttributeGroup() != null) {
//            update.set(qdo.attributeGroup, payload.getAttributeGroup());
//        }
        // 图文内容
        if (payload.getSkuContent() != null) {
            update.set(qdo.skuContent, payload.getSkuContent());
        }
        // 文档地址codes（“，”分割字符串）
        if (payload.getFileCodes() != null) {
            update.set(qdo.fileCodes, payload.getFileCodes());
        }
        // 状态udc[crm:product:sku_status]
        if (payload.getSkuStatus() != null) {
            update.set(qdo.skuStatus, payload.getSkuStatus());
        }
        // 排序号
        if (payload.getSortNo() != null) {
            update.set(qdo.sortNo, payload.getSortNo());
        }
        // 规格属性展示
        if (payload.getAttributeStr() != null) {
            update.set(qdo.attributeStr, payload.getAttributeStr());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 产品id
            if (nullFields.contains("spuId")) {
                update.setNull(qdo.spuId);
            }
            // 产品分类ID:crm_product_category.id(冗余)
            if (nullFields.contains("categoryId")) {
                update.setNull(qdo.categoryId);
            }
            // 产品名称
            if (nullFields.contains("skuName")) {
                update.setNull(qdo.skuName);
            }
            // 产品sku编号
            if (nullFields.contains("skuCode")) {
                update.setNull(qdo.skuCode);
            }
            // 标准价格
            if (nullFields.contains("standardPrice")) {
                update.setNull(qdo.standardPrice);
            }
            // 销售单位udc[crm:product:sku_sale_unit]
            if (nullFields.contains("saleUnit")) {
                update.setNull(qdo.saleUnit);
            }
            // 图片地址codes（“，”分割字符串）
            if (nullFields.contains("imgCodes")) {
                update.setNull(qdo.imgCodes);
            }
            // 规格属性Json
            if (nullFields.contains("attribute")) {
                update.setNull(qdo.attribute);
            }
            // 属性组Json
            if (nullFields.contains("attributeGroup")) {
                update.setNull(qdo.attributeGroup);
            }
            // 图文内容
            if (nullFields.contains("skuContent")) {
                update.setNull(qdo.skuContent);
            }
            // 文档地址codes（“，”分割字符串）
            if (nullFields.contains("fileCodes")) {
                update.setNull(qdo.fileCodes);
            }
            // 状态udc[crm:product:sku_status]
            if (nullFields.contains("skuStatus")) {
                update.setNull(qdo.skuStatus);
            }
            // 排序号
            if (nullFields.contains("sortNo")) {
                update.setNull(qdo.sortNo);
            }
            // 规格属性展示
            if (nullFields.contains("attributeStr")) {
                update.setNull(qdo.attributeStr);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 修改上下架状态
     *
     * @return
     */
    public long updateStatus(ProductSkuPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.skuStatus, payload.getSkuStatus())
                .where(qdo.id.eq(payload.getId()));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

