package com.elitesland.tw.tw5crm.server.product.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5crm.api.product.payload.ProductSpuPayload;
import com.elitesland.tw.tw5crm.api.product.query.ProductSpuQuery;
import com.elitesland.tw.tw5crm.api.product.vo.ProductSpuVO;
import com.elitesland.tw.tw5crm.server.product.entity.ProductSpuDO;
import com.elitesland.tw.tw5crm.server.product.entity.QProductSpuDO;
import com.elitesland.tw.tw5crm.server.product.repo.ProductSpuRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.dsl.BooleanExpression;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 商品主表SPU
 *
 * @author carl.wang
 * @date 2023-03-03
 */
@Repository
@RequiredArgsConstructor
public class ProductSpuDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ProductSpuRepo repo;
    private final QProductSpuDO qdo = QProductSpuDO.productSpuDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ProductSpuVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ProductSpuVO.class,
                qdo.id,
                qdo.createTime,
                qdo.createUserId,
                // 产品分类ID:crm_product_category.id
                qdo.categoryId,
                // 产品分类
                qdo.spuCategory,
                // 产品名称
                qdo.spuName,
                // 产品编号
                qdo.spuCode,
                // 简介
                qdo.spuProfile,
                // 产品类型
                qdo.spuType,
                // 负责人id
                qdo.manageUserId,
                // 归属部门
                qdo.orgId,
                // 图片地址codes（“，”分割字符串）
                qdo.imgCodes,
                // 规格属性Json
                qdo.attribute,
                // 属性组Json
                qdo.attributeGroup,
                // 状态udc[crm:product:spu_status]
                qdo.spuStatus,
                // 排序号
                qdo.sortNo
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ProductSpuVO> getJpaQueryWhere(ProductSpuQuery query) {
        JPAQuery<ProductSpuVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
//        // 常用基础查询条件拼装
//        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ProductSpuQuery query) {
        long total = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .where(where(query))
                .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ProductSpuQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 产品分类ID:crm_product_category.id 精确 */
        if (!ObjectUtils.isEmpty(query.getCategoryId())) {
            list.add(qdo.categoryId.eq(query.getCategoryId()));
        }

        /** 产品名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getSpuName())) {
            list.add(qdo.spuName.like(SqlUtil.toSqlLikeString(query.getSpuName())));
        }
        /** 产品编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getSpuCode())) {
            list.add(qdo.spuCode.like(SqlUtil.toSqlLikeString(query.getSpuCode())));
        }
        /** 简介 模糊 */
        if (!ObjectUtils.isEmpty(query.getSpuProfile())) {
            list.add(qdo.spuProfile.like(SqlUtil.toSqlLikeString(query.getSpuProfile())));
        }
        /** 产品类型 精确 */
        if (!ObjectUtils.isEmpty(query.getSpuType())) {
            list.add(qdo.spuType.eq(query.getSpuType()));
        }
        /** 负责人id 精确 */
        if (!ObjectUtils.isEmpty(query.getManageUserId())) {
            list.add(qdo.manageUserId.eq(query.getManageUserId()));
        }
        /** 归属部门 精确 */
        if (!ObjectUtils.isEmpty(query.getOrgId())) {
            list.add(qdo.orgId.eq(query.getOrgId()));
        }
        /** 图片地址codes（“，”分割字符串） 模糊 */
        if (!ObjectUtils.isEmpty(query.getImgCodes())) {
            list.add(qdo.imgCodes.like(SqlUtil.toSqlLikeString(query.getImgCodes())));
        }
        /** 规格属性Json 模糊 */
        if (!ObjectUtils.isEmpty(query.getAttribute())) {
            list.add(qdo.attribute.like(SqlUtil.toSqlLikeString(query.getAttribute())));
        }
        /** 属性组Json 模糊 */
        if (!ObjectUtils.isEmpty(query.getAttributeGroup())) {
            list.add(qdo.attributeGroup.like(SqlUtil.toSqlLikeString(query.getAttributeGroup())));
        }
        /** 状态udc[crm:product:spu_status] 精确 */
        if (!ObjectUtils.isEmpty(query.getSpuStatus())) {
            list.add(qdo.spuStatus.eq(query.getSpuStatus()));
        }
        /** 排序号 精确 */
        if (!ObjectUtils.isEmpty(query.getSortNo())) {
            list.add(qdo.sortNo.eq(query.getSortNo()));
        }
        //需要权限
        if (query.getPermissionFlag()) {
            BooleanExpression jpaQueryOr = qdo.createUserId.eq(query.getCreateUserId()).or(qdo.manageUserId.eq(query.getCreateUserId()));
            if (!ObjectUtils.isEmpty(query.getOrgIdList())) {
                jpaQueryOr = jpaQueryOr.or(qdo.orgId.in(query.getOrgIdList()));
            }
            list.add(jpaQueryOr);
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ProductSpuVO queryByKey(Long id) {
        JPAQuery<ProductSpuVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ProductSpuVO> queryListDynamic(ProductSpuQuery query) {
        JPAQuery<ProductSpuVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ProductSpuVO> queryPaging(ProductSpuQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<ProductSpuVO> jpaQuery = getJpaQueryWhere(query);
        List<ProductSpuVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ProductSpuVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ProductSpuDO save(ProductSpuDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ProductSpuDO> saveAll(List<ProductSpuDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ProductSpuPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 产品分类ID:crm_product_category.id
        if (payload.getCategoryId() != null) {
            update.set(qdo.categoryId, payload.getCategoryId());
        }
        // 产品分类
        if (payload.getSpuCategory() != null) {
            update.set(qdo.spuCategory, payload.getSpuCategory());
        }
        // 产品名称
        if (payload.getSpuName() != null) {
            update.set(qdo.spuName, payload.getSpuName());
        }
        // 产品编号
        if (payload.getSpuCode() != null) {
            update.set(qdo.spuCode, payload.getSpuCode());
        }
        // 简介
        if (payload.getSpuProfile() != null) {
            update.set(qdo.spuProfile, payload.getSpuProfile());
        }
        // 产品类型
        if (payload.getSpuType() != null) {
            update.set(qdo.spuType, payload.getSpuType());
        }
        // 负责人id
        if (payload.getManageUserId() != null) {
            update.set(qdo.manageUserId, payload.getManageUserId());
        }
        // 归属部门
        if (payload.getOrgId() != null) {
            update.set(qdo.orgId, payload.getOrgId());
        }
        // 图片地址codes（“，”分割字符串）
        if (payload.getImgCodes() != null) {
            update.set(qdo.imgCodes, payload.getImgCodes());
        }
        // 规格属性Json
//        if (payload.getAttribute() != null) {
//            update.set(qdo.attribute, payload.getAttribute());
//        }
//                // 属性组Json
//        if (payload.getAttributeGroup() != null) {
//            update.set(qdo.attributeGroup, payload.getAttributeGroup());
//        }
        // 状态udc[crm:product:spu_status]
        if (payload.getSpuStatus() != null) {
            update.set(qdo.spuStatus, payload.getSpuStatus());
        }
        // 排序号
        if (payload.getSortNo() != null) {
            update.set(qdo.sortNo, payload.getSortNo());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 产品分类ID:crm_product_category.id
            if (nullFields.contains("categoryId")) {
                update.setNull(qdo.categoryId);
            }
            // 产品分类
            if (nullFields.contains("spuCategory")) {
                update.setNull(qdo.spuCategory);
            }
            // 产品名称
            if (nullFields.contains("spuName")) {
                update.setNull(qdo.spuName);
            }
            // 产品编号
            if (nullFields.contains("spuCode")) {
                update.setNull(qdo.spuCode);
            }
            // 简介
            if (nullFields.contains("spuProfile")) {
                update.setNull(qdo.spuProfile);
            }
            // 产品类型
            if (nullFields.contains("spuType")) {
                update.setNull(qdo.spuType);
            }
            // 负责人id
            if (nullFields.contains("manageUserId")) {
                update.setNull(qdo.manageUserId);
            }
            // 归属部门
            if (nullFields.contains("orgId")) {
                update.setNull(qdo.orgId);
            }
            // 图片地址codes（“，”分割字符串）
            if (nullFields.contains("imgCodes")) {
                update.setNull(qdo.imgCodes);
            }
            // 规格属性Json
            if (nullFields.contains("attribute")) {
                update.setNull(qdo.attribute);
            }
            // 属性组Json
            if (nullFields.contains("attributeGroup")) {
                update.setNull(qdo.attributeGroup);
            }
            // 状态udc[crm:product:spu_status]
            if (nullFields.contains("spuStatus")) {
                update.setNull(qdo.spuStatus);
            }
            // 排序号
            if (nullFields.contains("sortNo")) {
                update.setNull(qdo.sortNo);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        // SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

