package com.elitesland.tw.tw5crm.server.product.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.util.PageUtil;
import com.elitesland.tw.tw5crm.api.product.payload.ProductCategoryAttrGroupDetailRefPayload;
import com.elitesland.tw.tw5crm.api.product.payload.ProductCategoryAttrGroupRefListPayload;
import com.elitesland.tw.tw5crm.api.product.payload.ProductCategoryAttrGroupRefPayload;
import com.elitesland.tw.tw5crm.api.product.query.*;
import com.elitesland.tw.tw5crm.api.product.service.ProductCategoryAttrGroupDetailRefService;
import com.elitesland.tw.tw5crm.api.product.service.ProductCategoryAttrGroupRefService;
import com.elitesland.tw.tw5crm.api.product.vo.*;
import com.elitesland.tw.tw5crm.server.product.convert.ProductCategoryAttrGroupRefConvert;
import com.elitesland.tw.tw5crm.server.product.dao.*;
import com.elitesland.tw.tw5crm.server.product.entity.ProductCategoryAttrGroupRefDO;
import com.elitesland.tw.tw5crm.server.product.repo.ProductCategoryAttrGroupRefRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 产品分类属性组关系表
 *
 * @author duwh
 * @date 2023-03-03
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class ProductCategoryAttrGroupRefServiceImpl implements ProductCategoryAttrGroupRefService {

    private final ProductCategoryAttrGroupRefRepo productCategoryAttrGroupRefRepo;
    private final ProductCategoryAttrGroupDetailRefService productCategoryAttrGroupDetailRefService;
    private final ProductCategoryAttrGroupDetailRefDAO productCategoryAttrGroupDetailRefDAO;
    private final ProductCategoryColumnRefDAO productCategoryColumnRefDAO;
    private final CrmBusinessAttributeGroupDAO crmBusinessAttributeGroupDAO;
    private final CrmBusinessAttributeGroupDetailDAO crmBusinessAttributeGroupDetailDAO;
    private final CrmBusinessTableColumnsDAO crmBusinessTableColumnsDAO;
    private final ProductCategoryDAO productCategoryDAO;
    private final ProductSpuDAO productSpuDAO;
    private final ProductSkuDAO productSkuDAO;

    @Override
    public PagingVO<ProductCategoryAttrGroupRefVO> paging(ProductCategoryAttrGroupRefQuery query) {
        Page<ProductCategoryAttrGroupRefDO> page = productCategoryAttrGroupRefRepo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder), query.getPageRequest());
        final Page<ProductCategoryAttrGroupRefVO> pageVo = page.map(ProductCategoryAttrGroupRefConvert.INSTANCE::toVo);
        pageVo.getContent().forEach(groupRefVO -> {
            // 实时获取
            detailsProcessDynamic(groupRefVO);
            // 冗余方式获取
            //detailsProcessRedundant(groupRefVO);
        });
        return PageUtil.toPageVo(pageVo);
    }

    /**
     * 实时获取 属性组下的属性
     *
     * @param groupRefVO 属性组关系
     */
    private void detailsProcessDynamic(ProductCategoryAttrGroupRefVO groupRefVO) {
        final List<CrmBusinessAttributeGroupDetailVO> attributeGroupDetailVOList = crmBusinessAttributeGroupDetailDAO.queryByGroupId(groupRefVO.getGroupId());
        final List<ProductCategoryAttrGroupDetailRefVO> attrGroupDetailRefVOList = attributeGroupDetailVOList.stream()
            .filter(crmBusinessAttributeGroupDetailVO -> null != crmBusinessAttributeGroupDetailVO.getStatus() && crmBusinessAttributeGroupDetailVO.getStatus().equals(1))
            .map(crmBusinessAttributeGroupDetailVO -> {
                ProductCategoryAttrGroupDetailRefVO attrGroupDetailRefVO = new ProductCategoryAttrGroupDetailRefVO();
                attrGroupDetailRefVO.setGroupRefId(groupRefVO.getId());
                attrGroupDetailRefVO.setCategoryId(groupRefVO.getCategoryId());
                attrGroupDetailRefVO.setCategoryName(groupRefVO.getCategoryName());
                attrGroupDetailRefVO.setGroupId(groupRefVO.getGroupId());
                attrGroupDetailRefVO.setGroupName(groupRefVO.getGroupName());
                // 根据属性id获取 其信息 冗余起来
                final Long attributeId = crmBusinessAttributeGroupDetailVO.getAttributeId();
                attrGroupDetailRefVO.setAttributeId(attributeId);

                // 方式2 读取最新的属性信息 进行属性信息冗余
                final CrmBusinessTableColumnsVO columnsVO = crmBusinessTableColumnsDAO.findByKey(attributeId);
                Assert.notNull(columnsVO, "属性不存在");
                if (ObjectUtils.isEmpty(attrGroupDetailRefVO.getColumnName())) {
                    attrGroupDetailRefVO.setColumnName(columnsVO.getColumnName());
                }
                if (ObjectUtils.isEmpty(attrGroupDetailRefVO.getColumnDesc())) {
                    attrGroupDetailRefVO.setColumnDesc(columnsVO.getColumnDesc());
                }
                if (ObjectUtils.isEmpty(attrGroupDetailRefVO.getAttributeDesc())) {
                    attrGroupDetailRefVO.setAttributeDesc(columnsVO.getAttributeDesc());
                }
                if (ObjectUtils.isEmpty(attrGroupDetailRefVO.getAttributeType())) {
                    attrGroupDetailRefVO.setAttributeType(columnsVO.getAttributeType());
                }
                if (ObjectUtils.isEmpty(attrGroupDetailRefVO.getComponentType())) {
                    attrGroupDetailRefVO.setComponentType(columnsVO.getComponentType());
                }
                if (ObjectUtils.isEmpty(attrGroupDetailRefVO.getAttributeScope())) {
                    attrGroupDetailRefVO.setAttributeScope(columnsVO.getAttributeScope());
                }
                if (ObjectUtils.isEmpty(attrGroupDetailRefVO.getAttributePrompt())) {
                    attrGroupDetailRefVO.setAttributePrompt(columnsVO.getAttributePrompt());
                }
                if (ObjectUtils.isEmpty(attrGroupDetailRefVO.getSelectionCode())) {
                    attrGroupDetailRefVO.setSelectionCode(columnsVO.getSelectionCode());
                }
                if (ObjectUtils.isEmpty(attrGroupDetailRefVO.getIsMultiple())) {
                    attrGroupDetailRefVO.setIsMultiple(columnsVO.getIsMultiple());
                }
                if (ObjectUtils.isEmpty(attrGroupDetailRefVO.getSortNo())) {
                    if (columnsVO.getSortNo() == null) {
                        columnsVO.setSortNo(0);
                    }
                    attrGroupDetailRefVO.setSortNo(columnsVO.getSortNo() != null ? columnsVO.getSortNo() : 0);
                }
                // 顺序 已前端传过来为准  不传默认属性的顺序
                if (ObjectUtils.isEmpty(attrGroupDetailRefVO.getSortNoSelf())) {
                    attrGroupDetailRefVO.setSortNoSelf(columnsVO.getSortNo());
                }
                return attrGroupDetailRefVO;
            }).collect(Collectors.toList());
        final String attrStr = attributeGroupDetailVOList.stream().map(CrmBusinessAttributeGroupDetailVO::getAttributeDesc).collect(Collectors.joining("、"));
        groupRefVO.setAttrStr(attrStr);
        // 明细
        groupRefVO.setDetails(attrGroupDetailRefVOList);
    }

    /**
     * 冗余方式获取 属性明细
     *
     * @param groupRefVO 裁判组签证官
     */
    private void detailsProcessRedundant(ProductCategoryAttrGroupRefVO groupRefVO) {
        final Long id = groupRefVO.getId();
        ProductCategoryAttrGroupDetailRefQuery groupDetailRefQuery = new ProductCategoryAttrGroupDetailRefQuery();
        groupDetailRefQuery.setGroupRefId(id);
        final List<ProductCategoryAttrGroupDetailRefVO> attrGroupDetailRefVOList = productCategoryAttrGroupDetailRefService.queryList(groupDetailRefQuery);
        final String attrStr = attrGroupDetailRefVOList.stream().map(ProductCategoryAttrGroupDetailRefVO::getCategoryName).collect(Collectors.joining("、"));
        groupRefVO.setAttrStr(attrStr);
        // 明细
        groupRefVO.setDetails(attrGroupDetailRefVOList);
    }

    @Override
    public List<ProductCategoryAttrGroupRefVO> queryList(ProductCategoryAttrGroupRefQuery query) {
        final List<ProductCategoryAttrGroupRefVO> productCategoryAttrGroupRefVOS = ProductCategoryAttrGroupRefConvert.INSTANCE.toVoList(productCategoryAttrGroupRefRepo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder)));
        productCategoryAttrGroupRefVOS.forEach(groupRefVO -> {
            // 实时获取
            detailsProcessDynamic(groupRefVO);
            // 冗余方式获取
            //detailsProcessRedundant(groupRefVO);
        });
        return productCategoryAttrGroupRefVOS;
    }

    @Override
    public ProductCategoryAttrGroupRefVO queryByKey(Long key) {
        ProductCategoryAttrGroupRefDO entity = productCategoryAttrGroupRefRepo.findById(key).orElseGet(ProductCategoryAttrGroupRefDO::new);
        Assert.notNull(entity.getId(), "不存在");
        ProductCategoryAttrGroupRefVO vo = ProductCategoryAttrGroupRefConvert.INSTANCE.toVo(entity);
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ProductCategoryAttrGroupRefVO insert(ProductCategoryAttrGroupRefPayload payload) {
        check(payload);
        dataProcess(payload);
        ProductCategoryAttrGroupRefDO entityDo = ProductCategoryAttrGroupRefConvert.INSTANCE.toDo(payload);
        final ProductCategoryAttrGroupRefDO save = productCategoryAttrGroupRefRepo.save(entityDo);
        final Long groupRefId = save.getId();
        final Long categoryId = save.getCategoryId();
        final Long groupId = save.getGroupId();
        if (payload.getId() != null && payload.getId() > 0) {
            // 先清空明细 再插入
            productCategoryAttrGroupDetailRefDAO.deleteSoftByGroupRefId(payload.getId());
        }
        // 保存明细
        insertDetails(payload, groupRefId, categoryId, groupId);
        return ProductCategoryAttrGroupRefConvert.INSTANCE.toVo(save);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<ProductCategoryAttrGroupRefVO> batchSaveOrUpdate(ProductCategoryAttrGroupRefListPayload payload) {
        // 待删除的数据
        if (!CollectionUtils.isEmpty(payload.getDeleteIdList())) {
            deleteSoft(payload.getDeleteIdList());
        }
        // 新增
        List<ProductCategoryAttrGroupRefVO> list = new ArrayList<>();
        if (!CollectionUtils.isEmpty(payload.getCategoryAttrGroupRefList())) {
            payload.getCategoryAttrGroupRefList().forEach(attrGroupRefPayload -> {
                list.add(insert(attrGroupRefPayload));
            });
        }
        return list;
    }

    /**
     * 检查数据
     *
     * @param payload
     */
    private void check(ProductCategoryAttrGroupRefPayload payload) {
        final Long categoryId = payload.getCategoryId();
        if (null == categoryId) {
            throw TwException.error("", "categoryId不能为空");
        }
        final Long groupId = payload.getGroupId();
        if (null == groupId) {
            throw TwException.error("", "groupId不能为空");
        }

        final List<ProductCategoryAttrGroupDetailRefPayload> details = payload.getDetails();
        if (!CollectionUtils.isEmpty(details)) {
            final Set<Long> attrIdList = details.stream().map(ProductCategoryAttrGroupDetailRefPayload::getAttributeId).collect(Collectors.toSet());
            if (CollectionUtils.isEmpty(attrIdList)) {
                throw TwException.error("", "details->attributeId参数缺失");
            }
            //  校验 规格属性中已存在该属性，不可重复选择！
            ProductCategoryColumnRefQuery categoryColumnRefQuery = new ProductCategoryColumnRefQuery();
            categoryColumnRefQuery.setCategoryId(categoryId);
            categoryColumnRefQuery.setColumnIdList(attrIdList);
            final List<ProductCategoryColumnRefVO> categoryColumnRefVOList = productCategoryColumnRefDAO.queryListDynamic(categoryColumnRefQuery);
            if (!CollectionUtils.isEmpty(categoryColumnRefVOList)) {
                final String collect = categoryColumnRefVOList.stream().map(ProductCategoryColumnRefVO::getAttributeDesc).collect(Collectors.joining("、"));
                throw TwException.error("", "规格属性中已存在【" + collect + "】属性，不可重复选择【" + payload.getGroupName() + "】！");
            }

            // 校验 其他属性组中 不能存在相同的属性
            ProductCategoryAttrGroupDetailRefQuery groupDetailRefQuery = new ProductCategoryAttrGroupDetailRefQuery();
            groupDetailRefQuery.setCategoryId(categoryId);
            groupDetailRefQuery.setAttributeIdList(attrIdList);
            groupDetailRefQuery.setGroupIdNotEq(groupId);
            final List<ProductCategoryAttrGroupDetailRefVO> attrGroupDetailRefVOList = productCategoryAttrGroupDetailRefDAO.queryListDynamic(groupDetailRefQuery);
            if (!CollectionUtils.isEmpty(attrGroupDetailRefVOList)) {
                final String collect = attrGroupDetailRefVOList.stream().map(ProductCategoryAttrGroupDetailRefVO::getAttributeDesc).collect(Collectors.joining("、"));
                final String groupNames = attrGroupDetailRefVOList.stream().map(ProductCategoryAttrGroupDetailRefVO::getGroupName).distinct().collect(Collectors.joining("、"));
                throw TwException.error("", "同级拓展属性组【" + groupNames + "】中已存在【" + collect + "】属性，不可重复选择【" + payload.getGroupName() + "】！");
            }


        }

    }

    /**
     * 数据处理
     *
     * @param payload
     */
    private void dataProcess(ProductCategoryAttrGroupRefPayload payload) {
        if (!StringUtils.hasText(payload.getCategoryName())) {
            final ProductCategoryVO categoryVO = productCategoryDAO.queryByKey(payload.getCategoryId());
            Assert.notNull(categoryVO, "分类不存在");
            payload.setCategoryName(categoryVO.getObjName());
        }
        if (!StringUtils.hasText(payload.getGroupName())) {
            final CrmBusinessAttributeGroupVO crmBusinessAttributeGroupVO = crmBusinessAttributeGroupDAO.queryByKey(payload.getGroupId());
            Assert.notNull(crmBusinessAttributeGroupVO, "属性组不存在");
            payload.setGroupName(crmBusinessAttributeGroupVO.getGroupName());
        }
        if (payload.getSortNo() == null) {
            payload.setSortNo(0);
        }
        if (payload.getSortNoSelf() == null) {
            payload.setSortNoSelf(0);
        }
    }

    private void insertDetails(ProductCategoryAttrGroupRefPayload payload, Long groupRefId, Long categoryId, Long groupId) {
        final List<ProductCategoryAttrGroupDetailRefPayload> details = payload.getDetails();
        final String groupName = payload.getGroupName();
        if (!CollectionUtils.isEmpty(details)) {
            details.forEach(detailRefPayload -> {
                detailRefPayload.setGroupRefId(groupRefId);
                detailRefPayload.setCategoryId(categoryId);
                detailRefPayload.setCategoryName(payload.getCategoryName());
                detailRefPayload.setGroupId(groupId);
                detailRefPayload.setGroupName(groupName);
                // 根据属性id获取 其信息 冗余起来
                final Long attributeId = detailRefPayload.getAttributeId();

                // 方式1 记录ProductCategoryAttrGroupDetailRefVO 冗余的信息

                // 方式2 读取最新的属性信息 进行属性信息冗余
                final CrmBusinessTableColumnsVO columnsVO = crmBusinessTableColumnsDAO.findByKey(attributeId);
                Assert.notNull(columnsVO, "属性不存在");
                if (ObjectUtils.isEmpty(detailRefPayload.getColumnName())) {
                    detailRefPayload.setColumnName(columnsVO.getColumnName());
                }
                if (ObjectUtils.isEmpty(detailRefPayload.getColumnDesc())) {
                    detailRefPayload.setColumnDesc(columnsVO.getColumnDesc());
                }
                if (ObjectUtils.isEmpty(detailRefPayload.getAttributeDesc())) {
                    detailRefPayload.setAttributeDesc(columnsVO.getAttributeDesc());
                }
                if (ObjectUtils.isEmpty(detailRefPayload.getAttributeType())) {
                    detailRefPayload.setAttributeType(columnsVO.getAttributeType());
                }
                if (ObjectUtils.isEmpty(detailRefPayload.getComponentType())) {
                    detailRefPayload.setComponentType(columnsVO.getComponentType());
                }
                if (ObjectUtils.isEmpty(detailRefPayload.getAttributeScope())) {
                    detailRefPayload.setAttributeScope(columnsVO.getAttributeScope());
                }
                if (ObjectUtils.isEmpty(detailRefPayload.getAttributePrompt())) {
                    detailRefPayload.setAttributePrompt(columnsVO.getAttributePrompt());
                }
                if (ObjectUtils.isEmpty(detailRefPayload.getSelectionCode())) {
                    detailRefPayload.setSelectionCode(columnsVO.getSelectionCode());
                }
                if (ObjectUtils.isEmpty(detailRefPayload.getIsMultiple())) {
                    detailRefPayload.setIsMultiple(columnsVO.getIsMultiple());
                }
                if (ObjectUtils.isEmpty(detailRefPayload.getSortNo())) {
                    if (columnsVO.getSortNo() == null) {
                        columnsVO.setSortNo(0);
                    }
                    detailRefPayload.setSortNo(columnsVO.getSortNo() != null ? columnsVO.getSortNo() : 0);
                }
                // 顺序 已前端传过来为准  不传默认属性的顺序
                if (ObjectUtils.isEmpty(detailRefPayload.getSortNoSelf())) {
                    detailRefPayload.setSortNoSelf(columnsVO.getSortNo());
                }
                productCategoryAttrGroupDetailRefService.insert(detailRefPayload);
            });

        }
    }

    /*@Override
    @Transactional(rollbackFor = Exception.class)
    public ProductCategoryAttrGroupRefVO update(ProductCategoryAttrGroupRefPayload payload) {
        ProductCategoryAttrGroupRefDO entity = productCategoryAttrGroupRefRepo.findById(payload.getId()).orElseGet(ProductCategoryAttrGroupRefDO::new);
        Assert.notNull(entity.getId(), "不存在");
        ProductCategoryAttrGroupRefDO entityDo = ProductCategoryAttrGroupRefConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        final ProductCategoryAttrGroupRefDO save = productCategoryAttrGroupRefRepo.save(entity);
        final Long groupRefId = save.getId();
        final Long categoryId = save.getCategoryId();
        final Long groupId = save.getGroupId();
        // 先清空明细 再插入
        productCategoryAttrGroupDetailRefDAO.deleteSoftByGroupRefId(entity.getId());
        insertDetails(payload, groupRefId, categoryId, groupId);
        return ProductCategoryAttrGroupRefConvert.INSTANCE.toVo(save);
    }*/

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            keys.stream().forEach(id -> {
                Optional<ProductCategoryAttrGroupRefDO> optional = productCategoryAttrGroupRefRepo.findById(id);
                if (!optional.isEmpty()) {
                    ProductCategoryAttrGroupRefDO entity = optional.get();
                    final Long categoryId = entity.getCategoryId();

                    // 校验属性组 对应分类是否被使用
                    ProductSpuQuery spuQuery = new ProductSpuQuery();
                    spuQuery.setCategoryId(categoryId);
                    spuQuery.setPermissionFlag(false);
                    final long spuCount = productSpuDAO.count(spuQuery);

                    ProductSkuQuery skuQuery = new ProductSkuQuery();
                    skuQuery.setCategoryId(categoryId);
                    final long skuCount = productSkuDAO.count(skuQuery);
                    if (spuCount > 0 || skuCount > 0) {
                        throw TwException.error("", "该属性组正被使用，无法删除");
                    }

                    entity.setDeleteFlag(1);
                    productCategoryAttrGroupRefRepo.save(entity);

                    // 删除明细
                    productCategoryAttrGroupDetailRefDAO.deleteSoftByGroupRefId(entity.getId());
                }
            });
        }
    }


}
