package com.elitesland.tw.tw5crm.server.product.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5crm.api.product.payload.ProductCategoryColumnRefListPayload;
import com.elitesland.tw.tw5crm.api.product.payload.ProductCategoryColumnRefPayload;
import com.elitesland.tw.tw5crm.api.product.query.*;
import com.elitesland.tw.tw5crm.api.product.service.ProductCategoryColumnRefService;
import com.elitesland.tw.tw5crm.api.product.vo.*;
import com.elitesland.tw.tw5crm.server.product.convert.ProductCategoryColumnRefConvert;
import com.elitesland.tw.tw5crm.server.product.dao.*;
import com.elitesland.tw.tw5crm.server.product.entity.ProductCategoryColumnRefDO;
import com.elitesland.tw.tw5crm.server.product.repo.ProductCategoryColumnRefRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * 产品分类属性关系
 *
 * @author duwh
 * @date 2023-03-02
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class ProductCategoryColumnRefServiceImpl implements ProductCategoryColumnRefService {

    private final ProductCategoryColumnRefRepo productCategoryColumnRefRepo;
    private final ProductCategoryAttrGroupRefDAO productCategoryAttrGroupRefDAO;
    private final ProductCategoryDAO productCategoryDAO;
    private final ProductCategoryColumnRefDAO productCategoryColumnRefDAO;
    private final CrmBusinessAttributeGroupDetailDAO crmBusinessAttributeGroupDetailDAO;
    private final CrmBusinessTableColumnsDAO crmBusinessTableColumnsDAO;
    private final ProductSpuDAO productSpuDAO;
    private final ProductSkuDAO productSkuDAO;

    @Override
    public PagingVO<ProductCategoryColumnRefVO> paging(ProductCategoryColumnRefQuery query) {
        return productCategoryColumnRefDAO.queryPaging(query);
    }

    @Override
    public List<ProductCategoryColumnRefVO> queryList(ProductCategoryColumnRefQuery query) {
        return productCategoryColumnRefDAO.queryListDynamic(query);
    }

    @Override
    public ProductCategoryColumnRefVO queryByKey(Long key) {
        ProductCategoryColumnRefDO entity = productCategoryColumnRefRepo.findById(key).orElseGet(ProductCategoryColumnRefDO::new);
        Assert.notNull(entity.getId(), "不存在");
        ProductCategoryColumnRefVO vo = ProductCategoryColumnRefConvert.INSTANCE.toVo(entity);
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ProductCategoryColumnRefVO insert(ProductCategoryColumnRefPayload payload) {
        // 数据检查
        check(payload);
        // 数据处理
        dataProcess(payload);
        ProductCategoryColumnRefDO entityDo = ProductCategoryColumnRefConvert.INSTANCE.toDo(payload);
        return ProductCategoryColumnRefConvert.INSTANCE.toVo(productCategoryColumnRefRepo.save(entityDo));
    }

    /**
     * 批量新增
     *
     * @param payload
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<ProductCategoryColumnRefVO> batchSaveOrUpdate(ProductCategoryColumnRefListPayload payload) {
        // 待删除的数据
        if (!CollectionUtils.isEmpty(payload.getDeleteIdList())) {
            deleteSoft(payload.getDeleteIdList());
        }

        // 新增
        List<ProductCategoryColumnRefVO> list = new ArrayList<>();
        if (!CollectionUtils.isEmpty(payload.getCategoryColumnRefList())) {
            payload.getCategoryColumnRefList().forEach(productCategoryColumnRefPayload -> {
                list.add(insert(productCategoryColumnRefPayload));
            });
        }
        return list;
    }

    /**
     * 检查
     *
     * @param payload 有效载荷
     */
    private void check(ProductCategoryColumnRefPayload payload) {
        final Long categoryId = payload.getCategoryId();
        if (null == categoryId) {
            throw TwException.error("", "categoryId不能为空");
        }
        final Long columnId = payload.getColumnId();
        if (null == columnId) {
            throw TwException.error("", "columnId不能为空");
        }

        // 校验属性 不能同时存在属性组下的属性里
        // 根据分类主键 获取属性组集合
        ProductCategoryAttrGroupRefQuery attrGroupRefQuery = new ProductCategoryAttrGroupRefQuery();
        attrGroupRefQuery.setCategoryId(categoryId);
        final List<ProductCategoryAttrGroupRefVO> attrGroupRefVOList = productCategoryAttrGroupRefDAO.queryListDynamic(attrGroupRefQuery);
        if (!CollectionUtils.isEmpty(attrGroupRefVOList)) {
            final List<Long> groupIds = attrGroupRefVOList.stream().map(ProductCategoryAttrGroupRefVO::getGroupId).distinct().collect(Collectors.toList());
            String groupNames = attrGroupRefVOList.stream().map(ProductCategoryAttrGroupRefVO::getGroupName).collect(Collectors.joining("、"));
            CrmBusinessAttributeGroupDetailQuery groupDetailQuery = new CrmBusinessAttributeGroupDetailQuery();
            groupDetailQuery.setGroupIdList(groupIds);
            groupDetailQuery.setAttributeId(columnId);
            final List<CrmBusinessAttributeGroupDetailVO> attributeGroupDetailVOList = crmBusinessAttributeGroupDetailDAO.queryListJoinDynamic(groupDetailQuery);
            if (!CollectionUtils.isEmpty(attributeGroupDetailVOList)) {
                final String attrsDesc = attributeGroupDetailVOList.stream().map(CrmBusinessAttributeGroupDetailVO::getAttributeDesc).distinct().collect(Collectors.joining("、"));
                throw TwException.error("", "扩展属性组【" + groupNames + "】中已存在该属性【" + attrsDesc + "】，不可重复选择！");
            }
        }
    }

    /**
     * 数据处理
     *
     * @param payload 有效载荷
     */
    private void dataProcess(ProductCategoryColumnRefPayload payload) {
        if (null == payload.getStatus()) {
            payload.setStatus(0);
        }
        // 分类冗余
        ProductCategoryVO categoryVO = productCategoryDAO.queryByKey(payload.getCategoryId());
        Assert.notNull(categoryVO, "分类不存在");
        payload.setCategoryName(categoryVO.getObjName());
        // 属性信息冗余
        final CrmBusinessTableColumnsVO columnsVO = crmBusinessTableColumnsDAO.findByKey(payload.getColumnId());
        Assert.notNull(columnsVO, "属性不存在");
        payload.setColumnName(columnsVO.getColumnName());
        payload.setColumnDesc(columnsVO.getColumnDesc());
        payload.setAttributeDesc(columnsVO.getAttributeDesc());
        payload.setAttributeType(columnsVO.getAttributeType());
        payload.setComponentType(columnsVO.getComponentType());
        payload.setAttributeScope(columnsVO.getAttributeScope());
        payload.setAttributePrompt(columnsVO.getAttributePrompt());
        payload.setSelectionCode(columnsVO.getSelectionCode());
        payload.setIsMultiple(columnsVO.getIsMultiple());
        if (columnsVO.getSortNo() == null) {
            columnsVO.setSortNo(0);
        }
        payload.setSortNo(columnsVO.getSortNo());
        // 顺序 已前端传过来为准  不传默认属性的顺序
        if (payload.getSortNoSelf() == null) {
            payload.setSortNoSelf(columnsVO.getSortNo());
        }

    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public ProductCategoryColumnRefVO update(ProductCategoryColumnRefPayload payload) {
        ProductCategoryColumnRefDO entity = productCategoryColumnRefRepo.findById(payload.getId()).orElseGet(ProductCategoryColumnRefDO::new);
        Assert.notNull(entity.getId(), "不存在");
        ProductCategoryColumnRefDO entityDo = ProductCategoryColumnRefConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return ProductCategoryColumnRefConvert.INSTANCE.toVo(productCategoryColumnRefRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            keys.stream().forEach(id -> {
                Optional<ProductCategoryColumnRefDO> optional = productCategoryColumnRefRepo.findById(id);
                if (!optional.isEmpty()) {
                    ProductCategoryColumnRefDO entity = optional.get();
                    final Long categoryId = entity.getCategoryId();
                    // 校验属性 对应分类是否被使用
                    ProductSpuQuery spuQuery = new ProductSpuQuery();
                    spuQuery.setPermissionFlag(false);
                    spuQuery.setCategoryId(categoryId);
                    final long spuCount = productSpuDAO.count(spuQuery);

                    ProductSkuQuery skuQuery = new ProductSkuQuery();
                    skuQuery.setCategoryId(categoryId);
                    final long skuCount = productSkuDAO.count(skuQuery);
                    if (spuCount > 0 || skuCount > 0) {
                        throw TwException.error("", "该属性正被使用，无法删除");
                    }

                    entity.setDeleteFlag(1);
                    productCategoryColumnRefRepo.save(entity);
                }
            });
        }
    }

}
