package com.elitesland.tw.tw5crm.server.product.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.util.PageUtil;
import com.elitesland.tw.tw5crm.api.product.dto.ProductAttrGroupJsonDTO;
import com.elitesland.tw.tw5crm.api.product.dto.ProductAttrJsonDTO;
import com.elitesland.tw.tw5crm.api.product.payload.ProductCategoryPayload;
import com.elitesland.tw.tw5crm.api.product.query.*;
import com.elitesland.tw.tw5crm.api.product.service.ProductCategoryService;
import com.elitesland.tw.tw5crm.api.product.vo.*;
import com.elitesland.tw.tw5crm.server.common.constants.GenerateSeqNumConstants;
import com.elitesland.tw.tw5crm.server.common.constants.ProductComponentType;
import com.elitesland.tw.tw5crm.server.product.convert.ProductCategoryConvert;
import com.elitesland.tw.tw5crm.server.product.dao.*;
import com.elitesland.tw.tw5crm.server.product.entity.ProductCategoryDO;
import com.elitesland.tw.tw5crm.server.product.repo.ProductCategoryRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 产品分类
 *
 * @author duwh
 * @date 2023-03-02
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class ProductCategoryServiceImpl extends BaseServiceImpl implements ProductCategoryService {

    private static final String TABLE_NAME = "crm_business_table_columns";
    private final ProductCategoryRepo productCategoryRepo;
    private final ProductCategoryDAO productCategoryDAO;
    private final CrmBusinessTableColumnsDAO crmBusinessTableColumnsDAO;
    private final ProductCategoryAttrGroupRefDAO productCategoryAttrGroupRefDAO;
    private final ProductCategoryColumnRefDAO productCategoryColumnRefDAO;
    private final CrmBusinessAttributeGroupDetailDAO crmBusinessAttributeGroupDetailDAO;
    private final ProductCategoryAttrGroupDetailRefDAO productCategoryAttrGroupDetailRefDAO;
    private final ProductSpuDAO productSpuDAO;
    private final ProductSkuDAO productSkuDAO;

    @Override
    public PagingVO<ProductCategoryVO> paging(ProductCategoryQuery query) {
        Page<ProductCategoryDO> page = productCategoryRepo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder), query.getPageRequest());
        return PageUtil.toPageVo(page.map(ProductCategoryConvert.INSTANCE::toVo));
    }

    @Override
    public List<ProductCategoryVO> queryList(ProductCategoryQuery query) {
        return ProductCategoryConvert.INSTANCE.toVoList(productCategoryRepo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder)));
    }

    @Override
    public ProductCategoryVO queryByKey(Long key) {
        ProductCategoryDO entity = productCategoryRepo.findById(key).orElseGet(ProductCategoryDO::new);
        Assert.notNull(entity.getId(), "不存在");
        ProductCategoryVO vo = ProductCategoryConvert.INSTANCE.toVo(entity);
        return vo;
    }

    /**
     * 获取规格属性
     *
     * @return
     */
    @Override
    public List<ProductAttrJsonDTO> getAttrs() {
        // 属性 组装json
        List<ProductAttrJsonDTO> attrJsonDTOList = attrJsonBuildByTableColumns();
        return attrJsonDTOList;
    }

    /**
     * 规格属性
     *
     * @return
     */
    private List<ProductAttrJsonDTO> attrJsonBuildByTableColumns() {
        List<ProductAttrJsonDTO> attrJsonDTOList = new ArrayList<>();
        CrmBusinessTableColumnsQuery crmBusinessTableColumnsQuery = new CrmBusinessTableColumnsQuery();
        crmBusinessTableColumnsQuery.setComponentType(ProductComponentType.SELECT.getCode());
        crmBusinessTableColumnsQuery.setIsMultiple(1);
        crmBusinessTableColumnsQuery.setStatus(1);
        crmBusinessTableColumnsQuery.setTableName(TABLE_NAME);
        final List<CrmBusinessTableColumnsVO> crmBusinessTableColumnsVOS = crmBusinessTableColumnsDAO.queryListDynamic(crmBusinessTableColumnsQuery);
        crmBusinessTableColumnsVOS.forEach(columnRef -> {
            ProductAttrJsonDTO attrJson =
                    ProductAttrJsonDTO.builder()
                            .attrId(columnRef.getId() + "")
                            .attrKey(columnRef.getColumnName())
                            .attrDesc(columnRef.getAttributeDesc())
                            .attrType(columnRef.getAttributeType())
                            .selectionCode(columnRef.getSelectionCode())
                            .sortNo(columnRef.getSortNo())
                            .sortNoSelf(columnRef.getSortNo())
                            .isMultiple(columnRef.getIsMultiple())
                            .componentType(columnRef.getComponentType())
                            .build();
            attrJsonDTOList.add(attrJson);
        });
        return attrJsonDTOList;
    }

    /**
     * 组装属性组（组下属性集合）、规格属性
     *
     * @param key 产品分类主键
     * @return ProductCategoryAttrsAndAttrGroupsVO
     */
    @Override
    public ProductCategoryAttrsAndAttrGroupsVO getAttrsAndAttrGroups(Long key) {
        ProductCategoryDO entity = productCategoryRepo.findById(key).orElseGet(ProductCategoryDO::new);
        final Long categoryId = entity.getId();
        Assert.notNull(categoryId, "产品分类不存在");
        ProductCategoryAttrsAndAttrGroupsVO resultVo = ProductCategoryConvert.INSTANCE.toVoAttrsAndAttrGroups(entity);

        // 查询属性 、属性组 组装json
        // 属性组 组装json
        final List<ProductAttrGroupJsonDTO> attrGroupJson = attrGroupsBuild(categoryId);
        resultVo.setGroups(attrGroupJson);

        // 属性 组装json
        List<ProductAttrJsonDTO> attrJsonDTOList = attrJsonBuild(categoryId);
        resultVo.setAttrs(attrJsonDTOList);
        return resultVo;
    }

    /**
     * 构建规格属性集合
     *
     * @param categoryId
     * @return List<ProductAttrJsonDTO>
     */
    @NotNull
    private List<ProductAttrJsonDTO> attrJsonBuild(Long categoryId) {
        List<ProductAttrJsonDTO> attrJsonDTOList = new ArrayList<>();
        ProductCategoryColumnRefQuery columnRefQuery = new ProductCategoryColumnRefQuery();
        columnRefQuery.setCategoryId(categoryId);
        // TODO 启用条件
        // columnRefQuery.setStatus(1);
        List<ProductCategoryColumnRefVO> categoryColumnRefVOList = productCategoryColumnRefDAO.queryListDynamic(columnRefQuery);
        categoryColumnRefVOList.forEach(columnRef -> {
            //方案一 (1) 实时拉取最新的属性配置信息
            // 根据columnRef.getColumnId() 实时查询 再赋值
            final CrmBusinessTableColumnsVO tableColumnsVO = crmBusinessTableColumnsDAO.findByKey(columnRef.getColumnId());
            ProductAttrJsonDTO attrJson =
                    ProductAttrJsonDTO.builder()
                            .attrId(columnRef.getColumnId() + "")
                            .attrKey(tableColumnsVO.getColumnName())
                            .attrDesc(tableColumnsVO.getAttributeDesc())
                            .attrType(tableColumnsVO.getAttributeType())
                            .selectionCode(tableColumnsVO.getSelectionCode())
                            .sortNo(tableColumnsVO.getSortNo())
                            .sortNoSelf(tableColumnsVO.getSortNo())
                            .isMultiple(tableColumnsVO.getIsMultiple())
                            .componentType(tableColumnsVO.getComponentType())
                            .build();

            //方案二（2）冗余方式读取 规格属性
            //ProductAttrJsonDTO attrJson =
            //    ProductAttrJsonDTO.builder()
            //        .attrId(columnRef.getColumnId())
            //        .attrKey(columnRef.getColumnName())
            //        .attrDesc(columnRef.getAttributeDesc())
            //        .attrType(columnRef.getAttributeType())
            //        .selectionCode(columnRef.getSelectionCode())
            //        .sortNo(columnRef.getSortNo())
            //        .sortNoSelf(columnRef.getSortNoSelf())
            //        .isMultiple(columnRef.getIsMultiple())
            //        .componentType(columnRef.getComponentType())
            //        .build();
            attrJsonDTOList.add(attrJson);
        });
        return attrJsonDTOList;
    }

    /**
     * 构建组装 属性组
     *
     * @param categoryId
     */
    private List<ProductAttrGroupJsonDTO> attrGroupsBuild(Long categoryId) {
        // 属性组
        List<ProductAttrGroupJsonDTO> groups = new ArrayList<>();
        ProductCategoryAttrGroupRefQuery attrGroupRefQuery = new ProductCategoryAttrGroupRefQuery();
        attrGroupRefQuery.setCategoryId(categoryId);
        // TODO 启用条件
        // attrGroupRefQuery.setStatus(1);
        // 分类 属性组关系
        List<ProductCategoryAttrGroupRefVO> productCategoryAttrGroupRefVOS = productCategoryAttrGroupRefDAO.queryListDynamic(attrGroupRefQuery);
        // 属性组下的属性集合
        productCategoryAttrGroupRefVOS.forEach(groupRef -> {
            //final Long groupRefId = groupRef.getId();
            final Long groupId = groupRef.getGroupId();

            List<ProductAttrJsonDTO> attrValueList = new ArrayList<>();
            //方案一 （1）（实时拉取组下的属性信息）
            dynamicGetAttrInfo(groupId, attrValueList);

            //方案二 （2) 冗余方式读取 冗余在分类关系表
            //redundanceGetAttrInfo(groupRefId, attrValueList);

            // 属性组
            ProductAttrGroupJsonDTO groupJson =
                    ProductAttrGroupJsonDTO
                            .builder()
                            .groupId(groupId + "")
                            .groupName(groupRef.getGroupName())
                            .sort(groupRef.getSortNo())
                            .sortSelf(groupRef.getSortNoSelf())
                            // 属性组下的属性集合
                            .attrList(attrValueList)
                            .build();
            groups.add(groupJson);
        });
        return groups;
    }

    /**
     * 冗余获取属性信息
     *
     * @param groupRefId    主表id
     * @param attrValueList
     */
    private void redundanceGetAttrInfo(Long groupRefId, List<ProductAttrJsonDTO> attrValueList) {
        ProductCategoryAttrGroupDetailRefQuery groupDetailRefQuery = new ProductCategoryAttrGroupDetailRefQuery();
        groupDetailRefQuery.setGroupRefId(groupRefId);
        // TODO 是否启用
        //groupDetailRefQuery.setStatus(1);
        List<ProductCategoryAttrGroupDetailRefVO> attrGroupDetailRefVOS
                = productCategoryAttrGroupDetailRefDAO.queryListDynamic(groupDetailRefQuery);
        attrGroupDetailRefVOS.forEach(groupDetailRef -> {
            ProductAttrJsonDTO attrJson =
                    ProductAttrJsonDTO.builder()
                            .attrId(groupDetailRef.getAttributeId() + "")
                            .attrKey(groupDetailRef.getColumnName())
                            .attrDesc(groupDetailRef.getAttributeDesc())
                            .attrType(groupDetailRef.getAttributeType())
                            .selectionCode(groupDetailRef.getSelectionCode())
                            .sortNo(groupDetailRef.getSortNo())
                            .sortNoSelf(groupDetailRef.getSortNoSelf())
                            .isMultiple(groupDetailRef.getIsMultiple())
                            .componentType(groupDetailRef.getComponentType())
                            .unitClass(groupDetailRef.getUnitClass())
                            .isRequired(groupDetailRef.getIsRequired())
                            .build();
            attrValueList.add(attrJson);
        });
    }

    /**
     * 实时动态获取属性信息
     *
     * @param groupId
     * @param attrValueList
     */
    private void dynamicGetAttrInfo(Long groupId, List<ProductAttrJsonDTO> attrValueList) {
        List<CrmBusinessAttributeGroupDetailVO> crmBusinessAttributeGroupDetailVOS =
                crmBusinessAttributeGroupDetailDAO.queryByGroupId(groupId);
        crmBusinessAttributeGroupDetailVOS.forEach(attr -> {
            ProductAttrJsonDTO attrJson =
                    ProductAttrJsonDTO.builder()
                            .attrId(attr.getAttributeId() + "")
                            .attrKey(attr.getColumnName())
                            .attrDesc(attr.getAttributeDesc())
                            .attrType(attr.getAttributeType())
                            .selectionCode(attr.getSelectionCode())
                            .sortNo(attr.getSortNo())
                            //.sortNoSelf(attr.getSortNo())
                            .isMultiple(attr.getIsMultiple())
                            .componentType(attr.getComponentType())
                            .unitClass(attr.getUnitClass())
                            .isRequired(attr.getIsRequired())
                            .build();
            attrValueList.add(attrJson);
        });
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ProductCategoryVO insert(ProductCategoryPayload payload) {
        // 数据校验
        check(payload);
        // 唯一性校验
        checkUnique(null, payload.getObjName());
        String code = generateSeqNum(GenerateSeqNumConstants.CRM_PRODUCT_CATEGORY_NO);
        payload.setObjNo(code);
        if (null != payload.getParentId() && payload.getParentId().equals(0L)) {
            payload.setParentId(null);
        }
        if (null == payload.getStatus()) {
            payload.setStatus(0);
        }
        ProductCategoryDO entityDo = ProductCategoryConvert.INSTANCE.toDo(payload);
        final ProductCategoryDO save = productCategoryRepo.save(entityDo);
        // 计算子节点数目
        entityDo.setSubCount(0);
        // 更新父节点菜单数目
        updateSubCnt(entityDo.getParentId());
        return ProductCategoryConvert.INSTANCE.toVo(save);
    }

    private void check(ProductCategoryPayload payload) {
        if (!StringUtils.hasText(payload.getObjName())) {
            throw TwException.error("", "请输入分类名称");
        }
    }

    /**
     * 更新节点数目
     *
     * @param pid pid
     */
    private void updateSubCnt(Long pid) {
        if (pid != null) {
            int count = productCategoryRepo.countByParentId(pid);
            productCategoryRepo.updateSubCntById(count, pid);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ProductCategoryVO update(ProductCategoryPayload payload) {
        ProductCategoryDO entity = productCategoryRepo.findById(payload.getId()).orElseGet(ProductCategoryDO::new);
        Assert.notNull(entity.getId(), "不存在");
        if (payload.getId().equals(payload.getParentId())) {
            throw TwException.error("", "上级不能为自己");
        }
        // 记录的父节点ID
        Long oldPid = entity.getParentId();
        Long newPid = payload.getParentId();
        ProductCategoryDO entityDo = ProductCategoryConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        final ProductCategoryDO save = productCategoryRepo.save(entity);
        // 计算父级菜单节点数目
        updateSubCnt(oldPid);
        updateSubCnt(newPid);
        return ProductCategoryConvert.INSTANCE.toVo(save);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            keys.stream().forEach(id -> {
                Optional<ProductCategoryDO> optional = productCategoryRepo.findById(id);
                if (!optional.isEmpty()) {
                    // 校验分类是否被使用
                    ProductSpuQuery spuQuery = new ProductSpuQuery();
                    spuQuery.setCategoryId(id);
                    spuQuery.setPermissionFlag(false);
                    final long spuCount = productSpuDAO.count(spuQuery);

                    ProductSkuQuery skuQuery = new ProductSkuQuery();
                    skuQuery.setCategoryId(id);
                    final long skuCount = productSkuDAO.count(skuQuery);
                    if (spuCount > 0 || skuCount > 0) {
                        throw TwException.error("", "该分类正被使用，无法删除");
                    }

                    ProductCategoryDO entity = optional.get();
                    entity.setDeleteFlag(1);
                    productCategoryRepo.save(entity);

                    //TODO 删除属性组、属性关系？
                }
            });
        }
    }

    /**
     * 懒加载产品列表
     *
     * @param pid pid
     * @return {@link List}<{@link ProductCategoryVO}>
     */
    @Override
    public List<ProductCategoryVO> queryAllCategory(Long pid) {
        List<ProductCategoryDO> categoryDOList;
        if (pid != null && !pid.equals(0L)) {
            categoryDOList = productCategoryRepo.findByParentId(pid);
        } else {
            categoryDOList = productCategoryRepo.findByParentIdIsNull();
        }
        return ProductCategoryConvert.INSTANCE.toVoList(categoryDOList);
    }

    /**
     * 构建树
     *
     * @param categoryVOList 类别volist
     * @return {@link List}<{@link ProductCategoryVO}>
     */
    @Override
    public List<ProductCategoryVO> buildTree(List<ProductCategoryVO> categoryVOList) {
        List<ProductCategoryVO> trees = new ArrayList<>();
        Set<Long> ids = new HashSet<>();
        for (ProductCategoryVO categoryDTO : categoryVOList) {
            if (categoryDTO.getParentId() == null) {
                trees.add(categoryDTO);
            }
            for (ProductCategoryVO it : categoryVOList) {
                if (categoryDTO.getId().equals(it.getParentId())) {
                    if (categoryDTO.getChildren() == null) {
                        categoryDTO.setChildren(new ArrayList<>());
                    }
                    categoryDTO.getChildren().add(it);
                    ids.add(it.getId());
                }
            }
        }
        if (trees.size() == 0) {
            trees = categoryVOList.stream().filter(s -> !ids.contains(s.getId())).collect(Collectors.toList());
        }
        return trees;
    }

    /**
     * 构建树结构
     *
     * @return {@link List}<{@link ProductCategoryVO}>
     */
    @Override
    public List<ProductCategoryVO> tree() {
        ProductCategoryQuery query = new ProductCategoryQuery();
        List<ProductCategoryVO> categoryVOList = queryList(query);
        List<ProductCategoryVO> tree = buildTree(categoryVOList);
        return tree;
    }

    @Override
    public long checkUnique(Long id, String objName) {
        ProductCategoryQuery categoryQuery = new ProductCategoryQuery();
        categoryQuery.setIdNotEq(id);
        categoryQuery.setObjNameEq(objName);
        final long count = productCategoryDAO.count(categoryQuery);
        if (count > 0) {
            throw TwException.error("", "存在相同产品名称");
        }
        return count;
    }


}
