package com.elitesland.tw.tw5crm.server.product.service;

import com.alibaba.fastjson.JSON;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitesland.tw.tw5.api.common.TwWorkFlowCommonVO;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemSelectionVO;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.service.TransactionUtilService;
import com.elitesland.tw.tw5.server.common.workFlow.ProcDefKey;
import com.elitesland.tw.tw5.server.common.workFlow.WorkflowUtil;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.RoleEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.WorkFlowStatusEnum;
import com.elitesland.tw.tw5.server.udc.UdcUtil;
import com.elitesland.tw.tw5crm.api.common.change.payload.ComBusinessChangePayload;
import com.elitesland.tw.tw5crm.api.common.change.service.ComBusinessChangeService;
import com.elitesland.tw.tw5crm.api.common.change.vo.ComBusinessChangeVO;
import com.elitesland.tw.tw5crm.api.product.payload.ProductPriceCustomerPayload;
import com.elitesland.tw.tw5crm.api.product.payload.ProductPriceDetailPayload;
import com.elitesland.tw.tw5crm.api.product.payload.ProductPriceOrgPayload;
import com.elitesland.tw.tw5crm.api.product.payload.ProductPricePayload;
import com.elitesland.tw.tw5crm.api.product.query.ProductPriceQuery;
import com.elitesland.tw.tw5crm.api.product.service.ProductPriceService;
import com.elitesland.tw.tw5crm.api.product.vo.*;
import com.elitesland.tw.tw5crm.server.common.change.changeTypeEnum.ChangeTypeEnum;
import com.elitesland.tw.tw5crm.server.common.change.dao.ComBusinessChangeDAO;
import com.elitesland.tw.tw5crm.server.common.constants.GenerateSeqNumConstants;
import com.elitesland.tw.tw5crm.server.product.convert.ProductPriceConvert;
import com.elitesland.tw.tw5crm.server.product.convert.ProductPriceCustomerConvert;
import com.elitesland.tw.tw5crm.server.product.convert.ProductPriceDetailConvert;
import com.elitesland.tw.tw5crm.server.product.convert.ProductPriceOrgConvert;
import com.elitesland.tw.tw5crm.server.product.dao.ProductPriceDAO;
import com.elitesland.tw.tw5crm.server.product.dao.ProductPriceDetailDAO;
import com.elitesland.tw.tw5crm.server.product.dao.ProductSkuDAO;
import com.elitesland.tw.tw5crm.server.product.entity.ProductPriceCustomerDO;
import com.elitesland.tw.tw5crm.server.product.entity.ProductPriceDO;
import com.elitesland.tw.tw5crm.server.product.entity.ProductPriceDetailDO;
import com.elitesland.tw.tw5crm.server.product.entity.ProductPriceOrgDO;
import com.elitesland.tw.tw5crm.server.product.repo.ProductPriceCustomerRepo;
import com.elitesland.tw.tw5crm.server.product.repo.ProductPriceOrgRepo;
import com.elitesland.tw.tw5crm.server.product.repo.ProductPriceRepo;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.TaskInfo;
import com.elitesland.workflow.WorkflowResult;
import com.elitesland.workflow.WorkflowService;
import com.elitesland.workflow.payload.CurrentTaskInfosPayload;
import com.elitesland.workflow.payload.StartProcessPayload;
import com.google.common.collect.Lists;
import com.xxl.job.core.log.XxlJobLogger;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 商品价目
 *
 * @author carl
 * @date 2023-03-14
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class ProductPriceServiceImpl extends BaseServiceImpl implements ProductPriceService {
    private final ComBusinessChangeService businessChangeService;
    private final TransactionUtilService transactionUtilService;
    private final ProductPriceRepo productPriceRepo;
    private final ProductPriceOrgRepo productPriceOrgRepo;
    private final ProductPriceCustomerRepo productPriceCustomerRepo;
    private final ProductPriceDAO productPriceDAO;
    private final ProductPriceDetailDAO detailDAO;
    private final ProductSkuDAO skuDAO;
    private final CacheUtil cacheUtil;
    private final WorkflowUtil workflowUtil;
    private final ComBusinessChangeDAO businessChangeDao;
    private final UdcUtil udcUtil;
    private final WorkflowService workflowService;
    @Value("${tw5.workflow.enabled:false}")
    private Boolean workflow_enabled;

    @Override
    public PagingVO<ProductPriceVO> queryPaging(ProductPriceQuery query) {
        //权限处理
        operPermissionFlag(query);

        PagingVO<ProductPriceVO> productPriceVOPagingVO = productPriceDAO.queryPaging(query);
        List<ProductPriceVO> priceVOs = productPriceVOPagingVO.getRecords();
        if (!ObjectUtils.isEmpty(priceVOs)) {
            List<Long> priceVOIds = priceVOs.stream().map(ProductPriceVO::getId).collect(Collectors.toList());
            //获取价目明细数据
            List<ProductPriceDetailVO> productPriceDetailVOS = detailDAO.queryByPriceIds(priceVOIds);
            priceVOs.forEach(priceVO -> {

                List<ProductPriceDetailVO> collect = productPriceDetailVOS.stream().filter(detailVO -> detailVO.getPriceId().equals(priceVO.getId())).collect(Collectors.toList());

                // 完善产品类型的税率 商机报价使用
                collect.forEach(e->{
                    PrdSystemSelectionVO systemSelection = cacheUtil.getSystemSelection("crm:product:spu_type");
                    if (systemSelection!=null){
                        List<PrdSystemSelectionVO> systemSelectionChildren = systemSelection.getChildren();
                        PrdSystemSelectionVO prdSystemSelectionVO = systemSelectionChildren.stream().filter(s -> s.getSelectionValue().equals(e.getSpuType())).findFirst().orElse(null);
                        if (prdSystemSelectionVO!=null){
                            String extString1 = prdSystemSelectionVO.getExtString1();
                            e.setTaxRate(extString1);
                        }
                    }
                });
                priceVO.setPriceDetailVOs(collect);
            });
        }
        // 获取工作流审批信息
        getTaskInfo(priceVOs);

        return productPriceVOPagingVO;
    }

    /**
     * 获取工作流审批信息
     *
     * @param voList 签证官列表
     */
    private void getTaskInfo(List<ProductPriceVO> voList) {
        // 查询当前处理的任务名称
        if (!CollectionUtils.isEmpty(voList)) {
            // 1、组装成map，key为流程实例Id,value为对象
            Map<String, TwWorkFlowCommonVO> map = voList
                .stream()
                .filter(vo -> org.apache.commons.lang3.StringUtils.isNoneBlank(vo.getProcInstId()))
                .collect(Collectors.toMap(TwWorkFlowCommonVO::getProcInstId, Function.identity()));
            if (null != map && map.isEmpty() == false) {
                // 2、查询当前处理的任务名称
                final Set<String> procInstIds = map.keySet();
                HashSet hashSet = new HashSet(procInstIds);
                // 3、设置任务名称到业务对象上
                CurrentTaskInfosPayload currentTaskInfosPayload = new CurrentTaskInfosPayload();
                currentTaskInfosPayload.setProcInstIds(hashSet);
                // 调用工作流程方法查询任务信息
                final WorkflowResult<HashMap<String, TaskInfo>> workflowResult = workflowService.currentTaskInfos(currentTaskInfosPayload);
                if (workflowResult.isSuccess()) {
                    HashMap<String, TaskInfo> currentTaskNames = workflowResult.getData();
                    // 3、设置任务名称到业务对象上
                    currentTaskNames.forEach((key, value) -> {
                        TwWorkFlowCommonVO vo = map.get(key);
                        vo.setTaskInfo(value);
                    });
                }
            }
        }
    }

    @Override
    public List<ProductPriceVO> queryList(ProductPriceQuery query) {
        return ProductPriceConvert.INSTANCE.toVoList(
                productPriceRepo.findAll(
                        (root, criteriaQuery, criteriaBuilder)
                                -> QueryHelp.getPredicate(root, query, criteriaBuilder)
                        , query.getPageRequest().getSort()
                )
        );
    }

    @Override
    public List<ProductPriceVO> queryListDynamic(ProductPriceQuery query) {
        return productPriceDAO.queryListDynamic(query);
    }

    @Override
    public ProductPriceVO queryByKey(Long key) {
        ProductPriceVO productPriceVO = productPriceDAO.queryByKey(key);
        if (ObjectUtils.isEmpty(productPriceVO)) {
            throw TwException.error("", "产品价目不存在，请核验！");
        }
        List<ProductPriceCustomerDO> custDOs = productPriceCustomerRepo.findByPriceIdAndDeleteFlag(key, 0);
        if (!ObjectUtils.isEmpty(custDOs)) {
            List<ProductPriceCustomerVO> productPriceCustomerVOS = ProductPriceCustomerConvert.INSTANCE.toVoList(custDOs);
            productPriceVO.setPriceCustomerVOs(productPriceCustomerVOS);
        }
        List<ProductPriceOrgDO> orgDOs = productPriceOrgRepo.findByPriceIdAndDeleteFlag(key, 0);
        if (!ObjectUtils.isEmpty(orgDOs)) {
            List<ProductPriceOrgVO> productPriceOrgVOS = ProductPriceOrgConvert.INSTANCE.toVoList(orgDOs);
            productPriceVO.setPriceOrgVOs(productPriceOrgVOS);
        }
        List<ProductPriceDetailVO> productPriceDetailVOs = detailDAO.queryByPriceIds(List.of(key));
        productPriceVO.setPriceDetailVOs(productPriceDetailVOs);
        return productPriceVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ProductPriceVO insert(ProductPricePayload payload) {
        //数据校验
        checkData(payload);

        ProductPriceDO entityDo = ProductPriceConvert.INSTANCE.toDo(payload);
        String code = generateSeqNum(GenerateSeqNumConstants.CRM_PRODUCT_PRICE);
        entityDo.setPriceCode(code);
        entityDo.setPriceStatus(WorkFlowStatusEnum.CREATE_WORK.getCode());
        entityDo = productPriceDAO.save(entityDo);
        Long id = entityDo.getId();
        //保存报价明细，销售组织，销售客户
        saveOtherDatas(payload, id);

        ProductPriceVO productPriceVO = ProductPriceConvert.INSTANCE.toVo(entityDo);
        //数据提交,执行工作流
        if (payload.getSubmit()) {
            payload.setId(id);
            submitProc(payload);

        }
        return productPriceVO;
    }

    /**
     * 创建工作流
     *
     * @param payload
     */
    void submitProc(ProductPricePayload payload) {
        ProcessInfo processInfo = new ProcessInfo();
        String status = WorkFlowStatusEnum.APPROVED_WORK.getCode();
        if (workflow_enabled) {
            status = WorkFlowStatusEnum.APPROVING_WORK.getCode();
            Long orgManageUserId = cacheUtil.getOrgManageUserId(payload.getOrgId());
            HashMap<String, Object> batchMap = new HashMap<>();
            batchMap.put("Activity_03fjkz8", Lists.newArrayList(orgManageUserId.toString()));
            //发起流程审批
            processInfo = workflowUtil.startProcess(StartProcessPayload.of(
                ProcDefKey.PRODUCT_PRICE.name(),
                payload.getPriceName() + "-产品价目审批流程",
                payload.getId() + "",
                batchMap)
            );
        }
        ProductPricePayload payload0 = new ProductPricePayload();
        payload0.setId(payload.getId());
        payload0.setProcInstId(processInfo.getProcInstId());
        payload0.setProcInstStatus(processInfo.getProcInstStatus());
        payload0.setSubmitTime(LocalDateTime.now());
        payload0.setPriceStatus(status);
        //开启事务执行修改，主要是修改审批状态
        transactionUtilService.executeWithRunnable(() -> {
            productPriceDAO.updateWorkFlow(payload0);
        });
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ProductPriceVO update(ProductPricePayload payload) {
        ProductPriceVO productPriceVO = productPriceDAO.queryByKey(payload.getId());
        if (!ObjectUtils.isEmpty(productPriceVO)) {
            //数据校验
            checkData(payload);
            //新建装填直接修改
            if (productPriceVO.getPriceStatus().equals(WorkFlowStatusEnum.CREATE_WORK.getCode())) {
                updateProductPrice(payload);
                //数据提交,执行工作流
                if (payload.getSubmit()) {
                    submitProc(payload);
                }
            } else if (productPriceVO.getPriceStatus().equals(WorkFlowStatusEnum.APPROVED_WORK.getCode())) {
                //激活状态的变更
                ProductPricePayload productPricePayload = ProductPriceConvert.INSTANCE.toPayload(productPriceVO);
                //插入变更数据
                insertPayload(productPricePayload);

                Long saveId = businessChangeService.save(ChangeTypeEnum.PROJECT_PRICE.getCode(), udcUtil.translate(productPricePayload), udcUtil.translate(payload), payload.getId() + "");

                //ProcessInfo processInfo = startChangeWorkFlow(productPriceVO, saveId);
                // 启动变更流程
                ProcessInfo processInfo = new ProcessInfo();
                // 激活
                String priceStatus = WorkFlowStatusEnum.APPROVED_WORK.getCode();
                String changeApprStatus = WorkFlowStatusEnum.APPROVED_WORK.name();
                if(workflow_enabled){
                    priceStatus = WorkFlowStatusEnum.CHANGING_WORK.getCode();
                    changeApprStatus = WorkFlowStatusEnum.CHANGING_WORK.getCode();

                    Long orgManageUserId = cacheUtil.getOrgManageUserId(productPriceVO.getOrgId());
                    HashMap<String, Object> batchMap = new HashMap<>();
                    batchMap.put("Activity_0wctf3c", Lists.newArrayList(orgManageUserId.toString()));
                    //发起流程审批
                    processInfo = workflowUtil.startProcess(StartProcessPayload.of(
                        ProcDefKey.PRODUCT_PRICE_CHANGE.name(),
                        productPriceVO.getPriceName() + "-产品价目变更审批流程",
                        saveId + "",
                        batchMap)
                    );
                }

                ProductPricePayload payload0 = new ProductPricePayload();
                payload0.setId(payload.getId());
                payload0.setSubmitTime(LocalDateTime.now());
                payload0.setPriceStatus(priceStatus);

                ComBusinessChangePayload businessChangePayload = new ComBusinessChangePayload();
                businessChangePayload.setId(saveId);
                businessChangePayload.setApprProcInstId(processInfo.getProcInstId());
                businessChangePayload.setApprStatus(priceStatus);
                businessChangePayload.setChangeStatus(changeApprStatus);
                //开启事务执行修改，主要是修改审批状态
                transactionUtilService.executeWithRunnable(() -> {
                    productPriceDAO.updateWorkFlow(payload0);
                    businessChangeDao.updateWorkFlow(businessChangePayload);
                });

                // 如果没有启用流程  则变更直接生效
                if (!workflow_enabled) {
                    ComBusinessChangeVO comBusinessChangeVO = businessChangeDao.queryByKey(saveId);

                    ProductPricePayload pricePayload = new ProductPricePayload();
                    pricePayload.setId(Long.valueOf(comBusinessChangeVO.getChangeDocId()));
                    pricePayload.setPriceStatus(WorkFlowStatusEnum.APPROVED_WORK.getCode());
                    pricePayload = JSON.parseObject(comBusinessChangeVO.getChangeContent(), ProductPricePayload.class);

                    pricePayload.setPriceStatus(WorkFlowStatusEnum.APPROVED_WORK.getCode());
                    LocalDate now = LocalDate.now();

                    LocalDate startTime = pricePayload.getStartTime();
                    LocalDate endTime = pricePayload.getEndTime();
                    if (!ObjectUtils.isEmpty(startTime)) {
                        if (startTime.isAfter(now)) {
                            //开始时间大于当前时间，表示还未开始
                            pricePayload.setPriceStatus(WorkFlowStatusEnum.PENDING_WORK.getCode());
                        } else {
                            if (!ObjectUtils.isEmpty(endTime)) {
                                if (endTime.isBefore(now)) {
                                    //结束时间小于于当前时间，表示已经结束
                                    pricePayload.setPriceStatus(WorkFlowStatusEnum.PENDING_WORK.getCode());
                                }
                            }
                        }
                    }
                    pricePayload.setApprovedTime(LocalDateTime.now());

                    ComBusinessChangePayload changePayload = new ComBusinessChangePayload();
                    changePayload.setId(comBusinessChangeVO.getId());
                    changePayload.setChangeStatus(WorkFlowStatusEnum.APPROVED_WORK.getCode());
                    businessChangeDao.updateWorkFlow(changePayload);

                    updateProductPrice(pricePayload);

                }
            } else {
                throw TwException.error("", "仅新建和激活状态支持变更，请核验！");
            }
            return null;
        } else {
            throw TwException.error("", "变更数据不存在，请核验！");
        }

    }

    /**
     * 插入变更数据
     *
     * @param payload
     */
    void insertPayload(ProductPricePayload payload) {
        List<ProductPriceDetailVO> productPriceDetailVOs = detailDAO.queryByPriceIds(List.of(payload.getId()));
        if (!ObjectUtils.isEmpty(productPriceDetailVOs)) {
            List<ProductPriceDetailPayload> productPriceDetailPayloads = ProductPriceDetailConvert.INSTANCE.toPayloadList(productPriceDetailVOs);
            payload.setPriceDetailPayloads((List<ProductPriceDetailPayload>) udcUtil.translate(productPriceDetailPayloads));

        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long updateStatus(List<Long> keys) {
        if (!keys.isEmpty()) {
            List<ProductPriceVO> productPriceVOS = productPriceDAO.queryByKeys(keys);
            if (ObjectUtils.isEmpty(productPriceVOS)) {
                throw TwException.error("", "操作数据不存在，请核验！");
            }
            List<String> collect = productPriceVOS.stream().map(ProductPriceVO::getPriceStatus).distinct().collect(Collectors.toList());
            if (collect.size() > 1) {
                throw TwException.error("", "相同数据状态可进行该操作，请核验！");
            }
            String priceStatus = collect.get(0);
            String priceStatusNew = priceStatus.equals(WorkFlowStatusEnum.PENDING_WORK.getCode()) ? WorkFlowStatusEnum.APPROVED_WORK.getCode() : WorkFlowStatusEnum.PENDING_WORK.getCode();

            //激活操作必须在有效期范围内
            if (priceStatusNew.equals(WorkFlowStatusEnum.APPROVED_WORK.getCode())) {
                LocalDate now = LocalDate.now();
                productPriceVOS.forEach(productPriceVO -> {
                    LocalDate startTime = productPriceVO.getStartTime();
                    LocalDate endTime = productPriceVO.getEndTime();
                    if (!ObjectUtils.isEmpty(startTime)) {
                        if (startTime.isAfter(now)) {
                            throw TwException.error("", "当前日期不在有效期范围内，请核验！");
                        }
                    }
                    if (!ObjectUtils.isEmpty(endTime)) {
                        if (endTime.isBefore(now)) {
                            throw TwException.error("", "当前日期不在有效期范围内，请核验！");
                        }
                    }
                });
            }
            productPriceDAO.updateStatus(keys, priceStatusNew);
        } else {
            throw TwException.error("", "操作数据不可为空，请核验！");
        }
        return null;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void xxxJobCheckProductPriceStatus() {
        XxlJobLogger.log("定时核验产品价目表状态核验开始...");
        int count = 0;
        List<ProductPriceVO> productPriceVOS = productPriceDAO.queryByDateAndStatus(LocalDate.now(), WorkFlowStatusEnum.APPROVED_WORK.getCode());
        if (!ObjectUtils.isEmpty(productPriceVOS)) {
            List<Long> collect = productPriceVOS.stream().map(ProductPriceVO::getId).collect(Collectors.toList());
            productPriceDAO.updateStatusForXxxJob(collect, WorkFlowStatusEnum.PENDING_WORK.getCode());
            count = collect.size();
        }
        XxlJobLogger.log("定时核验产品价目表状态 结束...变更数量：{}", count);
    }

    /**
     * 保存报价明细，销售组织，销售客户
     *
     * @param payload
     * @param id
     */
    void saveOtherDatas(ProductPricePayload payload, Long id) {
        //保存销售客户
        List<ProductPriceCustomerPayload> priceCustomerPayloads = payload.getPriceCustomerPayloads();
        if (!ObjectUtils.isEmpty(priceCustomerPayloads)) {
            List<ProductPriceCustomerDO> customerDOS = new ArrayList<>();
            priceCustomerPayloads.forEach(customerPayload -> {
                ProductPriceCustomerDO productPriceCustomerDO = ProductPriceCustomerConvert.INSTANCE.toDo(customerPayload);
                productPriceCustomerDO.setPriceId(id);
                customerDOS.add(productPriceCustomerDO);
            });
            productPriceCustomerRepo.saveAll(customerDOS);
        }
        //保存销售组织
        List<ProductPriceOrgPayload> priceOrgPayloads = payload.getPriceOrgPayloads();
        if (!ObjectUtils.isEmpty(priceOrgPayloads)) {
            List<ProductPriceOrgDO> orgDOS = new ArrayList<>();
            priceOrgPayloads.forEach(orgPayload -> {
                ProductPriceOrgDO productPriceOrgDO = ProductPriceOrgConvert.INSTANCE.toDo(orgPayload);
                productPriceOrgDO.setPriceId(id);
                orgDOS.add(productPriceOrgDO);
            });
            productPriceOrgRepo.saveAll(orgDOS);
        }
        List<ProductPriceDetailPayload> priceDetailPayloads = payload.getPriceDetailPayloads();
        Set<Long> skuIds = priceDetailPayloads.stream().map(ProductPriceDetailPayload::getSkuId).collect(Collectors.toSet());
        List<ProductSkuVO> productSkuVOS = skuDAO.queryByKeys(new ArrayList<>(skuIds));
        Map<Long, ProductSkuVO> productSkuVOMap = productSkuVOS.stream().collect(Collectors.toMap(ProductSkuVO::getId, Function.identity()));
        List<ProductPriceDetailDO> priceDetailDOs = new ArrayList<>();
        priceDetailPayloads.forEach(priceDetailPayload -> {
            ProductPriceDetailDO productPriceDetailDO = ProductPriceDetailConvert.INSTANCE.toDo(priceDetailPayload);
            ProductSkuVO productSkuVO = productSkuVOMap.get(productPriceDetailDO.getSkuId());
            //如果价格相等则改变价格是否变更状态
            if (productSkuVO.getStandardPrice().compareTo(productPriceDetailDO.getStandardPrice()) == 0) {
                productPriceDetailDO.setIsChange(false);
            }
            productPriceDetailDO.setPriceId(id);
            priceDetailDOs.add(productPriceDetailDO);
        });
        detailDAO.saveAll(priceDetailDOs);
    }

    /**
     * 执行数据变更
     *
     * @param payload
     */
    @Override
    public void updateProductPrice(ProductPricePayload payload) {
        ProductPriceDO entityDo = ProductPriceConvert.INSTANCE.toDo(payload);
        productPriceDAO.save(entityDo);
        if (!ObjectUtils.isEmpty(payload.getDelDetailIds())) {
            //删除前端选择
            detailDAO.deleteSoft(payload.getDelDetailIds());
        }
        Long id = entityDo.getId();
        //删除销售客户
        productPriceCustomerRepo.deleteByPriceId(id);
        //删除销售组织
        productPriceOrgRepo.deleteByPriceId(id);
        //保存报价明细，销售组织，销售客户
        saveOtherDatas(payload, id);
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            List<ProductPriceVO> productPriceVOS = productPriceDAO.queryByKeys(keys);
            productPriceVOS.forEach(productPriceVO -> {
                String priceStatus = productPriceVO.getPriceStatus();
                if (!priceStatus.equals(WorkFlowStatusEnum.CREATE_WORK.getCode()) && !priceStatus.equals(WorkFlowStatusEnum.PENDING_WORK.getCode())) {
                    throw TwException.error("", "仅支持新建和暂挂状态下的删除，请核验！");
                }
            });
            //删除价目
            productPriceDAO.deleteSoft(keys);
            //删除价目明细
            detailDAO.deleteSoftByPriceIds(keys);
        }
    }

    /**
     * 新建变更工作流
     *
     * @param productPriceVO
     * @return
     */
    private ProcessInfo startChangeWorkFlow(ProductPriceVO productPriceVO, Long saveId) {
        ProcessInfo processInfo = new ProcessInfo();
        if(workflow_enabled){
            Long orgManageUserId = cacheUtil.getOrgManageUserId(productPriceVO.getOrgId());
            HashMap<String, Object> batchMap = new HashMap<>();
            batchMap.put("Activity_0wctf3c", Lists.newArrayList(orgManageUserId.toString()));
            //发起流程审批
            processInfo = workflowUtil.startProcess(StartProcessPayload.of(
                ProcDefKey.PRODUCT_PRICE_CHANGE.name(),
                productPriceVO.getPriceName() + "-产品价目变更审批流程",
                saveId + "",
                batchMap)
            );
        }
        return processInfo;
    }

    /**
     * 数据校验
     *
     * @param payload
     */
    void checkData(ProductPricePayload payload) {
        if (!StringUtils.hasText(payload.getPriceName())) {
            throw TwException.error("", "价目表名称不可为空，请核验！");
        }
        if (ObjectUtils.isEmpty(payload.getOrgId())) {
            throw TwException.error("", "归属部门不可为空，请核验！");
        }
        if (ObjectUtils.isEmpty(payload.getManageUserId())) {
            throw TwException.error("", "负责人不可为空，请核验！");
        }
        if (ObjectUtils.isEmpty(payload.getCurrCode())) {
            throw TwException.error("", "币种为必填项，请核验！");
        }
        if (ObjectUtils.isEmpty(payload.getSaleCustType())) {
            throw TwException.error("", "销售客户类型不可为空，请核验！");
        } else {
            if (!payload.getSaleCustType().equals("all") && ObjectUtils.isEmpty(payload.getPriceCustomerPayloads())) {
                throw TwException.error("", "销售客户不可为空，请核验！");
            }
        }
        if (ObjectUtils.isEmpty(payload.getSaleOrgType())) {
            throw TwException.error("", "销售组织类型不可为空，请核验！");
        } else {
            if (!payload.getSaleOrgType().equals("all") && ObjectUtils.isEmpty(payload.getPriceOrgPayloads())) {
                throw TwException.error("", "销售组织不可为空，请核验！");
            }
        }
        if (ObjectUtils.isEmpty(payload.getPriceDetailPayloads())) {
            throw TwException.error("", "价目表明细不存在，请核验！");
        }
        Set<Long> skuIds = payload.getPriceDetailPayloads().stream().filter(priceDetailPayload -> priceDetailPayload.getSkuId() != null).map(ProductPriceDetailPayload::getSkuId).collect(Collectors.toSet());
        List<ProductSkuVO> productSkuVOS = skuDAO.queryByKeys(new ArrayList<>(skuIds));

        payload.getPriceDetailPayloads().forEach(priceDetailPayload -> {
            if (ObjectUtils.isEmpty(priceDetailPayload.getSkuId())) {
                throw TwException.error("", "未关联产品，请核验！");
            }
            if (ObjectUtils.isEmpty(priceDetailPayload.getMarkedMoney())) {
                throw TwException.error("", "价目表售价为空，请核验！");
            }
            if (ObjectUtils.isEmpty(priceDetailPayload.getDiscount())) {
                throw TwException.error("", "折扣比例为空，请核验！");
            }
            Optional<ProductSkuVO> first = productSkuVOS.stream().filter(productSkuVO -> productSkuVO.getId().equals(priceDetailPayload.getSkuId())).findFirst();
            if (first.isPresent()) {
                ProductSkuVO productSkuVO = first.get();
                //检验价格是否存在较大误差
//                BigDecimal bigDecimal = productSkuVO.getStandardPrice().multiply(priceDetailPayload.getDiscount()).divide(new BigDecimal(100)).setScale(2, RoundingMode.HALF_UP);
//                BigDecimal subtract = bigDecimal.subtract(priceDetailPayload.getMarkedMoney());

                priceDetailPayload.setSkuCode(productSkuVO.getSkuCode());
                priceDetailPayload.setSkuName(productSkuVO.getSkuName());
                priceDetailPayload.setSaleUnit(productSkuVO.getSaleUnit());
                priceDetailPayload.setStandardPrice(productSkuVO.getStandardPrice());
            } else {
                throw TwException.error("", "关联产品不存在，请核验！");
            }

        });

    }

    /**
     * 权限处理
     *
     * @param query
     */
    void operPermissionFlag(ProductPriceQuery query) {
        Boolean rolePermission = cacheUtil.hasSystemRolePermission(Arrays.asList(RoleEnum.SYS.getCode()));
        query.setPermissionFlag(!rolePermission);
        if (!rolePermission) {
            //需要处理权限
            Long userId = GlobalUtil.getLoginUserId();
            query.setLoginUserId(GlobalUtil.getLoginUserId());
            //负责组织
            List<Long> orgIdList = cacheUtil.getManagerOrgIdsByUserId(userId);
            Set<Long> orgIds = new HashSet<>();
            if (!ObjectUtils.isEmpty(orgIdList)) {
                orgIds.addAll(orgIdList);
            }
            //所在组织
            List<Long> longs = cacheUtil.getMyOrgIdsByUserId(userId);
            if (!ObjectUtils.isEmpty(longs)) {
                orgIds.addAll(longs);
            }
            query.setOrgIdList(orgIds);
        }
    }
}
