package com.elitesland.tw.tw5crm.server.product.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5crm.api.product.payload.ProductSkuPayload;
import com.elitesland.tw.tw5crm.api.product.query.ProductSkuQuery;
import com.elitesland.tw.tw5crm.api.product.service.ProductSkuService;
import com.elitesland.tw.tw5crm.api.product.vo.ProductSkuVO;
import com.elitesland.tw.tw5crm.server.product.convert.ProductSkuConvert;
import com.elitesland.tw.tw5crm.server.product.dao.ProductPriceDetailDAO;
import com.elitesland.tw.tw5crm.server.product.dao.ProductSkuDAO;
import com.elitesland.tw.tw5crm.server.product.entity.ProductPriceDetailDO;
import com.elitesland.tw.tw5crm.server.product.entity.ProductSkuDO;
import com.elitesland.tw.tw5crm.server.product.repo.ProductSkuRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 商品规格组合表SKU
 *
 * @author carl.wang
 * @date 2023-03-03
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class ProductSkuServiceImpl implements ProductSkuService {

    private final ProductSkuRepo productSkuRepo;
    private final ProductSkuDAO productSkuDAO;
    private final ProductPriceDetailDAO productPriceDetailDAO;

    @Override
    public PagingVO<ProductSkuVO> paging(ProductSkuQuery query) {
        query.setSkuStatus("on");
        return productSkuDAO.queryPaging(query);
    }

    @Override
    public List<ProductSkuVO> queryList(ProductSkuQuery query) {
        return productSkuDAO.queryListDynamic(query);
    }

    @Override
    public ProductSkuVO queryByKey(Long key) {
        ProductSkuDO entity = productSkuRepo.findById(key).orElseGet(ProductSkuDO::new);
        Assert.notNull(entity.getId(), "不存在");
        ProductSkuVO vo = ProductSkuConvert.INSTANCE.toVo(entity);
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ProductSkuVO insert(ProductSkuPayload payload) {
        ProductSkuDO entityDo = ProductSkuConvert.INSTANCE.toDo(payload);
        return ProductSkuConvert.INSTANCE.toVo(productSkuRepo.save(entityDo));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ProductSkuVO update(ProductSkuPayload payload) {
        ProductSkuDO entity = productSkuRepo.findById(payload.getId()).orElseGet(ProductSkuDO::new);
        Assert.notNull(entity.getId(), "不存在");
        ProductSkuDO entityDo = ProductSkuConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return ProductSkuConvert.INSTANCE.toVo(productSkuRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateStatus(ProductSkuPayload payload) {
        ProductSkuDO entity = productSkuRepo.findById(payload.getId()).orElseGet(ProductSkuDO::new);
        Assert.notNull(entity.getId(), "不存在");
        productSkuDAO.updateStatus(payload);
        if (payload.getSkuStatus().equals("off")) {
            //删除价目表数据
            productPriceDetailDAO.deleteSoftBySkuIds(List.of(payload.getId()));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            productSkuDAO.deleteSoft(keys);
            //删除价目表数据
            productPriceDetailDAO.deleteSoftBySkuIds(keys);
        } else {
            throw TwException.error("", "操作数据不可为空，请核验！");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveAll(List<ProductSkuPayload> skuPayloads) {
        List<ProductSkuDO> productSkuDOS = ProductSkuConvert.INSTANCE.toDoList(skuPayloads);
        productSkuDAO.saveAll(productSkuDOS);

        //更新价目表中的名称和价格
        List<Long> skuIds = new ArrayList<>();
        Map<Long, ProductSkuPayload> map = new HashMap<>();
        skuPayloads.forEach(skuPayload -> {
            if (skuPayload.getId() != null) {
                skuIds.add(skuPayload.getId());
                map.put(skuPayload.getId(), skuPayload);
            }
        });
        if (!ObjectUtils.isEmpty(skuIds)) {
            List<ProductPriceDetailDO> productPriceDetailDOS = productPriceDetailDAO.queryBySkuIds(skuIds);
            if (!ObjectUtils.isEmpty(productPriceDetailDOS)) {
                List<ProductPriceDetailDO> detailDOS = new ArrayList<>();
                productPriceDetailDOS.forEach(detailVO -> {
                    ProductSkuPayload productSkuPayload = map.get(detailVO.getSkuId());
                    if (!productSkuPayload.getSkuName().equals(detailVO.getSkuName())) {
                        detailDOS.add(detailVO);
                        detailVO.setSkuName(productSkuPayload.getSkuName());
                        if (productSkuPayload.getStandardPrice().compareTo(detailVO.getStandardPrice()) != 0) {
                            detailVO.setIsChange(true);
                        }
                    } else if (productSkuPayload.getStandardPrice().compareTo(detailVO.getStandardPrice()) != 0) {
                        detailVO.setIsChange(true);
                        detailDOS.add(detailVO);
                    }
                });
                if (detailDOS.size() > 0) {
                    productPriceDetailDAO.saveAll(detailDOS);
                }
            }
        }
    }
}
