package com.elitesland.tw.tw5crm.server.sale.dao;

import com.elitesland.tw.tw5crm.api.sale.payload.ExamInformationPayload;
import com.elitesland.tw.tw5crm.api.sale.query.ExamInformationQuery;
import com.elitesland.tw.tw5crm.api.sale.vo.ExamInformationVO;
import com.elitesland.tw.tw5crm.server.sale.entity.ExamInformationDO;
import com.elitesland.tw.tw5crm.server.sale.entity.QExamInformationDO;
import com.elitesland.tw.tw5crm.server.sale.repo.ExamInformationRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * sale
 *
 * @author kola
 * @date 2023-04-03
 */
@Repository
@RequiredArgsConstructor
public class ExamInformationDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ExamInformationRepo repo;
    private final QExamInformationDO qdo = QExamInformationDO.examInformationDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ExamInformationVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ExamInformationVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 主表主键 crm_sale_goal.goal_id
                qdo.goalId,
                // udc[crm:target:exam_target](线索数A(个)、商机金额B(元)、客户数(个)、合同额(元)、拜访数(个))
                qdo.examTarget,
                // 筛选字段key
                qdo.fieldKey,
                // 筛选字段名称
                qdo.fieldName,
                // 搜索类型
                qdo.searchType,
                // 搜索范围
                qdo.searchScope,
                // 搜索条件,多个条件以英文逗号隔开
                qdo.searchCondition,
                // 设置项KEY
                qdo.settingKey,
                // 设置项名称
                qdo.settingName,
                // 设置项值
                qdo.settingValue,
                // 拓展字段1
                qdo.ext1,
                // 拓展字段2
                qdo.ext2,
                // 拓展字段3
                qdo.ext3,
                // 拓展字段4
                qdo.ext4,
                // 拓展字段5
                qdo.ext5
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ExamInformationVO> getJpaQueryWhere(ExamInformationQuery query) {
        JPAQuery<ExamInformationVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
            }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ExamInformationQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ExamInformationQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 主表主键 crm_sale_goal.goal_id 精确 */
        if (!ObjectUtils.isEmpty(query.getGoalId())) {
            list.add(qdo.goalId.eq(query.getGoalId()));
        }
        /** udc[crm:target:exam_target](线索数A(个)、商机金额B(元)、客户数(个)、合同额(元)、拜访数(个)) 精确 */
        if (!ObjectUtils.isEmpty(query.getExamTarget())) {
            list.add(qdo.examTarget.eq(query.getExamTarget()));
        }
        /** 筛选字段key 精确 */
        if (!ObjectUtils.isEmpty(query.getFieldKey())) {
            list.add(qdo.fieldKey.eq(query.getFieldKey()));
        }
        /** 筛选字段名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getFieldName())) {
            list.add(qdo.fieldName.like(SqlUtil.toSqlLikeString(query.getFieldName())));
        }
        /** 搜索类型 精确 */
        if (!ObjectUtils.isEmpty(query.getSearchType())) {
            list.add(qdo.searchType.eq(query.getSearchType()));
        }
        /** 搜索范围 模糊 */
        if (!ObjectUtils.isEmpty(query.getSearchScope())) {
            list.add(qdo.searchScope.like(SqlUtil.toSqlLikeString(query.getSearchScope())));
        }
        /** 搜索条件,多个条件以英文逗号隔开 模糊 */
        if (!ObjectUtils.isEmpty(query.getSearchCondition())) {
            list.add(qdo.searchCondition.like(SqlUtil.toSqlLikeString(query.getSearchCondition())));
        }
        /** 设置项KEY 精确 */
        if (!ObjectUtils.isEmpty(query.getSettingKey())) {
            list.add(qdo.settingKey.eq(query.getSettingKey()));
        }
        /** 设置项名称 精确 */
        if (!ObjectUtils.isEmpty(query.getSettingName())) {
            list.add(qdo.settingName.eq(query.getSettingName()));
        }
        /** 设置项值 精确 */
        if (!ObjectUtils.isEmpty(query.getSettingValue())) {
            list.add(qdo.settingValue.eq(query.getSettingValue()));
        }
        /** 拓展字段1 精确 */
        if (!ObjectUtils.isEmpty(query.getExt1())) {
            list.add(qdo.ext1.eq(query.getExt1()));
        }
        /** 拓展字段2 精确 */
        if (!ObjectUtils.isEmpty(query.getExt2())) {
            list.add(qdo.ext2.eq(query.getExt2()));
        }
        /** 拓展字段3 精确 */
        if (!ObjectUtils.isEmpty(query.getExt3())) {
            list.add(qdo.ext3.eq(query.getExt3()));
        }
        /** 拓展字段4 精确 */
        if (!ObjectUtils.isEmpty(query.getExt4())) {
            list.add(qdo.ext4.eq(query.getExt4()));
        }
        /** 拓展字段5 精确 */
        if (!ObjectUtils.isEmpty(query.getExt5())) {
            list.add(qdo.ext5.eq(query.getExt5()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ExamInformationVO queryByKey(Long id) {
        JPAQuery<ExamInformationVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ExamInformationVO> queryListDynamic(ExamInformationQuery query) {
        JPAQuery<ExamInformationVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ExamInformationVO> queryPaging(ExamInformationQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<ExamInformationVO> jpaQuery = getJpaQueryWhere(query);
        List<ExamInformationVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ExamInformationVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ExamInformationDO save(ExamInformationDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ExamInformationDO> saveAll(List<ExamInformationDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ExamInformationPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    public long deleteSoftByGoalId(Long goalId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.goalId.eq(goalId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }


}

