package com.elitesland.tw.tw5crm.server.sale.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.util.PageUtil;
import com.elitesland.tw.tw5crm.api.sale.payload.SaleTargetPayload;
import com.elitesland.tw.tw5crm.api.sale.query.*;
import com.elitesland.tw.tw5crm.api.sale.service.ExamInformationService;
import com.elitesland.tw.tw5crm.api.sale.service.SaleTargetService;
import com.elitesland.tw.tw5crm.api.sale.vo.*;
import com.elitesland.tw.tw5crm.server.common.constants.SaleTargetExamTargetEnum;
import com.elitesland.tw.tw5crm.server.common.constants.SaleTargetSaleGoalTypeEnum;
import com.elitesland.tw.tw5crm.server.common.constants.SaleTargetStatisticalEnum;
import com.elitesland.tw.tw5crm.server.sale.convert.SaleTargetConvert;
import com.elitesland.tw.tw5crm.server.sale.dao.SaleGoalDAO;
import com.elitesland.tw.tw5crm.server.sale.dao.SaleTargetDAO;
import com.elitesland.tw.tw5crm.server.sale.entity.SaleGoalDO;
import com.elitesland.tw.tw5crm.server.sale.entity.SaleTargetDO;
import com.elitesland.tw.tw5crm.server.sale.repo.SaleGoalRepo;
import com.elitesland.tw.tw5crm.server.sale.repo.SaleTargetRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

import static com.elitesland.tw.tw5crm.server.sale.service.SaleGoalServiceImpl.monthDataConversion;
import static com.elitesland.tw.tw5crm.server.sale.service.SaleGoalServiceImpl.quarterDataConversion;

/**
 * sale
 *
 * @author kola
 * @date 2023-04-03
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class SaleTargetServiceImpl extends BaseServiceImpl implements SaleTargetService {

    private final SaleTargetRepo saleTargetRepo;
    private final SaleTargetDAO saleTargetDAO;
    private final ExamInformationService examInformationService;

    private final SaleGoalDAO saleGoalDAO;
    private final SaleGoalRepo saleGoalRepo;

    @Override
    public PagingVO<SaleTargetVO> paging(SaleTargetQuery query){
        Page<SaleTargetDO> page = saleTargetRepo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root,query,criteriaBuilder),query.getPageRequest());
        return PageUtil.toPageVo(page.map(SaleTargetConvert.INSTANCE::toVo));
    }

    @Override
    public PagingVO<SaleTargetVO> queryPaging(SaleTargetQuery query){
        return saleTargetDAO.queryPaging(query);
    }

    @Override
    public List<SaleTargetVO> queryList(SaleTargetQuery query){
        return SaleTargetConvert.INSTANCE.toVoList(
                saleTargetRepo.findAll(
                (root, criteriaQuery, criteriaBuilder)
                    -> QueryHelp.getPredicate(root, query, criteriaBuilder)
                , query.getPageRequest().getSort()
             )
        );
    }

    @Override
    public List<SaleTargetVO> queryListDynamic(SaleTargetQuery query){
        return saleTargetDAO.queryListDynamic(query);
    }

    @Override
    public SaleTargetVO queryByKey(Long key) {
        SaleTargetDO entity = saleTargetRepo.findById(key).orElseGet(SaleTargetDO::new);
        Assert.notNull(entity.getId(), "不存在");
        SaleTargetVO vo = SaleTargetConvert.INSTANCE.toVo(entity);
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public SaleTargetVO insert(SaleTargetPayload payload) {
        SaleTargetDO entityDo = SaleTargetConvert.INSTANCE.toDo(payload);
        return SaleTargetConvert.INSTANCE.toVo(saleTargetRepo.save(entityDo));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public SaleTargetVO update(SaleTargetPayload payload) {
        SaleTargetDO entity = saleTargetRepo.findById(payload.getId()).orElseGet(SaleTargetDO::new);
        Assert.notNull(entity.getId(), "不存在");
        SaleTargetDO entityDo = SaleTargetConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return SaleTargetConvert.INSTANCE.toVo(saleTargetRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            keys.stream().forEach(id -> {
                Optional<SaleTargetDO> optional = saleTargetRepo.findById(id);
                if (!optional.isEmpty()) {
                    SaleTargetDO entity = optional.get();
                    entity.setDeleteFlag(1);
                    saleTargetRepo.save(entity);
                }
             });
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoftByGoalId(Long goalId) {
        saleTargetDAO.deleteSoftByGoalId(goalId);
    }

    /**
     * 人员目标统计排行榜
     * @param query
     * @return
     */
    @Override
    public List<SaleTargetCountResultVO> queryCountUserOrder(SaleTargetStatisticalQuery query) {
        List<SaleTargetCountResultVO> resultList = new ArrayList<>();
        Long goalId = query.getGoalId();
        Long userId=query.getUserId();
        if (goalId == null) {
            throw TwException.error("", "goalId不能为空");
        }
        SaleGoalDO entity = saleGoalRepo.findById(goalId).orElseGet(SaleGoalDO::new);
        Assert.notNull(entity.getId(), "销售目标不存在");

        // 默认统计时间区间
        final Integer fiscalYear = entity.getFiscalYear();
        if (query.getStartTime() == null) {
            query.setStartTime(LocalDateTime.of(fiscalYear, 1, 1, 0, 0, 0));
        }
        if (query.getEndTime() == null) {
            query.setEndTime(LocalDateTime.of(fiscalYear, 12, 31, 23, 59, 59));
        }
        // 目标值（额定值）
        List<SaleTargetVO> saleTargetVOList = saleTargetDAO.countUserSaleTarget(goalId, "true",userId,null);
        SaleGoalVO saleGoalVO = saleGoalDAO.queryByKey(goalId);
        // 根据规则 去拉取实际完成值
        String goalType = saleGoalVO.getGoalType();
        // 考核指标：
        /** udc[crm:target:exam_target](线索数A(个)、商机金额B(元)、客户数(个)、合同额(元)、拜访数(个)) */
        String examTarget = saleGoalVO.getExamTarget();
        // 获取人员id
        Set<Long> userIds = saleTargetVOList.stream().map(SaleTargetVO::getDutyId).collect(Collectors.toSet());
        // 规则
        ExamInformationQuery examInformationQuery = new ExamInformationQuery();
        examInformationQuery.setGoalId(goalId);
        //获取考核规则
        List<ExamInformationVO> examInformationVOList = examInformationService.queryListDynamic(examInformationQuery);
        query.setExamInformationVOList(examInformationVOList);
        query.setUserIds(userIds);
        if (goalType.equals(SaleTargetSaleGoalTypeEnum.USER.getCode())) {
            // 考核指标:拜访数(个)
            if (examTarget.equals(SaleTargetExamTargetEnum.USER_VISIT.getCode())) {
                // 实际的完成值
                List<SaleTargetCountResultVO> visitList = saleTargetDAO.countUserVisitActual(query);
                saleTargetVOList.forEach(saleTargetVOS -> {
                    SaleTargetCountResultVO countResultVO = new SaleTargetCountResultVO();
                    countResultVO.setUserId(saleTargetVOS.getDutyId());
                    countResultVO.setProportion(BigDecimal.ZERO);
                    if (saleTargetVOS.getTargetCount().compareTo(BigDecimal.ZERO) == 0) {
                        saleTargetVOS.setTargetCount(new BigDecimal(1));
                    }
                    visitList.forEach(saleTargetVO -> {
                        if (saleTargetVOS.getDutyId().equals(saleTargetVO.getUserId())) {
                            countResultVO.setProportion(saleTargetVO.getActualCount().divide(saleTargetVOS.getTargetCount(), 6, RoundingMode.HALF_UP).multiply(new BigDecimal(100)).setScale(1, RoundingMode.HALF_UP));
                        }
                    });
                    resultList.add(countResultVO);
                });
            }
            //考核指标:线索数(个)
            if (examTarget.equals(SaleTargetExamTargetEnum.USER_CLUE.getCode())) {
                // 实际的完成值
                List<SaleTargetCountResultVO> clueList = saleTargetDAO.countClueByExamInformation(query);
                saleTargetVOList.forEach(saleTargetVOS -> {
                    SaleTargetCountResultVO countResultVO = new SaleTargetCountResultVO();
                    countResultVO.setUserId(saleTargetVOS.getDutyId());
                    countResultVO.setProportion(BigDecimal.ZERO);
                    if (saleTargetVOS.getTargetCount().compareTo(BigDecimal.ZERO) == 0) {
                        saleTargetVOS.setTargetCount(new BigDecimal(1));
                    }
                    clueList.forEach(saleTargetVO -> {
                        if (saleTargetVOS.getDutyId().equals(saleTargetVO.getUserId())) {
                            countResultVO.setProportion(saleTargetVO.getActualCount().divide(saleTargetVOS.getTargetCount(), 6, RoundingMode.HALF_UP).multiply(new BigDecimal(100)).setScale(1, RoundingMode.HALF_UP));
                        }
                    });
                    resultList.add(countResultVO);
                });
            }
            //考核指标:商机金额(元)
            if (examTarget.equals(SaleTargetExamTargetEnum.USER_BUSINESS.getCode())) {
                // 实际的完成值
                List<SaleTargetCountResultVO> oppoList = saleTargetDAO.countOppoByExamInformation(query);
                saleTargetVOList.forEach(saleTargetVOS -> {
                    SaleTargetCountResultVO countResultVO = new SaleTargetCountResultVO();
                    countResultVO.setUserId(saleTargetVOS.getDutyId());
                    countResultVO.setProportion(BigDecimal.ZERO);
                    if (saleTargetVOS.getTargetCount().compareTo(BigDecimal.ZERO) == 0) {
                        saleTargetVOS.setTargetCount(new BigDecimal(1));
                    }
                    oppoList.forEach(saleTargetVO -> {
                        if (saleTargetVOS.getDutyId().equals(saleTargetVO.getUserId())) {
                            countResultVO.setProportion(saleTargetVO.getActualCount().divide(saleTargetVOS.getTargetCount(), 6, RoundingMode.HALF_UP).multiply(new BigDecimal(100)).setScale(1, RoundingMode.HALF_UP));
                        }
                    });
                    resultList.add(countResultVO);
                });
            }
            //考核指标:合同额
            if (examTarget.equals(SaleTargetExamTargetEnum.USER_CONTRACT.getCode())){

            }
            //考核指标:客户数
            if (examTarget.equals(SaleTargetExamTargetEnum.USER_CUSTOMER.getCode())) {
                // 实际的完成值
                List<SaleTargetCountResultVO> customerList = saleTargetDAO.countCustomerByExam(query);
                saleTargetVOList.forEach(saleTargetVOS -> {
                    SaleTargetCountResultVO countResultVO = new SaleTargetCountResultVO();
                    countResultVO.setUserId(saleTargetVOS.getDutyId());
                    countResultVO.setProportion(BigDecimal.ZERO);
                    if (saleTargetVOS.getTargetCount().compareTo(BigDecimal.ZERO) == 0) {
                        saleTargetVOS.setTargetCount(new BigDecimal(1));
                    }
                    customerList.forEach(saleTargetVO -> {
                        if (saleTargetVOS.getDutyId().equals(saleTargetVO.getUserId())) {
                            countResultVO.setProportion(saleTargetVO.getActualCount().divide(saleTargetVOS.getTargetCount(), 6, RoundingMode.HALF_UP).multiply(new BigDecimal(100)).setScale(1, RoundingMode.HALF_UP));
                        }
                    });
                    resultList.add(countResultVO);
                });
            }

        } else if (goalType.equals(SaleTargetSaleGoalTypeEnum.CUSTOMER.getCode())) {

        }
        return resultList;
    }

    /**
     * 客户目标统计排行榜
     * @param query
     * @return
     */
    @Override
    public List<SaleTargetCountResultVO> countCustomerOrder(SaleTargetStatisticalQuery query) {
        List<SaleTargetCountResultVO> resultList = new ArrayList<>();
        Long goalId = query.getGoalId();
        if (goalId == null) {
            throw TwException.error("", "goalId不能为空");
        }
        SaleGoalDO entity = saleGoalRepo.findById(goalId).orElseGet(SaleGoalDO::new);
        Assert.notNull(entity.getId(), "销售目标不存在");

        // 默认统计时间区间
        final Integer fiscalYear = entity.getFiscalYear();
        if (query.getStartTime() == null) {
            query.setStartTime(LocalDateTime.of(fiscalYear, 1, 1, 0, 0, 0));
        }
        if (query.getEndTime() == null) {
            query.setEndTime(LocalDateTime.of(fiscalYear, 12, 31, 23, 59, 59));
        }
        // 目标值（额定值）
        List<SaleTargetVO> saleTargetVOList = saleTargetDAO.countCustomerSaleTarget(goalId, "true");
        SaleGoalVO saleGoalVO = saleGoalDAO.queryByKey(goalId);
        // 根据规则 去拉取实际完成值
        String goalType = saleGoalVO.getGoalType();
        // 考核指标：
        /** udc[crm:target:exam_target](销售量(个)、合同额(元)、回款额(元)、拜访数(个)) */
        String examTarget = saleGoalVO.getExamTarget();
        // 获取客户id
        Set<Long> customerIds = saleTargetVOList.stream().map(SaleTargetVO::getCustomerId).collect(Collectors.toSet());
        //获取目标人员id
        Set<Long> userIds = saleTargetVOList.stream().map(SaleTargetVO::getDutyId).collect(Collectors.toSet());
        // 规则
        ExamInformationQuery examInformationQuery = new ExamInformationQuery();
        examInformationQuery.setGoalId(goalId);
        //获取考核规则
        List<ExamInformationVO> examInformationVOList = examInformationService.queryListDynamic(examInformationQuery);
        query.setExamInformationVOList(examInformationVOList);
        query.setCustomerIds(customerIds);
        // 客户目标
        if (goalType.equals(SaleTargetSaleGoalTypeEnum.CUSTOMER.getCode())) {
            // 考核指标:拜访数(个)
            if (examTarget.equals(SaleTargetExamTargetEnum.CUSTOMER_VISIT.getCode())) {
                // 实际的完成值
                List<SaleTargetCountResultVO> visitList =  saleTargetDAO.countCustomerVisitActual(query);
                saleTargetVOList.forEach(saleTargetVOS ->{
                    SaleTargetCountResultVO countResultVO = new SaleTargetCountResultVO();
                    countResultVO.setCustomerId(saleTargetVOS.getCustomerId());
                    countResultVO.setCustomerName(saleTargetVOS.getCustomerName());
                    countResultVO.setProportion(BigDecimal.ZERO);
                    if (saleTargetVOS.getTargetCount().compareTo(BigDecimal.ZERO)==0){
                        saleTargetVOS.setTargetCount(new BigDecimal(1));
                    }
                    visitList.forEach(saleTargetVO->{
                        if (saleTargetVOS.getCustomerId().equals(saleTargetVO.getCustomerId())){
                            countResultVO.setProportion(saleTargetVO.getActualCount().divide(saleTargetVOS.getTargetCount(),6, RoundingMode.HALF_UP).multiply(new BigDecimal(100)).setScale(1,RoundingMode.HALF_UP));
                        }
                    });
                    resultList.add(countResultVO);
                });
            }
        }
        return resultList;
    }

    /**
     * 产品目标排行榜
     * @param query
     * @return
     */
    @Override
    public List<SaleTargetCountResultVO> countProductOrder(SaleTargetStatisticalQuery query) {
        List<SaleTargetCountResultVO> resultList = new ArrayList<>();
        Long goalId = query.getGoalId();
        if (goalId == null) {
            throw TwException.error("", "goalId不能为空");
        }
        SaleGoalDO entity = saleGoalRepo.findById(goalId).orElseGet(SaleGoalDO::new);
        Assert.notNull(entity.getId(), "销售目标不存在");

        // 默认统计时间区间
        final Integer fiscalYear = entity.getFiscalYear();
        if (query.getStartTime() == null) {
            query.setStartTime(LocalDateTime.of(fiscalYear, 1, 1, 0, 0, 0));
        }
        if (query.getEndTime() == null) {
            query.setEndTime(LocalDateTime.of(fiscalYear, 12, 31, 23, 59, 59));
        }
        // 目标值（额定值）
        List<SaleTargetVO> saleTargetVOList = saleTargetDAO.countProductSaleTarget(goalId, "true");
        SaleGoalVO saleGoalVO = saleGoalDAO.queryByKey(goalId);
        // 根据规则 去拉取实际完成值
        String goalType = saleGoalVO.getGoalType();
        // 考核指标：
        /** udc[crm:target:exam_target](销售量(个)、合同额(元)、回款额(元)、拜访数(个)) */
        String examTarget = saleGoalVO.getExamTarget();
        // 获取产品id
        Set<Long> skuIds = saleTargetVOList.stream().map(SaleTargetVO::getSkuId).collect(Collectors.toSet());
        //获取目标人员id
        Set<Long> userIds = saleTargetVOList.stream().map(SaleTargetVO::getDutyId).collect(Collectors.toSet());
        // 规则
        ExamInformationQuery examInformationQuery = new ExamInformationQuery();
        examInformationQuery.setGoalId(goalId);
        //获取考核规则
        List<ExamInformationVO> examInformationVOList = examInformationService.queryListDynamic(examInformationQuery);
        query.setExamInformationVOList(examInformationVOList);
        query.setSkuIds(skuIds);
        // 产品目标
        if (goalType.equals(SaleTargetSaleGoalTypeEnum.PRODUCT.getCode())) {
            saleTargetVOList.forEach(saleTargetVOS -> {
                SaleTargetCountResultVO countResultVO = new SaleTargetCountResultVO();
                countResultVO.setSkuId(saleTargetVOS.getSkuId());
                countResultVO.setSkuName(saleTargetVOS.getSkuName());
                saleTargetVOS.setTargetCount(saleTargetVOS.getTargetCount());
                countResultVO.setProportion(BigDecimal.ZERO);
                resultList.add(countResultVO);
            });
        }
        return resultList;
    }


    /**
     * 人员组织维度统计
     * @param query
     * @return
     */
    @Override
    public List<SaleTargetCountVO> queryUserOrgCount(SaleTargetUserOrgQuery query) {
        List<SaleTargetCountVO> resultList = new ArrayList<>();
        Long goalId = query.getGoalId();
        Long orgId=query.getOrgId();
        Long userId=query.getUserId();
        if (null == goalId) {
            throw TwException.error("", "goalId不能为空");
        }
        SaleGoalDO entity = saleGoalRepo.findById(goalId).orElseGet(SaleGoalDO::new);
        Assert.notNull(entity.getId(), "销售目标不存在");
        // 默认统计时间区间
        final Integer fiscalYear = entity.getFiscalYear();
        if (query.getStartTime() == null) {
            query.setStartTime(LocalDateTime.of(fiscalYear, 1, 1, 0, 0, 0));
        }
        if (query.getEndTime() == null) {
            query.setEndTime(LocalDateTime.of(fiscalYear, 12, 31, 23, 59, 59));
        }
        SaleGoalVO saleGoalVO = saleGoalDAO.queryByKey(goalId);
        // 根据规则 去拉取实际完成值
        String goalType = saleGoalVO.getGoalType();
        // 考核指标：
        /** udc[crm:target:exam_target](线索数A(个)、商机金额B(元)、客户数(个)、合同额(元)、拜访数(个)) */
        String examTarget = saleGoalVO.getExamTarget();
        //查询所有目标人员id
        List<SaleTargetVO> saleTargetVOLists = saleTargetDAO.countUserIds(goalId, "true",orgId,userId);
        if (saleTargetVOLists.size()==0){
            SaleTargetCountVO countResultVO = new SaleTargetCountVO();
            return null;
        }else {
            //获取负责人id
            Set<Long> userIds = saleTargetVOLists.stream().map(SaleTargetVO::getDutyId).collect(Collectors.toSet());
            // 目标值（额定值）
            List<SaleTargetVO> saleTargetVOList = saleTargetDAO.countUserOrgSaleTarget(goalId, "true",orgId,userId);
            //saleTargetVOS.stream().map(SaleTargetVO::getTargetCount).collect(Collectors.toSet())
            // 规则
            ExamInformationQuery examInformationQuery = new ExamInformationQuery();
            examInformationQuery.setGoalId(goalId);
            List<ExamInformationVO> examInformationVOS = examInformationService.queryListDynamic(examInformationQuery);
            // 人员目标
            if (goalType.equals(SaleTargetSaleGoalTypeEnum.USER.getCode())) {
                // 拜访数
                if (examTarget.equals(SaleTargetExamTargetEnum.USER_VISIT.getCode())) {
                    // 实际的完成值
                    List<SaleTargetCountVO> visitList =  saleTargetDAO.countOrgVisitByExamInformation(goalId,examInformationVOS,orgId,userId,userIds);
                    SaleTargetCountVO countResultVO = new SaleTargetCountVO();
                    saleTargetVOList.forEach(saleTargetVOS ->{
                        countResultVO.setTargetCount(saleTargetVOS.getTargetCount());
                        countResultVO.setProportion(BigDecimal.ZERO);
                        if (saleTargetVOS.getTargetCount().compareTo(BigDecimal.ZERO)==0){
                            saleTargetVOS.setTargetCount(new BigDecimal(1));
                        }
                        visitList.forEach(saleTargetVO->{
                            countResultVO.setSumCountLong(saleTargetVO.getSumCountLong());
                            countResultVO.setSumCount(saleTargetVO.getSumCount());
                            countResultVO.setProportion(saleTargetVO.getSumCount().divide(saleTargetVOS.getTargetCount(),6, RoundingMode.HALF_UP).multiply(new BigDecimal(100)).setScale(1, RoundingMode.HALF_UP));
                        });
                    });
                    resultList.add(countResultVO);
                }
                // 线索数
                if (examTarget.equals(SaleTargetExamTargetEnum.USER_CLUE.getCode())) {
                    // 实际的完成值
                    List<SaleTargetCountVO> clueList =  saleTargetDAO.countOrgClueByExamInformation(goalId,examInformationVOS,orgId,userId,userIds);
                    SaleTargetCountVO countResultVO = new SaleTargetCountVO();
                    saleTargetVOList.forEach(saleTargetVOS ->{
                        countResultVO.setTargetCount(saleTargetVOS.getTargetCount());
                        countResultVO.setProportion(BigDecimal.ZERO);
                        if (saleTargetVOS.getTargetCount().compareTo(BigDecimal.ZERO)==0){
                            saleTargetVOS.setTargetCount(new BigDecimal(1));
                        }
                        clueList.forEach(saleTargetVO->{
                            countResultVO.setSumCountLong(saleTargetVO.getSumCountLong());
                            countResultVO.setSumCount(saleTargetVO.getSumCount());
                            countResultVO.setProportion(saleTargetVO.getSumCount().divide(saleTargetVOS.getTargetCount(),6, RoundingMode.HALF_UP).multiply(new BigDecimal(100)).setScale(1, RoundingMode.HALF_UP));
                        });
                    });
                    resultList.add(countResultVO);
                }
                // 商机金额
                if (examTarget.equals(SaleTargetExamTargetEnum.USER_BUSINESS.getCode())) {
                    // 实际的完成值
                    List<SaleTargetCountVO> oppoList =  saleTargetDAO.countOrgOppoByExamInformation(goalId,examInformationVOS,orgId,userId,userIds);
                    SaleTargetCountVO countResultVO = new SaleTargetCountVO();
                    saleTargetVOList.forEach(saleTargetVOS ->{
                        countResultVO.setTargetCount(saleTargetVOS.getTargetCount());
                        countResultVO.setProportion(BigDecimal.ZERO);
                        if (saleTargetVOS.getTargetCount().compareTo(BigDecimal.ZERO)==0){
                            saleTargetVOS.setTargetCount(new BigDecimal(1));
                        }
                        oppoList.forEach(saleTargetVO->{
                            countResultVO.setSumCountLong(saleTargetVO.getSumCountLong());
                            countResultVO.setSumCount(saleTargetVO.getSumCount());
                            countResultVO.setProportion(saleTargetVO.getSumCount().divide(saleTargetVOS.getTargetCount(),6, RoundingMode.HALF_UP).multiply(new BigDecimal(100)).setScale(1, RoundingMode.HALF_UP));
                        });
                    });
                    resultList.add(countResultVO);
                }
                // 客户数
                if (examTarget.equals(SaleTargetExamTargetEnum.USER_CUSTOMER.getCode())) {
                    // 实际的完成值
                    List<SaleTargetCountVO> customerList =  saleTargetDAO.countOrgCustomerByExamInformation(goalId,examInformationVOS,orgId,userId,userIds);
                    SaleTargetCountVO countResultVO = new SaleTargetCountVO();
                    saleTargetVOList.forEach(saleTargetVOS ->{
                        countResultVO.setTargetCount(saleTargetVOS.getTargetCount());
                        countResultVO.setProportion(BigDecimal.ZERO);
                        if (saleTargetVOS.getTargetCount().compareTo(BigDecimal.ZERO)==0){
                            saleTargetVOS.setTargetCount(new BigDecimal(1));
                        }
                        customerList.forEach(saleTargetVO->{
                            countResultVO.setSumCountLong(saleTargetVO.getSumCountLong());
                            countResultVO.setSumCount(saleTargetVO.getSumCount());
                            countResultVO.setProportion(saleTargetVO.getSumCount().divide(saleTargetVOS.getTargetCount(),6, RoundingMode.HALF_UP).multiply(new BigDecimal(100)).setScale(1, RoundingMode.HALF_UP));
                        });
                    });
                    resultList.add(countResultVO);
                }
            }
            return resultList;
        }

    }

    /**
     * 客户维度统计
     * @param query
     * @return
     */
    @Override
    public List<SaleTargetCountVO> queryCustomerCount(SaleTargetCustomerQuery query) {
        List<SaleTargetCountVO> resultList = new ArrayList<>();
        Long goalId = query.getGoalId();
        if (null == goalId) {
            throw TwException.error("", "goalId不能为空");
        }
        SaleGoalDO entity = saleGoalRepo.findById(goalId).orElseGet(SaleGoalDO::new);
        Assert.notNull(entity.getId(), "销售目标不存在");
        // 默认统计时间区间
        final Integer fiscalYear = entity.getFiscalYear();
        if (query.getStartTime() == null) {
            query.setStartTime(LocalDateTime.of(fiscalYear, 1, 1, 0, 0, 0));
        }
        if (query.getEndTime() == null) {
            query.setEndTime(LocalDateTime.of(fiscalYear, 12, 31, 23, 59, 59));
        }
        SaleGoalVO saleGoalVO = saleGoalDAO.queryByKey(goalId);
        // 根据规则 去拉取实际完成值
        String goalType = saleGoalVO.getGoalType();
        // 考核指标：
        /** udc[crm:target:exam_target](线索数A(个)、商机金额B(元)、客户数(个)、合同额(元)、拜访数(个)) */
        String examTarget = saleGoalVO.getExamTarget();
        //目标值（额定）
        query.setExt1("true");
        //目标值（额定）
        List<SaleTargetVO> saleTargetVOList = saleTargetDAO.countCustomerSaleTarget(query);
        //查询用户和客户id
        List<SaleTargetVO> saleTargetVOLists = saleTargetDAO.countCustomerIdAndUserId(query);
        if (saleTargetVOLists.size()==0){
            SaleTargetCountVO countResultVO = new SaleTargetCountVO();
            return null;
        }else {
            //获取负责人id
            Set<Long> userIds = saleTargetVOLists.stream().map(SaleTargetVO::getDutyId).collect(Collectors.toSet());
            //获取客户id
            Set<Long> customerIds = saleTargetVOLists.stream().map(SaleTargetVO::getCustomerId).collect(Collectors.toSet());
            query.setCustomerIds(customerIds);
            query.setUserIds(userIds);
            // 规则
            ExamInformationQuery examInformationQuery = new ExamInformationQuery();
            examInformationQuery.setGoalId(goalId);
            List<ExamInformationVO> examInformationVOS = examInformationService.queryListDynamic(examInformationQuery);
            // 客户目标
            if (goalType.equals(SaleTargetSaleGoalTypeEnum.CUSTOMER.getCode())) {
                // 根据规则 去查询实际完成值 销售拜访任务
                if (examTarget.equals(SaleTargetExamTargetEnum.CUSTOMER_VISIT.getCode())) {
                    // 实际的完成值
                    List<SaleTargetCountVO> visitList =  saleTargetDAO.countCustomerByExamInformation(examInformationVOS,query);
                    SaleTargetCountVO countResultVO = new SaleTargetCountVO();
                    saleTargetVOList.forEach(saleTargetVOS ->{
                        countResultVO.setTargetCount(saleTargetVOS.getTargetCount());
                        countResultVO.setProportion(BigDecimal.ZERO);
                        if (saleTargetVOS.getTargetCount().compareTo(BigDecimal.ZERO)==0){
                            saleTargetVOS.setTargetCount(new BigDecimal(1));
                        }
                        visitList.forEach(saleTargetVO->{
                            countResultVO.setSumCountLong(saleTargetVO.getSumCountLong());
                            countResultVO.setSumCount(saleTargetVO.getSumCount());
                            countResultVO.setProportion(saleTargetVO.getSumCount().divide(saleTargetVOS.getTargetCount(),6, RoundingMode.HALF_UP).multiply(new BigDecimal(100)).setScale(1, RoundingMode.HALF_UP));
                        });
                    });
                    resultList.add(countResultVO);
                }
            }
            else if (goalType.equals(SaleTargetSaleGoalTypeEnum.USER.getCode())) {
            }
            return resultList;
        }

    }




    /**
     * 获取考核规则列表
     *
     * @param saleTargetVO
     */
    private void getExamInformationList(SaleTargetVO saleTargetVO) {
        // 规则
        ExamInformationQuery examInformationQuery = new ExamInformationQuery();
        Long goalId = saleTargetVO.getGoalId();
        examInformationQuery.setGoalId(goalId);
        List<ExamInformationVO> examInformationVOS = examInformationService.queryListDynamic(examInformationQuery);
            // 根据规则 去拉取实际完成值
            String goalType = saleTargetVO.getGoalType();
            // 人员目标
            if (goalType.equals(SaleTargetSaleGoalTypeEnum.USER.getCode())) {
                // 根据规则 去查询实际完成值
                if (saleTargetVO.getExt2().equals(SaleTargetExamTargetEnum.USER_VISIT.getCode())) {
                    // 销售拜访任务
//                    List<SaleTargetVO> visitList =  saleTargetDAO.countByExamInformation(examInformationVOS,saleTargetVO);
//                    visitList.forEach(visitCount -> {
//                        saleTargetVO.setActualCount(new BigDecimal(visitCount.));
//                    });
                }
            }
            //客户目标
            else if (goalType.equals(SaleTargetSaleGoalTypeEnum.CUSTOMER.getCode())) {
                // 根据规则 去查询实际完成值
                if (saleTargetVO.getExt2().equals(SaleTargetExamTargetEnum.CUSTOMER_VISIT.getCode())) {
                    // 销售拜访任务
//                    List<SaleTargetVO> visitList = saleTargetDAO.countByExamInformation(examInformationVOS,saleTargetVO);
                }
            }
    }

}
